;;;   -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-    ;;;
;;;                                                                       ;;;
;;;  Copyright (c) 1990 by James Crawford.                                ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'user)


(defun facts-about-morphism ()

  ;; Skolem slots should migrate to background KB:
  (a-assert "Skolem slots"
    '((:taxonomy (slots (skolem-slots)))))
  (a-assert "Fill in Skolem slots"
    ;; If the value in a "skolem slot" is ever queried, create one if
    ;; it is unknown.  Would like this to work for slots of any arity.
    ;; This could be done with:
    ;; '((:srules skolem-slots
    ;;       ((?p ?x &others ?y) <- (:forc ?y (?p ?x &others ?y)))))
    ;; but & variables are not yet supported.  For the moment we just
    ;; hack it in lisp (this hack really should not work, but it will
    ;; do until & variables are implemented).
    `((:srules skolem-slots
       ((?p ?x ?y) <-
	(:apply ,#'(lambda (x) (not (listp x))) (?y))
	(:apply ,#'(lambda (x) (not (framep x))) (?y))
	(:forc ?y (?p ?x ?y)))
       ((?p ?x ?y) <-
	(:apply ,#'listp (?y))
	(:bind ?last (car (last '?y)))
	(:apply ,#'(lambda (x) (not (framep x))) (?last))
	(:forc ?last (?p ?x ?y)))
#|
       ;; Force skolem functions to be functions:
       ((?p ?x ?y) 
	(:test (not (listp '?y)))
	(:retrieve (?p ?x ?y2))
	(:neq ?y ?y2)
	(:test (not (listp '?y2)))
	->
	(:lisp (format t "~% From skolem function ~a concluding ~a = ~a."
		       `?p '?y '?y2))
	(coreferent ?y ?y2))
       ((?p ?x ?y)
	(:test (listp '?y))
	(:retrieve (?p ?x ?y2))
	(:test (not (equal '?y '?y2)))
	(:test (listp '?y2))
	(:test (equal (butlast '?y) (butlast '?y2)))
	(:bind ?last (car (last '?y)))
	(:bind ?last2 (car (last '?y2)))
	->
	(:lisp (format t "~% From skolem function ~a concluding ~a = ~a."
		       `?p '?last '?last2))
	(coreferent ?last ?last2))
|#
       )))


  (a-assert "Taxonomy"
	    '((:taxonomy (things (representatives)))
	      (:taxonomy (slots (st-slots)))))

  (a-assert "Slots"
	    ;; We use st-slot to avoid application of any
	    ;; rules defined for slot in background KB.
	    '((:slot st-equal (sets sets))
	      (isa (:slot st-equal) st-slots)
	      (:slot st-subseteq (sets sets)
	       :comment "Subset or equal.")
	      (isa (:slot st-subseteq) st-slots)
	      (:slot st-member (sets objects))
	      (isa (:slot st-member) st-slots)
	      (:slot st-isa (objects sets))
	      (inverse st-isa st-member)
	      (isa (:slot st-isa) st-slots)

	      (:slot pr (sets sets sets)
	       :comment "(pr y z p) represents p = pr(y,z).")
	      (isa (:slot pr) st-slots)
	      (:slot inv-pr (sets sets sets)
	       :comment "(inv-pr p y z) <-> (pr y z p)")
	      (isa (:slot inv-pr) st-slots)
	      ;; Add inverse links:
	      (:rules sets
	       ((pr ?y ?z ?p) -> (inv-pr ?p ?y ?z))
	       ((inv-pr ?p ?y ?z) -> (pr ?y ?z ?p)))
	      (isa (:slot pr) skolem-slots)

	      (:slot sing (sets sets)
	       :cardinality 1
	       :comment "(sing x s) represents s = sing(x).")
	      (isa (:slot sing) st-slots)
	      (:slot inv-sing (sets sets))
	      (inverse sing inv-sing)
	      (isa (:slot inv-sing) st-slots)
	      (isa (:slot sing) skolem-slots)

	      (:slot op (sets sets sets)
	       :comment "(op x y o) represents o = <x,y>.")
	      (isa (:slot op) st-slots)
	      (:slot inv-op (sets sets sets)
	       :comment "(inv-op o x y) <-> (op x y o).")
	      (isa (:slot inv-op) st-slots)
	      ;; Add inverse links:
	      (:rules sets
	       ((op ?x ?y ?o) -> (inv-op ?o ?x ?y))
	       ((inv-op ?o ?x ?y) -> (op ?x ?y ?o)))
	      (isa (:slot op) skolem-slots)

	      (:slot isop (sets booleans)
	       :cardinality 1
	       :comment "isop = is ordered pair.")
	      (isa (:slot isop) st-slots)

	      (:slot rel (sets booleans)
	       :cardinality 1
	       :comment "rel = is relation.")
	      (isa (:slot rel) st-slots)

	      (:slot func (sets booleans)
	       :cardinality 1
	       :comment "func = is function.")
	      (isa (:slot func) st-slots)
	      (:slot sv (sets sets sets)
	       :comment "(sv x y sv) represents sv = sv(x,y)")
	      (isa (:slot sv) st-slots)
	      (isa (:slot sv) skolem-slots)

	      (:slot cmp (sets sets sets)
	       :comment "(cmp x y cmp) represents cmp = cmp(x,y).")
	      (isa (:slot cmp) st-slots)
	      (:slot inv-cmp (sets sets sets)
	       :comment "(inv-cmp cmp x y) <-> (cmp x y cmp).")
	      (isa (:slot inv-cmp) st-slots)
	      (:rules sets
	       ;; Add inverse links:
	       ((cmp ?x ?y ?cmp) -> (inv-cmp ?cmp ?x ?y))
	       ((inv-cmp ?cmp ?x ?y) -> (cmp ?x ?y ?cmp)))
	      (isa (:slot cmp) skolem-slots)

	      (:slot istrip (sets booleans)
	       :cardinality 1
	       :comment "istrip = is an ordered triple.")
	      (isa (:slot istrip) st-slots)

	      (:slot rel3 (sets booleans)
	       :cardinality 1
	       :comment "rel3 = is a triadic relation.")
	      (isa (:slot rel3) st-slots)

	      (:slot homo (sets sets sets)
	       :comment "(homo homomorphism relation1 relation2)")
	      (isa (:slot homo) st-slots)))

  (a-assert "Representatives"
	    '(;; We use representatives to implement universal
	      ;; quantification.
	      (:slot represents-set (representatives sets)
	       :cardinality 1)
	      (:slot rep (sets representatives)
	       :backlink represents-set
	       :cardinality 1)
	      (isa (:slot rep) skolem-slots)
	      (:rules sets
	       ;; Representatives are members:
	       ((rep ?set ?x) -> (st-isa ?x ?set)))))

  (a-assert "Conditional representatives"
	    '(;; Conditinal representatives let us represent:
	      ;; "For all x such that <condition>, <exp>".
	      (:slot conditional-rep (sets nil representatives)
	       :comment "(conditional-rep set condition rep) -- condition = (:lambda var path)")
	      (isa (:slot conditional-rep) skolem-slots)
	      (:rules sets
	       ;; Conditional representatives are members:
	       ((conditional-rep ?set ?cond ?x) -> (st-isa ?x ?set))
	       ;; Link conditional representative up to set:
	       ((conditional-rep ?set ?cond ?x) -> (represents-set ?x ?set))
	       ;; Conditional representatives satisfy their conditions:
	       ((conditional-rep ?set ?cond ?x) ->
		(:lisp (assert-condition '?x '?cond))))))

  (a-assert "Equality Axioms"
	    `(;; Substitute st-equal for st-equal (for set theory
	      ;; predicates).
	      (:srules st-slots
	       ((?p ?x ?y)
		<-
		;; Make sure slot is binary:
		(:apply ,#'(lambda (x) (not (listp x))) (?y))
		(:retrieve (st-equal ?x ?x2))
		(?p ?x2 ?y))
	       ((?p ?x ?y)
		<-
		;; Make sure slot is binary:
		(:apply ,#'(lambda (x) (not (listp x))) (?y))
		(:retrieve (st-equal ?y ?y2))
		(?p ?x ?y2))
	       ((?p ?x ?y ?z)
		<-
		(:retrieve (st-equal ?x ?x2))
		(?p ?x2 ?y ?z))
	       ((?p ?x ?y ?z)
		<-
		(:retrieve (st-equal ?y ?y2))
		(?p ?x ?y2 ?z))
	       ((?p ?x ?y ?z)
		<-
		(:retrieve (st-equal ?z ?z2))
		(?p ?x ?y ?z2)))))

  (a-assert "Set equality"
	    '(;; x=y iff x subseteq y and y subseteq x
	      (:rules sets
	       ((st-equal ?x ?y) -> (st-subseteq ?x ?y) (st-subseteq ?y ?x))
	       ((st-equal ?x ?y) <- (st-subseteq ?x ?y) (st-subseteq ?y ?x)))))

  (a-assert "Subset"
	    '(;; x subseteq y iff for all z, if z in x then z in y.
	      (:rules sets
	       ((st-subseteq ?x ?y) (st-member ?x ?m) -> (st-isa ?m ?y))
	       ((st-subseteq ?x ?y) <- (st-isa (rep ?x) ?y)))))

  (a-assert "Pair set"
	    '(;; x in pr(y,z) iff set(y), set(z), (x=y OR x=z).
	      ;; set(pr(x,y))
	      ;; (We ignore set restriction since pr slot typed to sets.)
	      (:rules sets
	       ((st-isa ?x ?pr-y-z) (inv-pr ?pr-y-z ?y ?z)
		(not (st-equal ?x ?y))
		-> (st-equal ?x ?z))
	       ((st-isa ?x ?pr-y-z) (inv-pr ?pr-y-z ?y ?z)
		(not (st-equal ?x ?z))
		-> (st-equal ?x ?y))
	       
	       ((st-isa ?x ?pr-y-z)
		<-
		(inv-pr ?pr-y-z ?y ?z) (st-equal ?x ?y))
	       ((st-isa ?x ?pr-y-z)
		<-
		(inv-pr ?pr-y-z ?y ?z) (st-equal ?x ?z)))))

  (a-assert "Singleton"
	    '(;; x in sing(y) iff set(y) and x=y.
	      ;; set(sing(x))
	      (:rules sets
	       ((st-isa ?x ?sing-y) (inv-sing ?sing-y ?y) -> (st-equal ?x ?y))
	       ((st-isa ?x ?sing-y) <- (inv-sing ?sing-y ?y) (st-equal ?x ?y)))))

  (a-assert "Ordered pair"
	    '(;; x in <y,z> iff x in pr(sing(y),pr(y,z)).
	      ;; set(<y,z>)
	      (:rules sets
	       ((st-isa ?x ?op-y-z) (inv-op ?op-y-z ?y ?z)
		->
		(st-isa ?x (pr (sing ?y) (pr ?y ?z))))
	       ((st-isa ?x ?op-y-z)
		<-
		(inv-op ?op-y-z ?y ?z)
		(st-isa ?x (pr (sing ?y) (pr ?y ?z)))))))

  (a-assert "Is an ordered pair"
	    '(;; isop(x) -> x = <sx(x),sy(x)>
	      ;; isop(x) <- x = <y,z>, set(y), set(z)
	      ;; set(sx(x))
	      ;; set(sy(x))
	      (:rules sets
	       ;; For ordered pairs we can create ?sx and ?sy:
	       ((inv-op ?x ?sx ?sy)
		<-
		(isop ?x true)
		(:forc (?sx ?sy)
		       (inv-op ?x ?sx ?sy)))
	       ;; If ?sx and ?sy exist then its an ordered pair:
	       ((isop ?x true)
		<-
		(inv-op ?x ?sx ?sy)))))

  (a-assert "Relation"
	   '(;; rel(x) <-> For all y in x, isop(y)
	     (:rules sets
	      ((rel ?x true) (st-member ?x ?y)
	       ->
	       (isop ?y true))
	      ((rel ?x true) <- (isop (rep ?x) true)))))

  (a-assert "Function and value"
	    '(;; func(x) -> rel(x)
	      ;; func(x), <y,z> in x -> z = sv(x,y)
	      ;; func(x) <- rel(x),
	      ;;    for all <sx,sy> in x such that <sx,sv(x,sx)> in x: sy = sv(x,sx)
	      ;; set(sv(x,y))
	      (:rules sets
	       ((func ?x true) -> (rel ?x true))

	       ((func ?x true)
		(st-member ?x ?op)
		(inv-op ?op ?y ?z)
		->
		(st-equal ?z (sv ?x ?y)))

	       ((func ?x true)
		<-
		(rel ?x true)
		(conditional-rep
		 ?x
		 (:lambda ?p
		   ((isop ?p true)
		    (represents-set ?p ?set)
		    (inv-op ?p ?sx ?sy)
		    (st-isa (op ?sx
			       (sv ?set ?sx))
			   ?set)))
		 ?rep)
		(inv-op ?rep ?rep-sx ?rep-sy)
		(st-equal ?rep-sy (sv ?x ?rep-sx))))))

  (a-assert "Composition of functions"
	    '(;; x in cmp(u,v) <->
	      ;;     isop(x), <sx(x),sv(u,sx(x))> in u, <sv(u,sx(x)),sy(x)> in v
	      ;; set(cmp(u,v))
	      (:rules sets
	       ((cmp ?u ?v ?c)
		(st-member ?c ?x)
		->
		(isop ?x true)
		(inv-op ?x ?sx ?sy)
		(st-isa (op ?sx (sv ?u ?sx)) ?u)
		(st-isa (op (sv ?u ?sx) ?sy) ?v))
	       ((st-member ?c ?x)
		<-
		(inv-cmp ?c ?u ?v)
		(isop ?x true)
		(inv-op ?x ?sx ?sy)
		;; Removed :retrieve
		(op ?sx (sv ?u ?sx) ?op1)
		(st-isa ?op1 ?u)
		;; Removed :retrieve
		(op (sv ?u ?sx) ?sy ?op2)
		(st-isa ?op2 ?v)))))

  (a-assert "Is an ordered triple"
	   '(;; istrip(x) -> x = <<sx(sx(x)),sy(sx(x))>,sy(x)>
	     ;; istrip(x) <- x = <u,v>, isop(u), set(v)
	     (:rules sets
	      ((inv-op ?x ?sx ?sy)
	       (isop ?sx true)
	       <-
	       (istrip ?x true)
	       (:forc (?sx ?sy)
		      (inv-op ?x ?sx ?sy)))
	      ((istrip ?x true)
	       <-
	       (inv-op ?x ?u ?v)
	       (isop ?u true)
	       (isa ?v sets)))))

  (a-assert "Is a triadic relation"
	    '(;; rel3(x) <-> for all y in x, istrip(x)
	     (:rules sets
	      ((rel3 ?x true) (st-member ?x ?y)
	       ->
	       (istrip ?y true))
	      ((rel3 ?x true) <- (istrip (rep ?x) true)))))

  (a-assert "Homonorphism"
	    '(;; homo(x,W1,W2) -> func(x), rel3(W1), rel3(W2)
	      ;;
	      ;; homo(x,W1,W2),
	      ;; <<u1,u2>,u3> in W1,
	      ;; <u1,sv(x,u1)> in x,
	      ;; <u2,sv(x,u2)> in x,
	      ;; <u2,sv(x,u3)> in x ->
	      ;; <<sv(x,u1),sv(x,u2)>,sv(x,u3)> in W2
	      ;; 
	      ;; homo(x,W1,W2) <- func(x), rel3(W1), rel3(W2),
	      ;;  for all <<e,f>,g> in W1:
	      ;;    <e,sv(x,e)> in x,
	      ;;    <f,sv(x,f)> in x,
	      ;;    <g,sv(x,g)> in x ->
	      ;;    <<sv(x,e),sv(x,f)>,sv(x,g)> in W2
	      (:rules sets

	       ((homo ?x ?W1 ?W2) -> (func ?x true) (rel3 ?W1 true) (rel3 ?W2 true))
	       ((homo ?x ?W1 ?W2)
		(st-member ?W1 ?trip)
		(inv-op ?trip ?sx ?u3)
		(inv-op ?sx ?u1 ?u2)
		(st-isa (op ?u1 (sv ?x ?u1)) ?x)
		(st-isa (op ?u2 (sv ?x ?u2)) ?x)
		(st-isa (op ?u3 (sv ?x ?u3)) ?x)
		->
		(st-isa (op (op (sv ?x ?u1) (sv ?x ?u2)) (sv ?x ?u3)) ?W2))
	       ((homo ?x ?W1 ?W2)
		<-
		(func ?x true)
		(rel3 ?W1 true)
		(rel3 ?W2 true)
		;;(:lisp (trace-logic))
		(conditional-rep
		 ?W1
		 (:lambda ?trip*
		   ((istrip ?trip* true)
		    (inv-op ?trip* ?sx* ?g*)
		    (inv-op ?sx* ?e* ?f*)
		    (st-isa (op ?e* (sv ?x ?e*)) ?x)
		    (st-isa (op ?f* (sv ?x ?f*)) ?x)
		    (st-isa (op ?g* (sv ?x ?g*)) ?x)))
		 ?rep)
		(inv-op ?rep ?sx2 ?g2)
		(inv-op ?sx2 ?e2 ?f2)
		(st-isa (op (op (sv ?x ?e2) (sv ?x ?f2)) (sv ?x ?g2))
		       ?W2))))))


(defun assert-condition (rep cond)
  (let ((free-var (second cond))
        (path (third cond)))
    ;;(break)
    (a-assert nil `((:bind ,free-var ',rep)
		    ,@path))
    ;;(trace-off)
    ))


(defun queries-about-morphism ()

  (a-assert "Theorem 1 antecedent: func(g), func(h)."
	    '((:a ?g (name ?g "G"))
	      (:a ?h (name ?h "H"))

	      (func ?g true)
	      (func ?h true)))

  (a-query "Theorem 1 consequent: func(g cmp h)."
	   '((func (cmp g h) true)))

  (a-assert "Theorem 2 antecedent: homo(hf,hp,hq) and homo(hg,hq,hr)."
	    '((:a ?hf (name ?hf "HF"))
	      (:a ?hp (name ?hp "HP"))
	      (:a ?hq (name ?hq "HQ"))
	      (:a ?hg (name ?hg "HG"))
	      (:a ?hr (name ?hr "HR"))

	      (:a ?fog (name ?fog "FOG"))

	      (homo ?hf ?hp ?hq)
	      (homo ?hg ?hq ?hr)
	      (cmp ?hf ?hg ?fog)))

  (a-query "Theorem 2 consequent: homo(hf cmp hg,hp,hr)."
	   '((homo fog hp hr))))

