;;; Copyright 1993, Benjamin Kuipers.
(in-package :user)

;;; ROYAL FAMILY
;;;
;;; A slightly longer example on reasoning about geneology.
;;;
;;; Mixed if-added and if-needed rules, and predicates at different
;;; levels of abstraction. 

(defun facts-about-families ()

  (tell '((:taxonomy (things
		      (objects
		       (physical-objects
			(people))
		       (physical-attributes
			(genders male female))))))
	:comment "Taxonomy.")

  ;; These slots, P(a,b) have the form, "a P of a is b".

  (tell '((:slot child    (people people))
	  (:slot son      (people people))
	  (:slot daughter (people people))
	  
	  (:slot parent (people people)  :cardinality 2)
	  (inverse parent child)
	  (:slot father (people people)  :cardinality 1)
	  (:slot mother (people people)  :cardinality 1)
	  
	  (:slot sibling (people people))
	  (inverse sibling sibling)
	  (:slot brother (people people))
	  (:slot sister  (people people))
	  
	  (:slot grandchild   (people people))
	  (:slot grandson     (people people))
	  (:slot grandaughter (people people))
	  
	  (:slot grandparent (people people)  :cardinality 4)
	  (inverse grandparent grandchild )
	  (:slot grandfather (people people)  :cardinality 2)
	  (:slot grandmother (people people)  :cardinality 2)
	  
	  (:slot uncle (people people))
	  (:slot aunt  (people people))
	  )
	:comment "Family relations.")

  ;; Implications are tied to gender-specific predicates.

  (tell '((:rules people
	   ((father ?x ?f) -> (parent ?x ?f) (gender ?f male))
	   ((father ?x ?f) <- (parent ?x ?f) (gender ?f male))
	   
	   ((mother ?x ?f) -> (parent ?x ?f) (gender ?f female))
	   ((mother ?x ?f) <- (parent ?x ?f) (gender ?f female))
	   
	   ((son ?x ?s) -> (child ?x ?s) (gender ?s male))
	   ((son ?x ?s) <- (child ?x ?s) (gender ?s male))
	   
	   ((daughter ?x ?d) -> (child ?x ?d) (gender ?d female))
	   ((daughter ?x ?d) <- (child ?x ?d) (gender ?d female))
	   
	   ((brother ?x ?b) -> (sibling ?x ?b) (gender ?b male))
	   ((brother ?x ?b) <- (sibling ?x ?b) (gender ?b male))
	   
	   ((sister ?x ?b) -> (sibling ?x ?b) (gender ?b female))
	   ((sister ?x ?b) <- (sibling ?x ?b) (gender ?b female))
	   
	   ((grandfather ?x ?gf) -> (grandparent ?x ?gf) (gender ?gf male))
	   ((grandfather ?x ?gf) <- (grandparent ?x ?gf) (gender ?gf male))
	   
	   ((grandmother ?x ?gf) -> (grandparent ?x ?gf) (gender ?gf female))
	   ((grandmother ?x ?gf) <- (grandparent ?x ?gf) (gender ?gf female))
	   
	   ((grandson ?x ?gs) -> (grandchild ?x ?gs) (gender ?gs male))
	   ((grandson ?x ?gs) <- (grandchild ?x ?gs) (gender ?gs male))
	   
	   ((grandaughter ?x ?gs) -> (grandchild ?x ?gs) (gender ?gs female))
	   ((grandaughter ?x ?gs) <- (grandchild ?x ?gs) (gender ?gs female))
	   ))
	:comment "Rules defining gender-specific relations as shorthand")

  ;; Then the rules to infer the more complex relations:

  (tell '((:rules people

	   ;; Aunt and Uncle are a bit different (there is no unisex term):
	   ((uncle ?x ?u) -> (gender ?u male))
	   ((aunt ?x ?a) -> (gender ?a female))
	   
	   ((grandparent ?a ?c)  <- (parent (parent ?a) ?c))
	   ((grandchild  ?a ?c)  <- (child  (child  ?a) ?c))
	   ;;
	   ((sibling ?x ?y) <- (child (parent ?x) ?y) (:neq ?x ?y))
	   ;;
	   ((uncle ?x ?u) <- (brother (parent ?x) ?u))
	   ((uncle ?x ?u) <- (husband (aunt ?x) ?u))
	   ((aunt ?x ?a)  <- (sister (parent ?x) ?a))
	   ((aunt ?x ?a)  <- (wife (uncle ?x) ?a))
	   ;;
	   ((husband ?w ?h)
	    <-
	    (gender ?w female) (spouse ?w ?h) (gender ?h male))
	   ((wife ?h ?w)
	    <-
	    (gender ?h male) (spouse ?h ?w) (gender ?w female))))
	:comment "More complex rules about family relations")
     )

;;; And the family tree of the British royal family.
;;;
(defun facts-about-royal-family ()
  (facts-about-families)
  (with-normal-output
      (facts-about-royals-v1)))

;;; There are several ways to state a database of facts.

(defun facts-about-royals-v1 ()

  (tell '((:a ?ch (name ?ch "Charles"))
	  (:a ?di (name ?di "Diana"))
	  (:a ?ha (name ?ha "Harry"))
	  (:a ?wi (name ?wi "William"))
	  (:a ?ph (name ?ph "Philip Mountbatten"))
	  (:a ?el (name ?el "Elizabeth II"))
	  (:a ?an (name ?an "Andrew"))
	  (:a ?sa (name ?sa "Sarah"))
	  ;;
	  (wife ?ch ?di)          ; This works because the variables point to the frames.
	  (son ?ch ?ha)
	  (son ?ch ?wi)
	  (son ?di ?ha)
	  (son ?di ?wi)
	  (father ?ch ?ph)
	  (mother ?ch ?el)
	  (wife ?ph ?el)
	  (son ?ph ?an)
	  (son ?el ?an)
	  (wife ?an ?sa)))
  :comment "Royal family.")

(defun facts-about-royals-v2 ()
  (tell '((:a ?ch (name ?ch "Charles"))
	  (:a ?di (name ?di "Diana"))
	  (:a ?ha (name ?ha "Harry"))
	  (:a ?wi (name ?wi "William"))
	  (:a ?ph (name ?ph "Philip Mountbatten"))
	  (:a ?el (name ?el "Elizabeth II"))
	  (:a ?an (name ?an "Andrew"))
	  (:a ?sa (name ?sa "Sarah")))
	:comment "Royal family.")
	       ;;
  (tell '((wife Charles Diana)     ; This depends on the preprocessor replacing
	  (son Charles Harry)      ; the name with the frame.
	  (son Charles William)
	  (son Diana Harry)
	  (son Diana William)
	  (father Charles "Philip Mountbatten")
	  (mother Charles Elizabeth)
	  (wife "Philip Mountbatten" Elizabeth)
	  (son "Philip Mountbatten" Andrew)
	  (son Elizabeth Andrew)
	  (wife Sarah Andrew))
	:comment "Facts by name"))

(defun facts-about-royals-v3 ()
  (tell '((:a ?ch (name ?ch "Charles"))
	  (:a ?di (name ?di "Diana"))
	  (:a ?ha (name ?ha "Harry"))
	  (:a ?wi (name ?wi "William"))
	  (:a ?ph (name ?ph "Philip Mountbatten"))
	  (:a ?el (name ?el "Elizabeth II"))
	  (:a ?an (name ?an "Andrew"))
	  (:a ?sa (name ?sa "Sarah"))
	  ;;
	  (wife Charles Diana)     ; This depends on the preprocessor replacing
	  (son Charles Harry)      ; the name with the frame.
	  (son Charles William)
	  (son Diana Harry)	; Oddly enough, it gave preprocessor warnings,
	  (son Diana William)	; but the right result!
	  (father Charles "Philip Mountbatten")
	  (mother Charles Elizabeth)
	  (wife Philip Elizabeth)
	  (son Philip Andrew)
	  (son Elizabeth Andrew)
	  (wife Sarah Andrew))
	:comment "Royal family."))


;;; ROYAL FAMILY QUERIES
;;;

(defun queries-about-royal-family ()
  (with-normal-output
      
  (ask '((grandfather William ?gf)
	 (grandmother William ?gm))
       :comment "Who are William's grandfathers and grandmothers?")
  ;;
  (ask '((uncle William ?u)
	 (aunt William ?v))
       :comment "Who are William's aunts and uncles?")
  ;;
  (ask '((parent William ?u))
       :comment "Who are William's parents?")
  ;;
  (ask '((:all-paths ((child Charles ?b)) ((gender ?b female))))
       :comment "Are all Charles' children female?")
  ;;
  (ask '((:all-paths ((child Charles ?b)) ((gender ?b male))))
       :comment "Are all Charles' children male?")
  ))
