;;;   -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-    ;;;
;;;                                                                       ;;;
;;;  Copyright (c) 1990 by James Crawford.                                ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; Set Theory.
;;;
;;; We implement the axioms given in "Mechanization of Analytic Reasoning
;;; about Sets" by Alan F. McMichael in AAAI-91 (the axioms are in column
;;; one on page 428).  We then show the commutivity and associativity of
;;; intersection.  Commutivity could not be shown using hyperresolution
;;; as recently as 1976 (McCharen, Overbeek, & Wos: "Problems and Experiments
;;; for and with Automated Theorem-Proving Programs. IEEE Transactions on
;;; Computers C-25:773-782).  Using linear resolution and a simple weighting
;;; strategy McMichael claims that commutivity can be shown in 130 steps
;;; and associativity in 19000 steps.

(in-package :CL-USER)


(defun facts-about-set-theory ()

  (a-assert "Taxonomy"
	    '((:taxonomy (things (representatives)))))

  (a-assert "Slots"
	    ;; We use t-slot to avoid application of any
	    ;; rules defined for slot in background KB.
	    '((:slot t-equal (sets sets))
	      (:slot t-subseteq (sets sets)
	       :comment "Subset or equal.")
	      (:slot t-inter (sets sets sets)
	       :comment "(t-inter x y z) <=> x = y intersect z.")
	      (:slot t-member (objects sets))))

  (a-assert "Representatives"
	    '(;; We use representatives to implement universal
	      ;; quantification.
	      (:slot rep (sets representatives)
	       :cardinality 1)
	      (:rules sets
	       ;; Create representative on demand:
	       ((rep ?set ?x) <- (:forc ?x (rep ?set ?x)))
	       ;; Representatives are members:
	       ((rep ?set ?x) -> (t-member ?x ?set)))))

  (a-assert "Axiom 1"
	    '(;; x=y iff x subseteq y and y subseteq x
	      (:rules sets
	       ((t-equal ?x ?y) -> (t-subseteq ?x ?y) (t-subseteq ?y ?x))
	       ((t-equal ?x ?y) <- (t-subseteq ?x ?y) (t-subseteq ?y ?x)))))

  (a-assert "Axiom 2"
	    '(;; x subseteq y iff for all z, if z in x then z in y.
	      (:rules sets
	       ((t-subseteq ?x ?y) (rep ?x ?z) -> (t-member ?z ?y))
	       ((t-subseteq ?x ?y) <- (rep ?x ?z) (t-member ?z ?y)))))

  (a-assert "Axiom 3"
	    '(;; x in (y intersect z) iff x in y and x in z
	      (:rules objects
	       ((t-member ?x ?i) (t-inter ?i ?y ?z)
		->
		(t-member ?x ?y) (t-member ?x ?z))
	       ((t-member ?x ?i)
		<-
		(t-inter ?i ?y ?z)
		(t-member ?x ?y) (t-member ?x ?z))))))


(defun queries-about-set-theory ()
  (a-assert "Set up a intersect b and b intersect a."
	    '((:a ?a (name ?a "A"))
	      (:a ?b (name ?b "B"))
	      (:a ?a-b (name ?a-b "A-B"))
	      (:a ?b-a (name ?b-a "B-A"))

	      (t-inter ?a-b ?a ?b)
	      (t-inter ?b-a ?b ?a)))

  (a-query "Commutivity"
	   '((t-equal a-b b-a)))

  (a-assert "Set up for associativity."
	    '((:a ?s1 (name ?s1 "S1"))
	      (:a ?s2 (name ?s2 "S2"))
	      (:a ?s3 (name ?s3 "S3"))

	      (:a ?s1-s2 (name ?s1-s2 "S1-S2"))
	      (:a ?s2-s3 (name ?s2-s3 "S2-s3"))
	      (:a ?s1-s2--s3 (name ?s1-s2--s3 "S1-S2--S3"))
	      (:a ?s1--s2-s3 (name ?s1--s2-s3 "S1--S2-S3"))

	      (t-inter ?s1-s2 ?s1 ?s2)
	      (t-inter ?s2-s3 ?s2 ?s3)
	      (t-inter ?s1-s2--s3 ?s1-s2 ?s3)
	      (t-inter ?s1--s2-s3 ?s1 ?s2-s3)))

  (a-query "Associativity"
	   '((t-equal s1-s2--s3 s1--s2-s3))))
