(in-package :user)

;;; The sterilization example shows us how to draw a universally
;;; quantified conclusion using only the Algernon machinery for
;;; reasoning about individuals.

;;; This example responds to a problem posed by John McCarthy, in
;;;   J. McCarthy, "Generality in artificial intelligence", in
;;;   ACM Turing Award Lectures:  The First Twenty Years.
;;;   NY:  ACM Press/Addison-Wesley, 1987, pp.257-267.

;;; "What can't be done without more of predicate calculus than Prolog
;;; allows is universal generalization.  Consider the reationale of
;;; canning.  We say that a container is sterile if it is sealed and
;;; all of the bacteria in it are dead.  This can be expressed as a
;;; fragment of a Prolog program as follows:
;;;     sterile(X) :- sealed(X), not alive-bacterium(Y,X).
;;;     alive-bacterium(Y,X) :- in(Y,X), bacterium(Y), alive(Y).
;;;
;;; However, a Prolog program incorporating this fragment directly can
;;; sterilize a container only by killing each bacterium individually
;;; and would require that some other part of the program successively
;;; generate the names of the bacteria.  It cannot be used to discover
;;; or rationalize canning -- sealing the container and then heating
;;; it to kill all the bacteria at once.  The reasoning rationalizing
;;; canning involves the use of quantifiers in an essential way."  (p.261)

(defun facts-about-steril ()
  (facts-about-representatives)

  (tell '((:taxonomy (physical-objects
		      (containers)
		      (living-things
		       (bacteria))))))
  
  (tell '((:slot in (physical-objects containers)
	   :comment "(in ?po ?c) = ?po is in the container ?c.")
	  (:slot bacteria-in (containers sets)
	   :comment "(bacteria-in ?c ?s) = The set of bacteria in ?c is ?s.")
	  (:slot temperature (physical-objects :number) :cardinality 1)
	  (:slot sterile (containers booleans)
	   :comment "(sterile ?c true) = ?c is sterile.")
	  (:slot sealed (containers booleans)
	   :comment "(sealed ?c true) = ?c is sealed.")
	  (:slot dead (living-things booleans)
	   :comment "(dead ?x true) = ?x is dead.")))

  (tell '((:rules containers

	   ;; This rule is really a type restriction on the slot
	   ;; bacteria-in -- the set of bacteria in a container
	   ;; must be a subset of the set of all bacteria:
	   ((bacteria-in ?c ?s) -> (imp-superset ?s bacteria))

	   ;; Any member of the set of bacteria in a container must be
	   ;; in the container:
	   ((bacteria-in ?c ?s) (member ?s ?b) -> (in ?b ?c))

	   ;; We are willing to create a frame for the set of bacteria in
	   ;; a container if we need one:
	   ((bacteria-in ?c ?s) <- (:forc ?s (bacteria-in ?c ?s)))

	   ;; A container is sterile if its sealed and all the bacteria in it are dead:
	   ((sterile ?c true)
	    <-
	    (sealed ?c true)
	    (bacteria-in ?c ?bac)
	    (representative ?bac ?r)
	    (dead ?r true)))

	  (:rules physical-objects
	   ;; Physical objects inherit the temperature of their container:
	   ((temperature ?po ?t) <- (in ?po ?c) (temperature ?c ?t)))

	  (:rules bacteria
	   ;; How to kill a bacteria:
	   ((dead ?b true) <- (temperature ?b ?t) (:test (>= ?t 212)))))))


(defun queries-about-steril ()
  (with-normal-output
  
      (tell '((:a ?c (name ?c "can")
	             (isa ?c containers))
	             (sealed ?c true)
	      (temperature ?c 300))
	    :comment "A sealed container is heated to 212 degrees")
    
    (ask '((:the ?c (name ?c "can"))
	   (sterile ?c true))
	 :comment "Is the container sterile?")))


(defun facts-about-representatives ()

  (tell '((:taxonomy (objects
		      (representatives)))))

  (tell '((:slot representative (sets representatives) :cardinality 1
                 :comment "(representative ?s ?r) = The representative of set ?s is ?r.")))
  
  (tell '((:rules sets

	   ;; Every set has a representative:
	   ((representative ?s ?r) <- (:forc ?r (representative ?s ?r)))

	   ;; Every representative is a member of the set:
	   ((representative ?s ?r) -> (member ?s ?r) (isa ?r ?s))
	   ((representative ?s ?r)
	    (isa ?r ?s)
	    (imp-superset ?s ?s2)
	    ->
	    (isa ?r ?s2))
	   ))))
