;;; -*- Mode:Common-Lisp; Package:USER; Base:10 -*-

;;; USER.LISP
;;; Copyright (c) 1990 by Coy Clifton Day, Jr.
;;; User interface for belief network systems.


;;; Belief network is initialized by the appropriate domain file,
;;; e.g. TRIAL.LISP for the simple trial example and
;;; PMES.LISP for the Portfolio Management Expert System.

;;; This code handles all interaction between the belief network
;;; code and the user.  Without this file, new evidence is entered
;;; by setting the appropriate slot value, program execution is
;;; monitored using the Algernon trace facility, and final results
;;; are examined by querying or viewing the frame contents for
;;; the nodes.


(defun bnet ()

  (loop
   (format t "~%bnet> ")
   (let ((command (read)))
     (case command
       ('help (bnet-help))
       ('exit (return t))
       ('pmes (load-pmes))
       ('forget (reset-algy) (load-kb 'domain)
		(format t "~% Forgetting all data.~%")
		(bnet-status))
       ('new (bnet-new))
       ('show (bnet-show))
       ('status (bnet-status))
       ('what (bnet-what))
       ('who (bnet-who))
       ('needs (bnet-needs))
       ('ignore (bnet-ignore))
       ('why (bnet-why))
       ('how (bnet-how))
       (t (if (symbolp command)
	      (format t "~% Unknown command ~A.~%" command)
	    (print (eval command))))))))


(defun bnet-help ()
  (format t "~%~%            BNET Commands~%
       Help        Display this message
       Exit        Quit BNET
       PMES        Load the Stock Market domain
       Forget      Forget all data entered for the current domain
       New         Allow user to enter new evidence
       Show        Show values for selected nodes
       Status      Display information about state of the network
       What        Detailed information about a node
       Who         Display a frame
       Needs       Display needed evidence
       Ignore      Undo the effect of a piece of evidence
       Why         Describe the anticipatory support for a node
                    (i.e. explain the need for certain information)
       How         Describe the causal support for a node
                    (i.e. explain how a value was generated)
       (Algy)      Go to Algy to run Algy commands~%"))


(defun load-pmes ()

  (load (format nil "~Apmes.lisp" @bnet-path))
  (reset-algy)
  (load-kb 'belief)
  (pmes-tree)
  (pmes-init)
  (init)
  (kb-snapshot 'domain))


(defun bnet-forget ()
  
  (reset-algy)
  (load-kb 'domain))

(defun init ()
  ;;; Initialize the belief network by setting the lambda values
  ;;; of all leaf nodes to be vectors of ones.  This causes propagation
  ;;; of belief values across the network and so may take a long time.

  (mapc #'(lambda (leaf)
	    (a-assert (format nil "Initializing ~a~%" leaf)
	      `((lambda ,leaf (:quote ,(mapcar #'(lambda (junk) 1.0)
					       (get-slot leaf 'values)))))))
	(query '((leaf current-context ?leaf)) :collect '?leaf))
  (terpri) (bnet-status))


(defun bnet-new ()
  ;;; Allows the user to present evidence.

  (format t "~% Which node do you have evidence for? ")
  (let ((node (read)))
    (terpri)
    (new-evidence node 
      (mapcar #'(lambda (value)
		  (format t "~% Enter the probability of your ~
                                observations given ~A : " value)
		  (read))
	      (get-slot node 'values)))
    (terpri) (bnet-status)))


(defun bnet-show ()
  ;;; Display requested node information.

  (format t "~% Which slot do you want to examine? ")
  (let ((slot (read)))
    (format t "~% For which node? ")
    (let ((node (read)))
      (format t "~%~%  ~A~%" (get-slot node slot)))))


(defun bnet-status ()
  ;;; Displays important information about the present state of
  ;;; the belief network.

  (let ((root (get-slot 'current-context 'root))
	(unknown (car (needs))))
    (format t "~% Present belief in ~A, ~A:~%" root (get-slot root 'description))
    (print-vals root 'belief)
    (format (not (null unknown)) "~% Need information about ~a.~%" (car (needs)))))


(defun bnet-what ()
  ;;; Provides a description of the indicated node, i.e.
  ;;; its meaning and possible values.

  (format t "~% Which node? ")
  (let* ((node (read))
	 (desc (get-slot node 'description)))

    (when desc
      (format t "~%~%   ~A" desc)
      (format t "~%~%   Possible values are:          Belief in value:~%")
      (print-vals node 'belief))))


(defun bnet-who ()
  ;;; Display the indicated frame, like the Algernon "who".

  (format t "~% Which frame do you want to examine? ")
  (let ((frame (read)))
    (a-assert "Examining frame" `((:show ,frame)))
    (terpri)))


(defun bnet-needs ()
  ;;; Display the nodes for which there is no data.  They are displayed
  ;;; in order of importance.

  (let ((unknowns (needs)))
    (cond ((null unknowns)
	   (format t "~% All evidence has been entered or ignored.~%"))
	  (t (format t "~% Need data for the following nodes:~%")
	     (mapc #'(lambda (node)
		       (format t "     ~A~%" node))
		   unknowns)))))


(defun needs ()
  ;;; Returns a list of the anticipatory nodes, in order of importance
  ;;; as determined by the variation in their belief values.

  (mapcar #'car (sort (mapcar #'(lambda (leaf)
				  (list (car leaf)
					(- (apply #'max (cadr leaf))
					   (apply #'min (cadr leaf)))))
			      (query '((leaf current-context ?a)
				       (:unp (dummy-child ?a ?b))
				       (belief ?a ?bel))
				     :collect '(?a ?bel)))
		      #'> :key #'cadr)))


(defun bnet-ignore ()
  ;;; Sends a "null" message to the requested leaf node.
  ;;; Has two effects.  First, it will erase previous information
  ;;; that was entered for the node.  Second, it will keep the
  ;;; system from asking for the node's value since it creates
  ;;; a dummy child for the new if one previously exists.
  ;;; If no evidence has been given for the node or if the node
  ;;; is already being ignored, this command has no effect on
  ;;; the values in the network.

  (format t "~% Which node do you want to ignore? ")
  (let ((node (read)))
    (new-evidence node
      (mapcar #'(lambda (junk) 1.0)
	      (get-slot node 'values)))
    (format t "~% Ignoring ~a~%" node)
    (let ((need (car (needs))))
    (format (not (null need)) "~% Need information about ~a.~%" need))))


(defun bnet-why ()
  ;;; Describe the anticipatory support a node has from above, i.e. why the
  ;;; system wants its value or why it assigned it its pi value.

  (format t "~% Which node do you want the anticipatory support for? ")
  (let ((node (read)))
    (format t "~%~%   Support (from above) for node ~A:~%" node)
    (print-vals node 'pi) (terpri)
    (let ((parent (get-slot node 'parent)))
      (cond (parent
	     (format t "   Support from ~A:~%" parent)
	     (mapc #'(lambda (value x)
		       (format t "     ~30A ~A~%" value x))
		   (get-slot parent 'values)
		   (get-slot node 'pi-message)))
	    (t (format t "~%   This support is a system constant.~%"))))))


(defun bnet-how ()
  ;;; Describe the diagnostic support a node has from below, i.e. how the
  ;;; user's evidence affected the belief value for the node.

  (format t "~% Which node do you want the diagnostic support for? ")
  (let* ((node (read))
	 (children (query `((child ,node ?child)) :collect '?child)))

    (format t "~%~%   Combined support (from below) for ~A:~%" node)
    (print-vals node 'lambda) (terpri)
    (cond (children
	   (mapc #'(lambda (child)
		     (format t "   Support from ~A:~%" child)
		     (mapc #'(lambda (value x)
			       (format t "     ~30A ~A~%" value x))
			   (get-slot node 'values)
			   (car (query `((lambda-message ,node ,child ?lambda-msg))
				       :collect '?lambda-msg))))
		 children))
	  (t (format t "~%   This support was obtained ~
                               directly from the user.~%")))))


(defun new-evidence (node lambda-msg)
  ;;; Assert new evidence to a leaf node by sending a lambda
  ;;; message from its dummy child.  A dummy is used
  ;;; so that the lambda message can be the same slot that
  ;;; is used between "real" nodes without making the dummy
  ;;; part of the network, in which case it would receive
  ;;; messages and compute its belief value.

  (a-assert nil
    `((:forc ?node2 (dummy-child ,node ?node2))
      (delete-lambda-msg ,node ?node2)
      (lambda-message ,node ?node2 (:quote ,lambda-msg)))))


(defun print-vals (node slot)
  ;;; Print the strings in the values slot of node along
  ;;; with the corresponding vector elements in the given slot.

  (mapc #'(lambda (value x)
	    (format t "     ~30A ~A~%" value x))
	(get-slot node 'values)
	(get-slot node slot)))


(defun get-slot (frame slot)
  ;;; Return the first value in slot of frame.

  (car (query `((,slot ,frame ?x)) :collect '?x)))


(defun exit-query (x)
  (throw 'query x))

(defun query (queries &key collect execute)
  (silent-output)
  (if (and collect execute)
      (error "Only one of :collect :execute is allowed."))
  (catch 'query
    (let ((query-results))
      (flet ((collect (thing)
	       (push thing query-results)))
	(a-query nil
		 (nconc (mapcar #'(lambda (q)
				    (if (keywordp (car q))
					q
					(list :retrieve q)))
				queries)
		     (cond (collect `((:apply ,#'collect (,collect))))
			(execute `((:apply ,(car execute) ,(cdr execute))))))))
      query-results)))

