;;;   -*- Syntax: Common-Lisp; Package: CL-USER; Base: 10; Mode: LISP -*- ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#+(or GCL LUCID) (in-package :user)
#-(or GCL LUCID) (in-package :cl-user)


;;;
;;; PEANO
;;;
;;; Encode a part of Peano arithmetic using frames.  The main purpose of
;;; this example is to test Algernon on large knoweldge-bases.
;;;
;;; Modified (mh) 17 Sep 1996 to work with the AAM.  Original author unknown.
;;;


(defun facts-about-peano ()

  (acom-reset)
  
  (a-assert "Numbers" '((:taxonomy (objects
				    (numbers zero)
				    (domains integers)))))

  (a-assert "Slots" '((:slot succ (numbers numbers)
                             :cardinality 1)
                      (:slot prev (numbers numbers))
                      (:slot num-value (numbers :NUMBER)
                             :comment "Equivalent in base 10")
                      (:slot sum (numbers numbers numbers)
                             :comment "(sum x y z) = x + y = z")
                      (:slot diff (numbers numbers numbers)
                             :comment "(diff x y z) = x - y = z")
                      (:slot mult (numbers numbers numbers)
                             :comment "(mult x y z) = x * y = z")

		      ;; For controls
		      (:slot max      (controls :NUMBER))
		      (:slot generate (controls numbers :number))
		      ))

  (a-assert "Slot cardinalities" '((cardinality prev 1)
				   (inverse prev succ)))
  

  (a-assert "Arithmetic"
            '((:rules numbers
                      ;; 0 + x = x
                      ((sum zero ?x ?x) <-)

                      ;; x + y = z <- (x-1) + y = w and w + 1 = z
                      ((sum ?x ?y ?z) <- (prev ?x ?prev) (sum ?prev ?y ?w)
				      (succ ?w ?z))

                      ;; x - y = z <- (x-1) - y = w and w + 1 = z
                      ((diff ?x ?x zero) <-)
                      ((diff ?x ?y ?z) <- (prev ?x ?prev)
				          (diff ?prev ?y ?w)
					  (succ ?w ?z))

                      ;; x * y = z <- (x-1) * y = w and w + y = z
                      ((mult zero ?x zero) <-)
                      ((mult ?x ?y ?z) <- (prev ?x ?prev)
				          (mult ?prev ?y ?w)
					  (sum ?y ?w ?z)))))
  (a-assert "Zero"
            '((num-value zero 0)))


;;; Make frames for numbers less than n:
;;;
  (tell '((:rules domains
		  ((max ?domain ?n)
		   ->
		   (generate ?domain zero 0)))

	  (:rules domains
	   ((generate ?domain ?number ?value)
	    ->
	    (:bind ?new-value (+ ?value 1))
	    (max ?domain ?max)
	    (:test (< ?new-value ?max))
	    (:a (?new-number Numbers)
	     (succ  ?number ?new-number)
	     (num-value ?new-number ?new-value)
	     )
	    (generate ?domain ?new-number ?new-value)))
	  ))
  )

(defun add (a b)

  (ask `((:the ?addend1 (member Numbers ?addend1) (num-value ?addend1 ,a))
	 (:the ?addend2 (member Numbers ?addend2) (num-value ?addend2 ,b))
	 (sum ?addend1 ?addend2 ?sum)
	 (num-value ?sum ?sum-value)
	 (:eval (format T "~%~3D + ~3D  = ~3D",a ,b ?sum-value))
	 )
       )
  )

(defun diff (a b)

  (ask `((:the ?op1 (member Numbers ?op1) (num-value ?op1 ,a))
	 (:the ?op2 (member Numbers ?op2) (num-value ?op2 ,b))
	 (diff ?op1 ?op2 ?difference)
	 (num-value ?difference ?difference-value)
	 (:eval (format T "~%~3D - ~3D  = ~3D",a ,b ?difference-value))
	 )
       )
  )

(defun mult (a b)

  (ask `((:the ?op1 (member Numbers ?op1) (num-value ?op1 ,a))
	 (:the ?op2 (member Numbers ?op2) (num-value ?op2 ,b))
	 (mult ?op1 ?op2 ?product)
	 (num-value ?product ?product-value)
	 (:eval (format T "~%~3D * ~3D  = ~3D",a ,b ?product-value))
	 )
       )
  )



(defun queries-about-peano (&OPTIONAL (max-n 15))

  ;;Generate numbers
  (setq max-n (max max-n 15))

  (tell `((max Integers ,max-n))
	:comment (format nil "Creating numbers from 1 to ~D" (- max-n 1)))

  (format t "~%Addition.")
  (add 1 1)
  (add 1 4)
  (add 2 7)
  (add 4 5)
  (add 5 9)

  (format t "~%Subtraction.")
  (diff  1 1)
  (diff  4 1)
  (diff  7 2)
  (diff  5 4)
  (diff  9 5)
  (diff 14 3)

  (format t "~%Multiplication.")
  (mult 1 1)
  (mult 1 4)
  (mult 2 7)
  (mult 4 3)
  (mult 5 2)
  )

