;;;; -*- Mode:Lisp; Package:CL-USER; Syntax:COMMON-LISP; Base:10 -*-

(in-package :CL-USER)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  Functions for the AAM programmer
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 


(defparameter *ALL-ALGY-FILES*

    '("aam-package"		;; Set up the required packages.
      "utilities"		;; Code to handle rules, etc.
      "rule-functions"
      "rule-cache"
      "unify"                   ;; Unification code.
      "machine"			;; AAM processor and compiler.
      "trace"                   ;; Trace functions
      "monitor"                 ;; Performance monitoring
      "compiler"                ;; compiler and preprocessor.
      "prep"
      "processor"
      "inference-control"	;; Inference control for the processor.
      "print"                   ;; User-level printing routines
      "user"			;; User interface to the AAM.
      "version"			;; Load the version number.
      ))

(defparameter *ALL-ALGY-KBS*
    '("algy-core-kb"
      "algy-common-kb"
      "algy-set-theory-kb"))

(defparameter *algy-default-kb* "algy-core-kb")



(defun check-file-mod-times (files)

  (let ((need-recompile   NIL))
    (dolist (file files)

      (let ((lisp-file   (algy-source-file-name file))
	    (binary-file (algy-object-file-name file))
	    )

	(cond ((probe-file lisp-file)
	       (cond ((probe-file binary-file)
		      (let ((lisp-date   (file-write-date lisp-file))
			    (binary-date (file-write-date binary-file)))
			(when (> lisp-date binary-date)
			  (setq need-recompile T))
			))

		     (T    (setq need-recompile T))
		     ))
	      (T
	       (format *error-output* "No such file: ~(~a~)." lisp-file))))

      (when need-recompile (return))
      )

    need-recompile
    )
  )



;;; ---------------  USEFUL FUNCTIONS  -------------------


(defun load-or-compile-algy ()

  (if (check-file-mod-times (append *ALL-ALGY-KBS* *ALL-ALGY-FILES*))
      (progn
	(format *trace-output* "~2%Some Algernon source files have been updated.")
	(format *trace-output* "~%Compiling and loading Algernon...")
	(compile-algy))
    ;;else
    (progn
      (format *trace-output* "~2%Loading Algernon...")
      (load-algy)
      )
    )
  )


;;; COMPILE-ALGY

(defun compile-algy ()

  ;; Load source files
  
  (dolist (filename *ALL-ALGY-FILES*)
    (unless (string= filename "algy-core-kb")
      (load-algy-source-file filename)))

  ;; Compile ALGY

  (dolist (filename *ALL-ALGY-FILES*)
    (compile-algy-file filename))

  ;; Compile the KBs too

  (dolist (filename *ALL-ALGY-KBS*)
    (compile-algy-file filename))

  ;; Load the compiled files

  (load-algy)
  )


;;; LOAD-ALGY

(defun load-algy ()

  (dolist (filename *ALL-ALGY-FILES*)
    (load-algy-file filename))

  (load-algy-file *algy-default-kb*)
  )


(defun compile-algy-file (filename)
  (compile-file (algy-source-file-name filename) :output-file (algy-object-file-name filename))
  )


(defun load-algy-source-file (filename)


  (let ((src-filename  (algy-source-file-name filename)))
    (if (probe-file src-filename)
	(load src-filename)
      ;;else
        NIL
	))
  )

	       
(defun load-algy-object-file (filename)

  (let ((obj-filename  (algy-object-file-name filename)))
    (if (probe-file obj-filename)
	(load obj-filename)
      ;;else
        NIL
	))
  )

(defun load-algy-example-file (filename)

  (let ((ex-filename  (algy-example-object-file-name filename)))
    (if (probe-file ex-filename)
	(load ex-filename)
      ;;else
      (progn
	(setq ex-filename  (algy-example-source-file-name filename))
	(if (probe-file ex-filename)
	    (load ex-filename)
	  ;;else
	  NIL)
	))
    )
  )

	       
(defun load-algy-file (filename)

  (or (load-algy-object-file  filename)
      (load-algy-source-file  filename)
      (load-algy-example-file filename)
      )
  )

