;;;; -*- Mode:Lisp; Package:AAM; Syntax:COMMON-LISP; Base:10 -*-

(in-package :AAM)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  The AAM compiler.  Returns AAM code.
;;;
;;;  Top-level:  (AAM:aam-compile <form>)
;;;  Top-level:  (AAM:aam-compile-rule <rule-frame>)
;;;
;;; 12 Jun 1996 (mh)  Modified for Allie v2.0 instruction set.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

;;;-------------------------
;;; Bound variable analysis
;;;-------------------------
;;;
;;; Since an access path is executed left to right, we can usually
;;; determine at compile time which variables in the expression
;;; have been previously bound.  The only exception is the key of
;;; an if-needed rule.  In that case, we don't know which arguments 
;;; were bound.
;;;
;;; Think of each clause as being inside a LET form which lists
;;; all of the variables of that clause that were not mentioned
;;; in a previous clause.  Then each clause has access to the
;;; LET-bindings of previous clauses.
;;;
;;; 12 Nov 1996
;;;
;;; We can also determine at compile-time the type of the variables
;;; as long as the slots are defined at the time the expression is
;;; compiled.  We can then propagate types and do type-checking at
;;; compile-time, not at runtime.
;;;  
;;;-------------------------


(defparameter *FAIL-IS-GOOD* NIL
  "Sometimes, like during compilation of :UNP or :FAIL,
failure is considered good.  In that case, we handle the
:FAIL? command differently (it becomes :FAIL?-AND-SKIP).")

(defparameter *SUCCEED-IS-GOOD* NIL
  "Sometimes, like during compilation of :UNP or :FAIL,
failure is considered good.  In that case, we handle the
:FAIL? command differently (it becomes :SUCCEED?-AND-SKIP).")

(defparameter *NO-BINDINGS* NIL
  "Used in recursive call to COMPILE by 'compile-the'.")

(defparameter *compile-mode* NIL
  "Set to :THE or :FORC if we are compiling one of those forms.  The
compilation of :UNP is different depending on what mode the compiler is in.")

(defparameter *make-continuation* T
  "Set to NIL to suppress continuations.  Used by compile-special-form
to stop continuations for special forms.")

(defparameter *THE-var* NIL
  "The variable about which uniqueness is in question when
compiling :THE.")

(defparameter *skip-label* NIL
  "The current label to skip to.  Used during compilation.")

(defparameter *succeed-label* NIL
  "The current label to skip to when SUCCEED? clause is processed.")

(defparameter *ARGUMENT-TYPES* NIL
  "Information about argument types propagated during compilation.")

(defparameter *COMPILED-RULE-NAME* NIL
  "The rule currently being compiled.")


;;; ---------  COMPILER routines  -----------------


(defun aam-compile (path mode bindings &OPTIONAL (compiling-rule? NIL) (argument-types NIL))
  "Returns AAM machine code to be executed.  Bindings only
occur in the case where a form is compiled during execution of
a path.  For example, when a rule is defined during execution
of a rule, or when a rule is compiled at runtime.

Mode is ASSERT or QUERY."

  (cond ((null path)   NIL)

	(T  (setq *ARGUMENT-TYPES* argument-types)

	    (when (or (atom path) (atom (car path)))
	      (cerror "Return NIL"
		      "AAM compiles paths, not atoms or clauses: '~A'." path)
	      (return-from AAM-COMPILE NIL)
	      )
	    
	    ;; Preprocess returns a path.
	    
	    (let ((the-path   (preprocess path))
		  (index      1))
	      
	      (compile-path the-path index mode bindings (mapcar #'car (car bindings))
			    compiling-rule?)  ;;last arg is argument-type info
	      )
	    )
	)
  )



(defun same-set (set1 set2)

  (and (subsetp set1 set2) (subsetp set2 set1)))


(defun find-if-needed-rule-code (rule bound-vars)
  
  (let* ((code-sets   (CL-USER::rule-code rule))
	 (code        (cdr (assoc bound-vars code-sets :test #'same-set)))
	 )

    (when (null code)
      (format *trace-output*
	      "~%Compiling ~S for bound variables ~S." rule bound-vars)
      (setq code
	(gen-rule-code rule :IF-NEEDED
		       (CL-USER::rule-antecedent rule)
		       (CL-USER::rule-consequent rule)
		       1 rule bound-vars))

      (sfs:kb-delete-values rule 'CL-USER::code *value)
      (sfs:kb-put-value rule 'CL-USER::code *value
			(cons (cons bound-vars code) code-sets))
      )
    code
    )
  )
      

(defun generate-var-sets (clause)
  "Used to generate possible combinations of bound variables in
the consequent of an if-needed rule when it is activated."

  ;; clause is (slot frame arg1 arg2 arg3 ... argn)
  ;; We want about 4 to start:  (f), (f arg1), (f arg1 arg2), (f arg1 ... argn)
  ;; There will be duplicates iff the clause only has a few arguments.


  (let ((count      0)      ;; Count number of sets generated
	(sets       nil)    ;; Collect sets
	(set        nil)
	(the-clause (delete-if-not
		     #'variable-p
		     (copy-list
		      (cdr (if (negated-p clause) (negate clause) clause)))))
	)

    (push the-clause sets)  ;; Always generate for fully-bound case.
    
    (loop
      (when (or (>= count 4)
		(ground-clause-p the-clause))     ;; NIL is ground (!)
	(return))
      
      (setq set (append set (list (pop the-clause))))
      (pushnew set sets :test *equalp-test*) (incf count)
      )
    
    sets
    )
  )

			  

(defun aam-compile-rule (rule-frame &OPTIONAL (bound-variables NIL))
  "RULE-TYPE is :IF-ADDED or :IF-NEEDED"
  
  (let ((bound-vars NIL)
	(antecedent (CL-USER::rule-antecedent rule-frame))
	(consequent (CL-USER::rule-consequent rule-frame))
	(rule-type  (if (eq (CL-USER::rule-direction rule-frame) *forward)
			:IF-ADDED
		      ;;else
		        :IF-NEEDED))
	(index      (CL-USER::rule-index rule-frame))
	(root-rule  (CL-USER::rule-root  rule-frame))
	(*compiled-rule-name* rule-frame)
	)

    ;; Generate the code.
    ;;    IF-ADDED rules get one compilation.
    ;;    IF-NEEDED rules get multiple compilations to match possible
    ;;      variations on sets of variables that are bound at runtime.

    (cond ((eq rule-type :IF-ADDED)
	   ;; All vars in first clause of the
	   ;; antecedent will be bound.
	   (setq bound-vars
	     (append bound-variables
		     (variable-arguments (car antecedent))))
	   (gen-rule-code rule-frame rule-type antecedent consequent index
			  root-rule bound-vars)
	   )

	  (T    ;; IF-NEEDED
	     
	   (mapcar
	    #'(lambda (var-set)
		(setq *compiled-rule-name*
		  (intern (format nil "~A~{-~A~}" rule-frame var-set)
			  :CL-USER))
			  
		(cons var-set
		      (gen-rule-code rule-frame rule-type antecedent
				     consequent index root-rule
				     (union var-set bound-variables))))
	    (generate-var-sets (car consequent)))
	   )
	  )
    )
  )


(defun gen-rule-code (rule-frame rule-type antecedent consequent
		      index root-rule bound-vars)


  (nconc
   (compile-path-aux
    (if (eq rule-type :IF-ADDED)
	(cdr antecedent)
      ;;else
      antecedent)

    (if (eq rule-type :IF-ADDED)
	(1+ index)
      ;;else
      index)
    :QUERY bound-vars NIL (or root-rule rule-frame))
	 
   ;; Need to BRANCH 'cause the last clause may have multiple bindings.
   ;; The last clause normally isn't BRANCHed by the compiler because
   ;; there is no need if it is not in a rule.
   (append
    `(:BRANCH)
    ;;	  :PUSH :PATH ,(filter-path consequent)
    (let ((*backward-chain* NIL))
      (compile-path-aux consequent 1 :ASSERT
			(append bound-vars (mapcan #'variable-arguments
						   antecedent))
			(list :LABEL :END  :RETURN :NOTHING
			      :LABEL :FAIL :RETURN :FAILURE)
			NIL)  ;; Don't want continuations.
      )
    )
   )
  )


(defun filter-path (path)

  ;; Filter out clauses that shouldn't be in the PATH list.

  (remove-if
   #'(lambda (clause)
       (member (slot clause)
	       '(:A :ALL-PATHS :ASK :FORC :OR
		 :RETRIEVE :THE :UNP)))
   path))


(defun compile-path (path index mode bindings bound-variables
		     &OPTIONAL (compiling-rule? NIL))
  "Compiles a path."


  (append
   (unless *no-bindings*            ;; Used by :SUBR
     `(:PUSH :BINDINGS ,bindings))
   (compile-path-aux path index mode bound-variables
		     (if compiling-rule?
			 (list :LABEL :END  :RETURN :NOTHING
			       :LABEL :FAIL :RETURN :FAILURE)
		       ;;else
		       (list :LABEL :SUCC :RETURN :SUCCESS
			     :LABEL :FAIL :RETURN :FAILURE))
		     compiling-rule?))
  )


(defun compile-path-no-return (path index mode bindings bound-variables
			       &OPTIONAL (compiling-rule? NIL))
  "Compiles a path.  Used for recursive compiles."

  (append
   `(:PUSH :BINDINGS ,(if (null bindings) NIL (list bindings)))
   (compile-path-aux path index mode bound-variables NIL
		     compiling-rule?))
  )


(defun compile-path-aux (path index mode bound-variables code
			 &OPTIONAL (compiling-rule? NIL))
  "Recursive form of compile-path."

;;  (break "In Compile-path-aux")
  
  (if (null path)
      code
    ;;ELSE
    (if (null (cdr path))
	(compile-clause (car path) index mode bound-variables
			code :LAST-CLAUSE compiling-rule?)
      ;;ELSE

      (nconc
       (compile-clause (car path) index mode bound-variables
		       NIL  NIL  compiling-rule?)
       (compile-path-aux (cdr path) (1+ index) mode
			 (union (variable-arguments (car path))
				bound-variables)
			 code
			 compiling-rule?)
       )
      )
    )
  )

#|      (compile-clause (car path) index mode bound-variables
		      (compile-path-aux (cdr path) (1+ index) mode
					(union (variable-arguments (car path))
					       bound-variables)
					code
					compiling-rule?)
		      NIL
		      compiling-rule?)
|#


(defun compile-clause (clause index mode bound-variables code
			      &OPTIONAL (last-clause? NIL) (compiling-rule? NIL))
  "Compiles one clause."

  ;;; Possibly change the mode...

  ;; If in QUERY mode and analysis shows that the clause will be
  ;; ground at runtime, we can make it VERIFY. 
  ;; If the status is unknown, we will make it QUERY-OR-VERIFY.
  ;;   (we try to avoid QUERY-OR-VERIFY).

  ;; Similarly, if we are in ASSERT mode and analysis shows that after the
  ;; :SUBST, the clause will not be ground, we need to do a
  ;; query instead of an assert.

  (if (and (eq mode :ASSERT)
	   (not (will-be-ground-p clause bound-variables)))
      (setq mode :QUERY))


  ;;; Check for violation of path restriction.
  ;;; Not valid for some special forms

  (let ((frame (frame clause)))
    (when (and (variable-p frame)
               (not (ignore-path-restriction-for (slot clause)))
	       (not (member frame bound-variables :test #'eq)))
      (aam-compile-warning :PATH-RESTRICTION clause *compiled-rule-name*)
      ))


  ;;; 12 Nov 1996 (mh)
  ;;; Check for argument type violations
  (let ((slot  (slot  clause))
	(frame (frame clause))
	(args  (arguments clause))
	)
    (validate-argument-types slot (cons frame args))
    )


  ;;; Compile the code

  ;;; If we are compiling a rule, we may need to create a continuation, but 
  ;;; only with non-first clauses of if-added rules that are queries.
  ;;; Thus compile-query needs to know if we are compiling a rule.

  ;;; All types need to know if we are compiling the last clause because there is no :BRANCH
  ;;; after the last clause.

  (if (aam-special-form-p (slot clause))
      (compile-special-form clause index mode bound-variables code last-clause?
			    compiling-rule?)
    ;;ELSE
      (case mode
	(:QUERY  (compile-query  clause index bound-variables code
					last-clause? compiling-rule?))
	(:ASSERT (compile-assert clause index bound-variables code
				 last-clause? compiling-rule?))
	(:VERIFY (compile-verify clause       bound-variables code
				 last-clause?))
	)
      )
  )


(defun compile-query (clause index bound-variables code
		      last-clause? compiling-rule?)

  (append

   `(:CLAUSE ,clause)

   (when bound-variables '(:SUBST))

   (when (and compiling-rule? *make-continuation*)
     `(:CONTINUATION ,(continuation-rule-name compiling-rule? index)))

   (when *backward-chain* '(:SAVE :ACTIVATE-IN :SELECT))
   
   (cond (*FAIL-IS-GOOD*
	  `(:QUERY :FAIL?    ,*skip-label* :MERGE-BINDINGS))
	 
	 (*SUCCEED-IS-GOOD*
	  `(:QUERY :SUCCEED? ,*skip-label*))
	 
	 (T
	  `(:QUERY :FAIL? :FAIL :MERGE-BINDINGS))
	 )

   (unless last-clause? '(:BRANCH))

   code
   )
  )


(defun compile-assert (clause index bound-variables code last-clause?
		       compiling-rule?)

  ;; Modified 6 Sep 1996 (mh)
  ;; Need to branch around the ACTIVATE if the result was
  ;; already known.

  (declare (ignore last-clause?))

  (let ((known-label  (new-label)))

    (append
     
     `(:CLAUSE ,clause)

     (when (and bound-variables (not (ground-clause-p clause)))
       '(:SUBST))
     
     (when (and compiling-rule? *make-continuation*)
       `(:CONTINUATION ,(continuation-rule-name compiling-rule? index)))
     
     (cond (*FAIL-IS-GOOD*
	    `(:ASSERT :FAIL?    ,*skip-label*))

	   (*SUCCEED-IS-GOOD*
	    `(:ASSERT :SUCCEED? ,*skip-label*))

	   (T
	    `(:ASSERT :KNOWN? ,known-label :FAIL? :FAIL))
	   )
     
     (when *forward-chain* '(:ACTIVATE-IA :BRANCH))
     
     `(:LABEL ,known-label)
     
     code
     )
    )
  )


(defun compile-verify (clause bound-variables code last-clause?)

  (append

   `(:CLAUSE ,clause)

   (if (and bound-variables (not (ground-clause-p clause)))
       '(:SUBST))

   (cond (*FAIL-IS-GOOD*
	  `(:VERIFY :FAIL?    ,*skip-label*))
	 
	 (*SUCCEED-IS-GOOD*
	  `(:VERIFY :SUCCEED? ,*skip-label*))
	 
	 (T
	  `(:VERIFY :FAIL? :FAIL))
	 )

   (unless last-clause? '(:BRANCH))

   code
   )
  )



(defun compile-special-form (clause index mode bound-variables code
			     last-clause? compiling-rule?)
  
  ;;; :LAST-CLAUSE means no need to branch after this expression.
  ;;; because it doesn't cause the search tree to branch.

  ;;; No continuations for special forms

  (let* ((form                 (slot clause))
	 (*make-continuation*  (if *make-continuation*
				   (continuable-p clause)))
	 )
    (ccase form
      (:A
       (compile-a clause mode bound-variables code last-clause?
		  compiling-rule?))

      (:ALL-PATHS (compile-all-paths clause mode bound-variables code
				     NIL compiling-rule?))

      (:ALL-PATHS-PART-1 (compile-all-paths-part-1 clause mode bound-variables code
						   NIL compiling-rule?))

      (:ALL-PATHS-PART-2 (compile-all-paths-part-2 clause mode bound-variables code
						   NIL compiling-rule?))

      (:ANY       (compile-any clause mode bound-variables code
			       last-clause? compiling-rule?))

      (:ASK      (compile-ask     clause mode bound-variables
				              code :LAST-CLAUSE))

      (:BIND     (compile-special-with-merge  clause mode bound-variables
				              code :LAST-CLAUSE))
      ((:BOUNDP :UNBOUNDP)
       (compile-special-with-fail  clause mode NIL code :LAST-CLAUSE))

      (:BRANCH   (nsubstitute
		  :BRANCH-USER :BRANCH
		  (compile-special-with-merge  clause mode bound-variables
				              code NIL)
		  :count 1))

      (:CLEAR-SLOT  (compile-special  clause mode bound-variables 
				      code :LAST-CLAUSE))

      (:DELETE      (compile-delete  clause bound-variables code)) 

      (:EVAL     (compile-special             clause mode bound-variables
				              code :LAST-CLAUSE))
      ((:FAIL :UNP)
       (compile-unp clause mode bound-variables code last-clause?
		    compiling-rule?))
       
       
      (:FORC
       (compile-forc clause mode bound-variables code last-clause?
		  compiling-rule?))

      (:NEQ      (compile-special-with-fail   clause mode bound-variables
				              code :LAST-CLAUSE))

      ((:NO-COMPLETION :NO-CONTINUATION)
                 (let ((*make-continuation* NIL))
		   (compile-path-aux (cdr clause) index mode bound-variables
				     code compiling-rule?)))

      (:OR       (compile-or  clause index mode bound-variables code
			      last-clause? compiling-rule?))

      (:RETRIEVE (let ((*backward-chain* NIL)) 
;;                   (append `(:PUSH :PATH (,(second clause)))
		   (compile-clause (second clause) index mode bound-variables
				   code last-clause? compiling-rule?)))

      (:RULES    (compile-special             clause mode bound-variables
				              code :LAST-CLAUSE))
      (:SHOW     (compile-special             clause mode bound-variables
				              code :LAST-CLAUSE))
      (:SLOT     (compile-special             clause mode bound-variables
				              code :LAST-CLAUSE))
      (:SRULES   (compile-special             clause mode bound-variables
				              code :LAST-CLAUSE))

      (:STOP     (compile-special             clause mode bound-variables
				              code :LAST-CLAUSE))

      (:TAXONOMY (compile-taxonomy            clause bound-variables
					      code compiling-rule?))

      (:TEST     (compile-special-with-fail   clause mode bound-variables
				              code :LAST-CLAUSE))
      (:THE      (compile-the clause mode bound-variables code last-clause?
		              compiling-rule?))

;;;      (:ZERO-OR-ONE
;;;       (compile-zero-or-one clause mode bound-variables code last-clause?
;;;			    compiling-rule?))
      )
    )
  )


(defun compile-special (clause mode bound-variables code last-clause?)
  "Compile a special form."
  
  (declare (ignore mode bound-variables))

  ;; Prepares a call to the lower-level language (e.g. LISP)

  ;; It is too hard to test whether the clause is ground or
  ;; not, because the clause is in a different place in each
  ;; special form.  So we always do a :SUBST.

  (append
   `(:CLAUSE ,clause :SUBST)
   (list (slot clause))           ;; Contains the operation, like :BIND
   (unless last-clause?  (list :BRANCH))
   code
   )
  )


(defun compile-special-with-fail (clause mode bound-variables code last-clause?)
  "Compile a special form."
  
  (declare (ignore mode bound-variables))

  ;; Prepares a call to the lower-level language (e.g. LISP)
  (append
   `(:CLAUSE ,clause :SUBST)
   (list (slot clause) :FAIL? :FAIL :MERGE-BINDINGS)
   (unless last-clause?  (list :BRANCH))
   code
   )
  )


(defun compile-special-with-merge (clause mode bound-variables code last-clause?)
  "Compile a special form."
  
  (declare (ignore mode bound-variables))

  ;; Prepares a call to the lower-level language (e.g. LISP)
  (append
   `(:CLAUSE ,clause :SUBST)
   (list (slot clause) :MERGE-BINDINGS)  ;; Contains the operation, like :BIND
   (unless last-clause?  (list :BRANCH))
   code
   )
  )


(defun compile-subr (var path mode bound-variables compiling-rule?)
  "Returns the appropriate code."

  (declare (ignore compiling-rule?))

  ;; Tell AAM-COMPILE that we are not compiling a rule so that it will
  ;; place the return value in the RESULTS register, not in
  ;; the BINDINGS register.  This way we know where to find the
  ;; result at runtime.

  `(:SUBR ,var
	  ,(let ((*NO-BINDINGS*      T)
		 (*FAIL-IS-GOOD*     NIL)
		 (*SUCCEED-IS-GOOD*  NIL))
	     (aam-compile path mode
			  (list (mapcar #'list bound-variables))
			  NIL *ARGUMENT-TYPES*)))
  )
  
  
(defun compile-a (clause mode bound-variables code last-clause?
		  compiling-rule?)
  "Compile the :A special form."
  
  (declare (ignore mode last-clause?))

  ;; Changed (mh) 16 Sep 1996 to new :A syntax.
  ;; (:A <var>       <path>)     -- syntax 1
  ;; (:A (<var> Set) <path>)     -- syntax 2

  ;;   create a new frame and 
  ;;   assert the path for that variable.
  ;;   optionally assert that it is a member of the set.
  
  (let* ((var-or-list   (second clause))
	 (var           (if (symbolp var-or-list)
			    var-or-list
			  ;;ELSE
			    (first var-or-list)))
	 (set           (if (symbolp var-or-list)
			    NIL
			  ;;else
			    (second var-or-list)))
	 )

    (append
     `(:CLAUSE ,clause)

     (when bound-variables '(:SUBST))

     `(:KB-DEF-FRAME ,var :MERGE-BINDINGS)
     
     (when set
       (append

	;; Added reverse ISA link, even though a rule will fire to do it.
	;; Turns out the rule executes too late at runtime and error 
	;; checking routines have problems.
	;; (mh) 10 Oct 1996
	(compile-path-aux `((,*member ,set ,var)(,*isa ,var ,set)) 2 :ASSERT
			  (cons var bound-variables) NIL compiling-rule?)))

     (compile-path-aux
      (cddr clause) 1 :ASSERT (cons var bound-variables) NIL compiling-rule?)
     
     ;; We need a :BRANCH at the end, but the compilation
     ;; of the path will do that.  Unless there is no path...
     (unless (cddr clause)
       (list :BRANCH))

     code
     )
    )
  )


(defun compile-all-paths (clause mode bound-variables code last-clause?
			       compiling-rule?)

  (declare (ignore last-clause?))
  
  ;; We'll have to compile this as a SUBR since it clears
  ;; the results register if it fails, and we don't want that.

  (let ((success-label (new-label))
	(conj-id       (new-conj-id))
	)

    (nconc
     (list :SUBR NIL)

     (list
      (nconc 
       (compile-subr NIL (list (list :ALL-PATHS-PART-1 (second clause)))
		     mode bound-variables compiling-rule?)
       
       ;; If the first path fails, this is a successful :ALL-PATHS.
       ;; We don't need to execute the second path.
       
       (list :FAIL? success-label :CONJ-BRANCH conj-id)
       
       (compile-subr NIL (list (list :ALL-PATHS-PART-2 (third clause)))
		     mode (union (mapcan #'variable-arguments (second clause))
				 bound-variables) compiling-rule?)
       
       `(:SUCCEED? ,success-label)
       
       `(:CONJ-CLEAR :ALL :SKIP :FAIL :LABEL ,success-label
		     :LABEL :SUCC :RETURN :SUCCESS
		     :LABEL :FAIL :RETURN :FAILURE)
       ))

     (list :FAIL? :FAIL)

     code
     )
    )
  )


(defun compile-all-paths-part-1 (clause mode bound-variables code last-clause?
				 compiling-rule?)

  (declare (ignore last-clause? mode))

  (nconc
   (compile-path-aux (second clause) 1 :QUERY bound-variables NIL compiling-rule?)
   code)
  )
  

(defun compile-all-paths-part-2 (clause mode bound-variables code last-clause?
				 compiling-rule?)

  (declare (ignore last-clause?))
  
  (let ((conj-id :ALL))   ;; Means clear everything from the ACTIONS register.
    (nconc
     (compile-path-aux (second clause) 1 mode bound-variables NIL compiling-rule?)

     ;; The :LABEL :SUCC at the end is for the case where we are compiling
     ;; the antecedent of a rule and it doesn't have a :RETURN :SUCC at the end.

     (list :SKIP :SUCC :LABEL :FAIL :CONJ-CLEAR conj-id :SKIP :FAIL :LABEL :SUCC)

     code
     )
    )
  )


(defun compile-any (clause mode bound-variables code last-clause? compiling-rule?)

  (declare (ignore last-clause?))

  (let ((path (cdr clause)))

    (nconc
     (compile-subr NIL path mode bound-variables compiling-rule?)
     (list :FAIL? :FAIL)
     (list :ANY :MERGE-BINDINGS)
     code)
    )
  )



(defun compile-ask (clause mode bound-variables code last-clause?)

  (declare (ignore mode last-clause?))

  (let ((ask-clause (second clause))
	(skip-label   (new-label))
	(finish-label (new-label))
	)

    (when (and (variable-p (slot ask-clause))
	       (not (member (slot ask-clause) bound-variables :test #'eq)))
      (aam-compile-warning :PATH-RESTRICTION clause *compiled-rule-name*)
      )

    (when (and (variable-p (frame ask-clause))
	       (not (member (frame ask-clause) bound-variables :test #'eq)))
      (aam-compile-warning :PATH-RESTRICTION clause *compiled-rule-name*)
      )

    ;; Okay to compile.  Two cases: ground or non-ground

    (cond 
	  ;; Case 1: ground at runtime.
	  ((will-be-ground-p ask-clause bound-variables)

	   (append
	    `(:CLAUSE ,clause
	      :SUBST :ASK :FAIL? ,skip-label
		  ;;; Assert the confirmed clause
	      :CLAUSE ,ask-clause :SUBST :ASSERT :FAIL? :FAIL
	      :MERGE-BINDINGS)
	    
	    (when *forward-chain* '(:ACTIVATE-IA :BRANCH))

	    ;; The user answered NO.  We assert the negative and
	    ;; then FAIL.
	    `(:SKIP ,FINISH-LABEL
	      :LABEL ,skip-label
		  ;;; Assert the negative of the clause
	      :CLAUSE ,(negate ask-clause) :SUBST :ASSERT
	      :FAIL? :FAIL :MERGE-BINDINGS)
	    
	    (when *forward-chain* '(:ACTIVATE-IA :BRANCH))

	    `(:SKIP :END :LABEL ,FINISH-LABEL)

	    code)
	   )

	  (T   ;; Case 2: non-ground at runtime.

	   (append
	    `(:CLAUSE ,clause
	      :SUBST :ASK :MERGE-BINDINGS
	      :CLAUSE ,ask-clause :SUBST :ASSERT :FAIL? :FAIL)
	    
	    (when *forward-chain* '(:ACTIVATE-IA :BRANCH))

	    code
	    )
	   )
	  )
    )
  )


(defun compile-delete (clause bound-variables code)
  "Compile the :DELETE special form."
  
  ;; Prepares a call to the lower-level language (e.g. LISP)

  (append
   `(:CLAUSE ,(second clause))
   (when (and bound-variables (not (ground-clause-p (second clause))))
     '(:SUBST))
   (list (slot clause))           ;; Contains the operation, like :BIND
   code
   )
  )


(defun compile-or (clause index mode bound-variables code last-clause?
		   compiling-rule?)
  "Compile the :OR special form."

  (declare (ignore index last-clause?))
  
  ;; :OR evaluates each path and if successful it skips to the
  ;; end without evaluating the others.

  ;; We use :SUBR here, but send in NIL as the variable
  ;; to be saved.  At runtime, this causes all the variables
  ;; to be saved, which is what we want for :OR.
  
  ;; The general pattern will be 
  ;;    :SUBR     NIL   <path1 code>
  ;;    :SUCCEED? success-label
  ;;    :SUBR     NIL   <path2 code>
  ;;    :SUCCEED? success-label
  ;;     ...
  ;;    :SUBR     NIL   <pathn code>
  ;;    :SUCCEED? success-label
  ;;    :LABEL    :FAIL
  ;;    success-label


  (let ((success-label       (new-label)))

    (nconc
     (mapcan 
      #'(lambda (path)
	  (nconc
	   (compile-subr NIL path mode bound-variables compiling-rule?)
	   (list :SUCCEED? success-label)))
      (cdr clause))

     (nconc
      (list :SKIP :FAIL :LABEL success-label :MERGE-BINDINGS)
      code)
     )
    )
  )


(defun compile-unp (clause mode bound-variables code last-clause?
		    compiling-rule?)
  "Compile the :UNP and :FAIL special forms."
  
  (declare (ignore mode last-clause?))

  (append 
   (compile-subr *the-var* (cdr clause) :QUERY bound-variables compiling-rule?)

   (case *compile-mode*
     
     (:FORC `(:SUCCEED? ,*succeed-label* :MERGE-BINDINGS))

     ((NIL)    ;; Just a plain old :UNP or :FAIL
      ;; Normally, *succeed-is-good* means do :SUCCEED?,
      ;; but since we are :UNP, we need to do the opposite.
      (cond (*SUCCEED-IS-GOOD*
	     (append `(:FAIL? ,*skip-label*
					  :MERGE-BINDINGS :BRANCH)
		     code))
	    (T
	     (append `(:SUCCEED? :FAIL :MERGE-BINDINGS :BRANCH)
		     code))
	    )
      )
     )
   )
  )


(defun compile-forc (clause mode bound-variables code last-clause?
			    compiling-rule?)
  "Compile the :FORC special form."
  
  (declare (ignore last-clause?))

  (let* ((*succeed-label*  (new-label))
	 (*compile-mode*   (car clause))
	 (var-and-type     (second clause))
	 (*the-var*        (if (symbolp var-and-type)
			       var-and-type
			     ;;ELSE
			     (first var-and-type)))
	 )
    
    (append

     (compile-path-aux `((:UNP ,@(cddr clause))) 1 :QUERY 
			     bound-variables NIL compiling-rule?)

     (compile-path-aux `((:A ,var-and-type ,@(cddr clause))) 1 mode
		       bound-variables NIL compiling-rule?)

     `(:LABEL ,*succeed-label* :MERGE-BINDINGS :BRANCH)
     code
     )
    )
  )


(defun compile-taxonomy (clause bound-variables code compiling-rule?)
  "Compile the :TAXONOMY special form."
  
  (compile-path-aux (make-taxonomy (second clause)) 1 :ASSERT
		    bound-variables code compiling-rule?)
  )


(defun compile-the (clause mode bound-variables code last-clause?
                           compiling-rule?)
  "Compile the :THE special form."
  
  (declare (ignore last-clause?))

  ;; The path in the clause must be processed as a subroutine call,
  ;; because we can't have it branch and continue on outside of the
  ;; :THE form.

  (let* ((A-label          (new-label))
	 (SUCCEED-label    (new-label))
	 (var-and-type     (second clause))
	 (var              (if (symbolp var-and-type) var-and-type
			     (first var-and-type)))
	 (path             (cddr clause))
	 )
    
    (append
     (compile-subr var path :QUERY bound-variables compiling-rule?)

     `(:FAIL? ,A-label 
	      :UNIQUE? ,var ,SUCCEED-LABEL
	      :SKIP    :FAIL
	      :LABEL ,A-label 
       )
     
     (compile-path-aux `((:A ,var-and-type ,@path)) 1 mode bound-variables NIL compiling-rule?)

     `(:LABEL ,SUCCEED-label :MERGE-BINDINGS)
     code
     )
    )
  )


;;;------------  UTILITIES  ------------------

(defun rule-pc (rule &OPTIONAL (stream *standard-output*) (version-to-print nil))
  "Pretty prints the rule's code, either all versions or a numbered version (0-based).
The numbered version is relevant only for if-needed rules."

  (let ((version-num  0))
    (if (if-needed-rule-p rule)
	(dolist (version (car (sfs::kb-get-values rule 'CL-USER::code
						  'CL-USER::value)))
	  (when (or (null version-to-print)
		    (= version-to-print version-num))
	    (format stream "~2%-------------------------")
	    (format stream "~%version ~D:  ~S bound."
		    version-num (car version))
	    (pc (cdr version) stream)
	    )
	  (incf version-num)
	  )
      ;;else
      (pc (car (sfs::kb-get-values rule 'CL-USER::code 'CL-USER::value)) stream))
    )
  )


(defun pc (compiled-code &OPTIONAL (stream *standard-output*))
  "Pretty prints the compiled code."
  
  (format stream "~%-------  code  ----------")

  (let ((op            NIL)
	(num-args      0)
	(code          compiled-code)
	(labeled-line  NIL))

    (loop
      (when (null code)
	(return))

      (setq op        (pop code)
            num-args  (lookup-opcode-numargs op))

      (unless num-args
	(format *error-output*
		"~%AAM internal error: ~S is an unknown opcode." op)
	(break "Fix it!")
	)

      ;; We could be printing a label, printing the rest of a labeled
      ;; line, or printing an ordinary instruction.

      (cond ((eq op :LABEL)
	       (format stream "~%~10S" (pop code))
	       (setq labeled-line T))

	    (labeled-line
	       (setq labeled-line NIL)
	       (format stream "~18S" op)
	       (dotimes (i num-args)
		 (format stream "  ~S" (pop code))))

	    (T
	     (format stream "~%~10A~18S" " " op)
	     (dotimes (i num-args)
	       (format stream "  ~S" (pop code)))
	     ))
      )
    (format stream "~%")
    )
  (format stream "-------------------------")

;;;  compiled-code
  NIL
  )
      

(defun new-label ()

  ;;; Generates a new label for skipping.
  ;;; Skipping occurs in :UNP, :FAIL and :FORC
  (gentemp "L-" (find-package :keyword)))


(defun new-conj-id ()

  ;;; Generates a new id for a conjunctive branch closure.
  (gentemp "CJ-" (find-package :keyword)))


;;;; -------------  COMPILE-TIME TYPE CHECKING  ------------------

(defvar *lisp-type-domains* '(:NUMBER :STRING :SYMBOL :LIST))


(defun validate-argument-types (slot arguments)
  "Returns a new set of argument types."

  (if (member slot '(:A :THE :FORC))
      (validate-special-form-argument-types slot arguments)
    ;;else
    (unless (or (variable-p slot) (keywordp slot)
		(null CL-USER::*CHECK-SLOT-DOMAINS*)
		(eq slot *isa)        ;; Don't validate these slots or commands.
		(eq slot *member)
		(eq slot *name)
		(eq slot *disjoint)
		(eq slot :BOUNDP)
		(eq slot :UNBOUNDP)
		)
      (let ((types   (sfs:kb-get-slot-domains slot))
	    (clause  (cons slot arguments))
	    )
      
	;; Check for undefined slot
      
	(when (not (sfs:kb-frame-p slot))
	  (aam-compile-error :SLOT-UNDEFINED clause slot))
      
	;; Check for simple error - length mismatch
      
	(when (/= (length types) (length arguments))
	  (if (> (length types) (length arguments))
	      (aam-compile-error :SLOT-ARGS-TOO-FEW clause)
	    ;;ELSE
	    (aam-compile-error :SLOT-ARGS-TOO-MANY clause)))
      
	(#-GCL loop #+GCL sloop:sloop
	       for type in types
	       as  arg  in arguments
	       do
	       (type-eq arg type clause)
	       )
	)
      )
    )
  )


(defun validate-special-form-argument-types (slot arguments)
  "Validates :THE, :A and :FORC, which have special args."

  ;; (:THE (var TYPE) (forms)...)
  ;; need to associate 'var' with 'TYPE', if given.
  
  (when (consp (car arguments))
    (type-eq (first (car arguments)) (second (car arguments)) (cons slot arguments))
    )
  )


(defconstant *untyped-var* :XYZ-UNTYPED-XYZ)

(defun lookup-type (arg type-list)

  ;;; Since NIL is a valid type, we have to be careful.
  (let ((entry  (assoc arg type-list)))
    (if entry
	(cdr entry)
      ;;else
      *untyped-var*)
    )
  )


(defun constant-type-eq (arg type clause)
  "Returns T if the type of ARG matches TYPE.
Clause argument is for printing in the error message."


  ;;    (unless (or (eq slot *isa)
  ;;		(eq slot *name)
  (unless (not CL-USER::*check-slot-domains*)

    (cond (;; CASE 1: TYPE is NIL (which means any type is okay)
	   (null type) T)

	  ;; CASE 2: TYPE is a LISP type.
	  ((member type *lisp-type-domains*)
	   (if (typep arg (find-symbol (string type)))
	       T
	     ;;else
	     (aam-compile-error :DOMAIN-MISMATCH clause arg type)
	     ))

	  ;; CASE 3: TYPE is a frame
	  ((SFS:kb-frame-p type)
	   (cond

	    ;; Same frame is okay.
	    ((eq arg type)  T)

	    ;; A string may be converted to a frame
	    ;; Probably should check here, but how do
	    ;; we check when the string may be the
	    ;; public-name of many frames?
	    ;; Added 31 Oct 1997, mh
	    ((stringp arg)  T)

	    ;; Arg not a frame is not okay.
	    ((not (sfs:kb-frame-p arg))
	     (aam-compile-error :DOMAIN-MISMATCH clause arg type))

	    ;; Domain not a frame is not okay.
	    ((not (sfs:kb-frame-p type))
	     (aam-compile-error :ILLEGAL-DOMAIN clause type))

	    ;; We check to see whether the ISA relation is known:
	    ((member type (mapcar #'car (sfs:kb-get-values arg *isa *value)))
	     T)

	    ;; check to see whether the ISA relation is known to be false:
	    ((member type
		     (mapcar #'car (sfs:kb-get-values arg *isa *n-value)))

	     (aam-compile-error :UNACCEPTABLE-DOMAIN clause arg type))

	    ;; Is it provable that it ISA the type?
	    ((isa-p arg type)
	     T)

	    ;; Otherwise, an error
	    (T
	     (aam-compile-error :UNKNOWN-DOMAIN-MATCH clause arg type))
	    )
	   )
	  )				;cons
    )
  )


;;; don't we need to check bindings here too !!!???
;;; What bindings?  We are compiling!

(defun type-eq (arg type clause)

  ;; CLAUSE argument is for inclusion in error messages.

  ;; for constants, check the type.
  ;; for variables, if it has an entry of the same type, accept it
  ;;                otherwise, add an entry.

  (let (var-type)

    (or (null type)         ;; Any type will do

      (cond ((variable-p arg)
	     (setq var-type (lookup-type arg *ARGUMENT-TYPES*))

	     (cond ((eq var-type *untyped-var*)
		    (push (cons arg type) *ARGUMENT-TYPES*))

		   ((eq var-type type)     T)

		   ((isa-p var-type type)  T)

		   (T
		    (aam-compile-error :DOMAIN-MISMATCH clause arg type))
		   )
	     )
	    
	    ((constant-type-eq arg type clause)  T)
	    )
      )
    )
  )
			  

;;;; -------------  ERROR HANDLING  ------------------


(defun aam-compile-error (error-type clause &REST other-args)
  "Prints a message about the error and breaks."

  (ccase error-type
    (:PATH-RESTRICTION  
     (format *error-output*
             "~%;;*AAM COMPILE ERROR*  Path restriction error in ~S"
             clause)
     )
    
    (:SLOT-ARGS-TOO-FEW
     (format *error-output*
             "~%;;*AAM COMPILE ERROR*  Too few args supplied for slot ~S in ~S"
             (slot clause) clause))
        
    (:SLOT-ARGS-TOO-MANY
     (format *error-output*
             "~%;;*AAM COMPILE ERROR*  Too many args supplied for slot ~S in ~S"
             (slot clause) clause))
    
    (:DOMAIN-MISMATCH
     (format *error-output*
             "~%;;*AAM COMPILE ERROR*  Slot argument '~S' is not of type '~S' in ~S" 
             (first other-args) (second other-args) clause))
    
    (:ILLEGAL-DOMAIN
     (format *error-output*
             "~%;;*AAM COMPILE ERROR*  Unknown domain type '~S' for slot '~S'"
	     (first other-args) (slot clause)))

    (:UNACCEPTABLE-DOMAIN
     (format *error-output*
             "~%;; *AAM COMPILE ERROR*~%;; The frame ~S is not of type ~S, in ~S."
             (first other-args) (second other-args) clause))
    
    (:UNKNOWN-DOMAIN-MATCH
     (format *error-output*
             "~%;; *AAM COMPILE ERROR*~%;; Whether ~S is of type ~S is unknown in ~S." 
             (first other-args) (second other-args) clause))

    (:SLOT-UNDEFINED
     (format *error-output*
             "~%;; *AAM COMPILE ERROR*~%;; The slot '~S' is undefined in ~S"
             (first other-args) clause))
    )
  (break "To ignore error, continue...but faulty code may be generated.")
  T
  )



(defun aam-compile-warning (type clause rule-name)
  "Prints a warning message."

  (format *error-output* "~2%********************************")
  (format *error-output* "~%* Algernon Compilation Warning ")
  (format *error-output* "~%* ~S warning in clause ~S " type clause)
  (when rule-name (format *error-output* "of ~S" rule-name))
  (format *error-output* "~%* the rule being compiled may be incorrect.")
  (format *error-output* "~%********************************")
  )
