;;;; -*- Mode:Lisp; Package:AAM; Syntax:COMMON-LISP; Base:10 -*-

(in-package :AAM)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  Definitions for the AAM.  
;;;
;;;  *AAM-SPECIAL-FORMS*
;;;  *OPCODES*
;;;
;;;  environment structure
;;;  rule-closure structure
;;;
;;;  28 Oct 96 (mh) Added query-history  (aka the old depnet).
;;;  29 Oct 97 (nm) Replaced #'less-general by #'instance-p in
;;;                 query-previously-made-p.
;;;  30 Oct 97 (nm) Added instance-p to the *OPCODES* list for
;;;                 monitoring.
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

(defparameter  *backward-chain*  T
  "Bind to NIL to disable backward-chaining.")

(defparameter  *forward-chain*  T
  "Bind to NIL to disable forward-chaining.")



;;; This list has each opcode, its number of args, and
;;; the name of the processor function that handles it.

(defparameter *OPCODES*     
    '((:ACTIVATE-IA         0 op-activate-ia)
      (:ACTIVATE-IN         0 op-activate-in)
      (:ALL-PATHS           0 op-all-paths)
      (:ANY                 0 op-any)
      (:ASK                 0 op-ask)
      (:ASSERT              0 op-assert)
      (:ASSUME              0 op-assume)
      (:BOUNDP              0 op-boundp)
      (:BRANCH              0 op-branch)
      (:CLAUSE              1 op-clause)        ;; Compiler-generated
      (:CLEAR-SLOT          0 op-clear-slot)
      (:CONJ-BRANCH         1 op-conj-branch)   ;; Compiler-generated
      (:CONJ-CLEAR          1 op-conj-clear)    ;; Compiler-generated
      (:CONTINUATION        1 op-continuation)
      (:DELETE              0 op-delete)
      (:FAIL?               1 op-fail?)
      (:KB-DEF-FRAME        1 op-kb-def-frame)    
      (:KNOWN?              1 op-known?)
      (:LABEL               1 op-label)         ;;; essentially a no-op
      (:MERGE-BINDINGS      0 op-merge-bindings)
      (:POP                 1 op-pop)
      (:PUSH                2 op-push)
      (:QUERY               0 op-query)
      (:RETURN              1 op-return)
      (:SELECT              0 op-select)
      (:SKIP                1 op-skip)
      (:STOP                0 op-stop)
      (:SUBR                2 op-subr)
      (:SUBST               0 op-subst)
      (:SUCCEED?            1 op-succeed?)
      (:UNBOUNDP            0 op-unboundp)
      (:UNIQUE?             2 op-unique?)
      (:VERIFY              0 op-verify)

      (:CLAUSES             1 op-clauses)   
      (:INSTANTIATE         0 op-instantiate)  ;;; Lower-level forms
      (:KB-RETRIEVE         1 op-kb-retrieve)  ;;;  generated by the compiler.
      (:SAVE                0 op-save)

      (:BIND                0 op-bind)
      (:BRANCH-USER         0 op-branch-user)
      (:EVAL                0 op-eval)             ;;; Special forms
      (:NEQ                 0 op-neq)             
      (:RULES               0 op-rules)
      (:SHOW                0 op-show)
      (:SLOT                0 op-slot)
      (:SRULES              0 op-srules)
      (:TAXONOMY            0 op-taxonomy)
      (:TEST                0 op-test)

      (:CACHE-LOOKUP        0 NIL)                 ;;; Only for monitoring.
      (:CACHE-CREATE        0 NIL)
      (:INSTANCE-P          0 NIL)
      (:UNIFY               0 NIL)
      (:UNIFY-FAILED        0 NIL)
      (:UNIFY-INST          0 NIL)
      (:UNIFY-QUERY         0 NIL)
      (:UNIFY-MORE-GEN      0 NIL)
      (:INSTANTIATE-SUCC    0 NIL)     ;;; successful instantiations
      (:INSTANTIATE-CONT    0 NIL)     ;;; successful closure instantiations
      (:REASONING-FAIL      0 NIL)
      (:REASONING-SUCC      0 NIL)
      (:REASONING-OK        0 NIL)
      (:RETRIEVE-RULES      0 NIL)
      ))


(defun lookup-opcode-numargs (opcode)
  (second (assoc opcode *OPCODES*)))


(defun lookup-opcode-fn (opcode)
  (third (assoc opcode *OPCODES* :test #'eq)))


;;; ------------  PREDICATES  ------------


;;; These are user-level special forms recognized by the AAM compiler.
;;; If you change this list, change the function VARIABLE-ARGUMENTS
;;; in "utilities.lisp" also.

(defparameter *AAM-SPECIAL-FORMS* '(:A :ALL-PATHS :ALL-PATHS-PART-1 :ALL-PATHS-PART-2
				    :ANY :ASK :BIND :BOUNDP
				    :BRANCH :CLEAR-SLOT :DELETE :EVAL
				    :FAIL :FORC :NEQ :NO-COMPLETION :NO-CONTINUATION :OR
				    :UNP :RETRIEVE :RULES :SHOW
				    :SLOT :SRULES :STOP :TAXONOMY :TEST :THE
				    :UNBOUNDP))


;;; These are primarily used by the compiler.

(defun aam-special-form-p (form)
  (member form *AAM-SPECIAL-FORMS*))


(defparameter *NO-PATH-RESTRICTION-FORMS*
    '(:A :ANY :ASK :BIND :BOUNDP :FORC :THE CL-USER::NAME :RETRIEVE :SHOW :STOP :UNBOUNDP))

(defun ignore-path-restriction-for (operation)
  (member operation *NO-PATH-RESTRICTION-FORMS*))

(defun continuable-p (clause)
  "Returns T if a continuation is relevant for the clause.
Most special forms, like :A, are not continuable."

  ;; There are two things that make a clause not continuable.
  ;;
  ;; First, the semantics may not be meaningful.  For example,
  ;; If a clause of a rule fails, we want to reactivate the rule
  ;; later when we have evidence that the clause will succeed.
  ;; But a correctly written :SLOT or :FORC will always succeed.
  ;; Therefore, there is no need to make a continuation.
  ;;
  ;; The second reason is syntactic - it may not be clear 
  ;; where to store the continuation for a clause like :SLOT
  ;; or :BIND or especially :TAXONOMY.
  ;;
  ;; Basically a clause is not continuable if its main operator
  ;; is a special form.
  
  (not (member (slot clause) *AAM-SPECIAL-FORMS*))
  )


(defun ground-clause-p (form)
  (notany #'variable-p 
          (if (negated-p form) (negate form) form)))

(defun ground-path-p (form)
  (every #'ground-clause-p form))


(defun will-be-ground-p (clause bound-vars)
  "Returns T if the clause will be ground at runtime if
all of the 'bound-vars' are instantiated."
  
  ;; NULL means it will be a ground clause at runtime.
  (null (set-difference (variable-arguments clause)
			bound-vars))
  )



;;; ------------  STRUCTURES  ------------

(defstruct activation
  clause		     ;;; clause being processed
  index			     ;;; clause number being processed
  bindings		     ;;; binding list for the path
  activate-ia-clause         ;;; Clause used to activate rules
  activate-in-clause         ;;; Clause used to activate rules
  rule                       ;;; the rule being executed.
  code			     ;;; code to be processed when activated.
  key                        ;;; activation clause for if-needed rules.
  conj-id                    ;;; ID to allow conjunctive branches to be easily deleted.
  )

(defstruct rule-closure
  rule                       ;;; Name of the rule.
  bindings                   ;;; Bindings for the rule
  key                        ;;; Instantiated key clause of activated closure
  lookup-vars                ;;; Bound vars used to look up the appropriate compiled rule.
  id                         ;;; Unique id to prevent duplicates.
  )




(defun make-aam-activation (&KEY
			    (bl       NIL)
			    (rule     :USER-COMMAND)
			    (code     :REST)
			    (conj-id  NIL))

  (make-activation  :clause              (copy-list *R-CLAUSE*)
		    :index               *R-INDEX*
		    :bindings             bl
                    :activate-ia-clause  *R-ACTIVATE-IA*
                    :activate-in-clause  *R-ACTIVATE-IN*
                    :rule                (or *R-RULE* rule)
		    :key                 (sublis bl *R-KEY*)
		    :conj-id             conj-id
		    :code     (case code
				(:SELECT  (cdr (member :SELECT *R-CODE*)))
				(:REST    *R-CODE*))
		    )
  )
  

;;;need a closure-id to prevent duplicate closures from 
;;;being propagated.
(defun new-closure-id ()
  (gentemp "CLOSURE-" (find-package :keyword)))


(defun make-aam-closure (&KEY rule bindings key (lookup-vars NIL))

  (make-rule-closure  :rule         rule
		      :bindings     bindings
		      :key          key
		      :lookup-vars  lookup-vars
		      :id           (new-closure-id)
		      )
  )
  


;;; ------------------  QUERY HISTORY  ------------------
;;;
;;; Traversing search trees that have already been traversed
;;; can lead to exponential runtimes.  Rule continuations
;;; on backward-chaining rules eliminate the need to traverse
;;; the search tree again because if new facts are available
;;; they will be computed by the forward-chaining rule 
;;; continuations.
;;;
;;; Therefore we have the Query History, which in previous
;;; versions of Algernon was called the Depnet.  We:
;;;
;;;   - store every query that caused if-needed rules to
;;;     be activated, so that duplicate queries are not performed.
;;;
;;;   - Before if-needed rules are activated, check to see
;;;     whether the clause has already been queried.
;;;
;;; Technically, there are several assumptions that must be
;;; true in order for this to work:
;;;
;;;  1) Algernon stores (asserts) the consequents of all successful 
;;;     if-needed rules.
;;;
;;;  2) Algernon creates rule continuations on every antecedent clause
;;;     of an if-needed rule.
;;;
;;;  3) No if-needed rules were asserted between the time of the two 
;;;     duplicate queries.  
;;;
;;; The last one is problematical.  We'll skip over it for now...


;;; QUERY-PREVIOUSLY-MADE-P
;;;
;;; Returns T if the query or a more general one has already been
;;; performed.

(defun query-previously-made-p (clause)

  (let ((frame  (frame clause))
	(slot   (slot  clause))
	)

    (member clause (sfs::kb-get-values frame slot *queries)
	    :test #'instance-p)
    )
  )



;;; ADD-QUERY-TO-HISTORY
;;;
;;; Stores the query in the history.

(defun add-query-to-history (clause)

  ;; We don't have to check for membership because if it were
  ;; there already we wouldn't be storing it.

  (let ((frame  (frame clause))
	(slot   (slot  clause))
	)

    (sfs::kb-put-value frame slot *queries clause)
    )
  )


;;; CLEAR-QUERY-HISTORY
;;;
;;; clears the query history for a frame-slot.  This is done
;;; when the user deletes values from a slot.

(defun clear-query-history (frame slot)

  (sfs::kb-delete-values frame slot *queries)
  )
