;;;; -*- Mode:Lisp; Package:AAM; Syntax:COMMON-LISP; Base:10 -*-

(in-package :AAM)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  Performance monitoring for the AAM.
;;;
;;;  (monitor-reset)      -- resets all statistical counters and 
;;;                          turns monitoring on.
;;;  (monitor-on)         -- Turns monitoring on.
;;;  (monitor-off)        -- Turns monitoring off.
;;;  (monitor-stats)      -- Shows results.
;;;
;;;  (monitor <op>)       -- Increment the counter for this operation.
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; 30 Oct 1997 (nm)      copy *aam-stats* in monitor-stats so that
;;;                       sort doesn't destroy the original


;;; This is an a-list of (op #calls time) tuples

(defparameter *AAM-STATS*  NIL)

(defparameter *AAM-MONITORING*  NIL)     ;; Toggle for monitoring



(defun MONITOR (op time)
  (when *AAM-MONITORING*
    (let ((record (assoc op *AAM-STATS*)))
      (incf (second record))
      (incf (third record) time)
      )
    )
  )

#|
;;; Functions are faster...?
(defmacro MONITOR (op time)
  `(when *AAM-MONITORING*
     (let ((record (assoc ,op *AAM-STATS*)))
       (incf (second record))
       (incf (third record) ,time)
       )
     )
  )
|#

#+ALLEGRO
(defmacro with-aam-monitoring (op &REST body)

  `(if *AAM-MONITORING*
     (let ((start-time  (excl::get-internal-run-times))
	   )
       (multiple-value-prog1
	   (progn ,@body)
	 (monitor ,op (- (excl::get-internal-run-times) start-time))
	 )
       )
     ;;else
     (progn ,@body)
     )
  )

;;; This includes GC times, which I don't really want.
#-ALLEGRO
(defmacro with-aam-monitoring (op &REST body)

  `(if *AAM-MONITORING*
     (let ((start-time  (get-internal-real-time))
	   )
       (multiple-value-prog1
	   (progn ,@body)
	 (monitor ,op (- (get-internal-real-time) start-time))
	 )
       )
     ;;else
     (progn ,@body)
     )
  )


(defun monitor-reset ()
  
  (setq *AAM-STATS*
	(sort (mapcar #'(lambda (opcode-info)
			  (list (car opcode-info) 0 0))
		      *OPCODES*)
	      #'string-lessp
	      :key #'car
	      )
	)

  (monitor-on)
  )


(defun monitoring-p () *aam-monitoring*)



(defun monitor-on ()

  (setq *aam-monitoring* T)
  (when (>= aam::*trace-level* 2)
    (format T "~%AAM performance monitoring is now ON."))
  )


(defun monitor-off ()

  (setq *aam-monitoring* NIL)
  (when (>= aam::*trace-level* 2)
    (format T "~%AAM performance monitoring is now OFF."))
  )


(defun monitor-stats ()

  (format T "~%-------------------------------------")
  (format T "~%     AAM Performance Monitoring")
  (format T "~%-------------------------------------")
  (format T "~%  instruction           calls    time  ms/call")
  (format T "~%----------------------------------------------")

;;; This prints out all the info
  #|
  (dolist (monitor-info *AAM-STATS*)
    (format T "~%~20S ~8D  ~6d  ~7,3f"
    (first monitor-info) (second monitor-info) (third monitor-info))
    )
|#

;;; This prints out the info in sorted order, eliminating zeroes.
  (let ((total-calls  0))

    (dolist (monitor-info (sort (copy-list *AAM-STATS*) #'> :key #'second))
      (unless (zerop (second monitor-info))
	(format T "~%~20S ~8D  ~6d"
		(first monitor-info) (second monitor-info)
		(third monitor-info))
	;;Print the ms/call if it is meaningful (> 1 ms per call)
	(if (and (> (third monitor-info) (second monitor-info))
		 (not (eq (first monitor-info) :ASK)))
	    (format T "  ~7,3f"
		(float (/ (third monitor-info) (second monitor-info))))
	  ;;else
	    (format T "     -")
	    )
	(incf total-calls (second monitor-info))
	)
      )

    (format T "~%----------------------------------------------")
    (format T "~%Total Calls:         ~8D" total-calls)
    (format T "~2%")
    )
  )
