;;; -*- Mode:Common-Lisp; Package:AAM; Syntax:COMMON-LISP; Base:10 -*-

(in-package :AAM)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  The AAM processor.  Processes code for the AAM.
;;;  The code is produced by compiler.lisp.
;;;
;;;  Top-level:  (aam-process <code>)
;;;
;;; ------------------------------------------------------------
;;;  11 Jun 1996 (mh) updated for new AAM instructions
;;;   7 Aug 1996 (mh) Added :SLOT, :RULE and :SRULE
;;;  29 Aug 1997 (nm) Changes to accomodate new ALGY-UNIFY in :BIND,
;;;                   :BRANCH-USER, :QUERY, TRY-TO-INSTANTIATE.
;;;                   MORE-GENERAL and LESS-GENERAL are no longer
;;;                   used.  INSTANCE-P is used instead.  
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 


;;; Miscellaneous for nice printing
(defparameter *LAST-COMMAND* NIL)

(defparameter *TOP-LEVEL-EXIT-TAG*     NIL
  "Used for the user command :STOP to throw to the top level.")

;;; PROCESSOR REGISTERS

(defparameter *R-ACTIONS*      NIL    "Stack or queue of paths to be followed.")
(defparameter *R-ACTIVATE-IA*  NIL    "Activation clause for if-added rules.")
(defparameter *R-ACTIVATE-IN*  NIL    "Activation clause for if-needed rules.")
(defparameter *R-BINDINGS*     NIL    "Stack of binding lists.")
(defparameter *R-CLAUSE*       NIL    "The clause currently being executed.")
(defparameter *R-CODE*         NIL    "Code register (like Program Counter).")
(defparameter *R-CONTEXT*      NIL    "Stack of execution contexts.")
(defparameter *R-INDEX*        NIL    "Number of clause being processed.")
(defparameter *R-KEY*          NIL    "Key clause of if-needed rules.")
(defparameter *R-RESULTS*      NIL    "Binding lists stored for return to user.")
(defparameter *R-RULE*         NIL    "Rule being processed.")
      

;; old -- (defparameter *CLOSURES*       NIL    "Accumulate rule activations.")

(defparameter *equalp-test*
    #+GCL #'equal                   ;; Fix this when GCL gets the EQUALP function.  18 Oct 1996
    #-GCL #'equalp)



;; An alist of 'street names' and 'internal names' of registers.
(defparameter *AAM-REGISTERS*
    '((:ACTIONS     . *R-ACTIONS*)
      (:ACTIVATE-IA . *R-ACTIVATE-IA*)
      (:ACTIVATE-IN . *R-ACTIVATE-IN*)
      (:BINDINGS    . *R-BINDINGS*)
      (:CLAUSE      . *R-CLAUSE*)
      (:CODE        . *R-CODE*)
      (:CONTEXT     . *R-CONTEXT*)
      (:INDEX       . *R-INDEX*)
      (:KEY         . *R-KEY*)
      (:RESULTS     . *R-RESULTS*)
      (:RULE        . *R-RULE*)
      ))


#|
(defparameter *activate-rules-code-1*
  '(
      :clause  (cl-user::?frame cl-user::isa cl-user::value) ;; for every set
      :subst
      :kb-retrieve  cl-user::?set
      :fail?     :FAIL
      :merge-bindings
      :branch                   ;; Branch for every set
      :clause  (cl-user::?set cl-user::?slot cl-user::?facet);;  for every rule
      :subst
      :kb-retrieve  cl-user::?closure
      :fail?     :FAIL
      :merge-bindings
      :branch                   ;; Branch for every rule
      :clause  (cl-user::?frame cl-user::?slot)              ;;for every clause
      :subst
      :clauses      cl-user::?clause
      :fail?     :FAIL
      :merge-bindings
      :branch                   ;; Branch for every clause
      :clause  (cl-user::?closure cl-user::?clause)          ;; activate a rule
      :subst
      :instantiate
      :return       :NOTHING
      :label        :FAIL 
      :return       :FAILURE
    )
  )


;;; This is exactly the same as above, except the first query is deleted. 

(defparameter *activate-rules-code-2*
  '(
      :clause  (cl-user::?frame cl-user::?slot cl-user::?facet);;for every rule
      :subst
      :kb-retrieve  cl-user::?closure
      :fail?     :FAIL
      :merge-bindings
      :branch                   ;; Branch for every rule
      :clause  (cl-user::?frame cl-user::?slot)              ;;for every clause
      :subst
      :clauses      cl-user::?clause
      :fail?     :FAIL
      :merge-bindings
      :branch                   ;; Branch for every clause
      :clause  (cl-user::?closure cl-user::?clause)          ;; activate a rule
      :subst
      :instantiate
      :return    :NOTHING
      :label     :FAIL 
      :return    :FAILURE       ;; Okay as default behavior
    )
  )
|#

;;; ---------------  Main processor function  ------------------

(defun aam-process (code bindings &KEY (rule NIL))
  "This should be re-entrant."

  ;; The :RULE argument is a hack to allow us to make the trace 
  ;; say that rule activation is taking place.

  (let ((opcode         NIL)
	(op-fn          NIL)

	(*R-ACTIONS*      NIL)
	(*R-ACTIVATE-IA*  NIL)
	(*R-ACTIVATE-IN*  NIL)
        (*R-BINDINGS*     bindings)
	(*R-CLAUSE*       NIL)
	(*R-CODE*         code)
	(*R-CONTEXT*      NIL)
	(*R-INDEX*        1)
	(*R-KEY*          NIL)
	(*R-RESULTS*      NIL)
	(*R-RULE*         rule)

	(top-level        NIL)

	;; Handle recursive calls to 'aam-process'.
	(monitoring-previously-on  (and *TOP-LEVEL-EXIT-TAG*
					(monitoring-p)))
        (set-trace-level           *trace-level*)
	)


    (when (>= *TRACE-LEVEL* 3)
      (format *trace-output* "~%AAM initialized."))

    (when (or (>= *trace-level* 4) (< *trace-level* 0))
      (unless monitoring-previously-on    
	(monitor-reset)
	(sfs::monitor-reset)))

    ;; Set up the main loop.  Set a 'CATCH' so that
    ;; if the user wants to exit in the middle of the session
    ;; via :STOP, they can.
    
    (when (null *TOP-LEVEL-EXIT-TAG*)
      (setq top-level T)
      (setq *TOP-LEVEL-EXIT-TAG* (gensym)))
    
    (catch (if top-level *TOP-LEVEL-EXIT-TAG* :DUMMY)
      (loop
	(when (and (NULL *R-CODE*)
		   (NULL *R-ACTIONS*))
	  (return))

	;; RULE TRACING
	(cond ((rule-traced-p *R-RULE*)
	       (setq set-trace-level  *trace-level*)
	       (setq *trace-level*    2))
	      ;;else
	      ((setq *trace-level*    set-trace-level))
	      )
        

	(when (null *R-CODE*)
	  (setq *R-CODE* '(:SELECT)))

	(setq opcode (pop *R-CODE*))
	(setq op-fn  opcode)

	(when (>= *TRACE-LEVEL* 3)
	  (format *trace-output* "~%~S    " opcode)
	  (dotimes (i (lookup-opcode-numargs opcode))
	    (format *trace-output* "~S    " (elt *R-CODE* i))
	    )
	  (finish-output *trace-output*)
	  )

	(if (null op-fn)
	    (cerror "Skip this one."
		    "Unknown opcode: ~S" opcode)
	  ;;ELSE
	  (funcall op-fn))
	

	(when (>= *TRACE-LEVEL* 4)
	  (format *trace-output* "~%after opcode ~S:" opcode)
	  (show-registers))
	)
      )

    (when top-level
      (setq *TOP-LEVEL-EXIT-TAG* NIL))
    
    (when (or (>= *trace-level* 4) (< *trace-level* 0))
      (unless monitoring-previously-on
	(monitor-off)
	(sfs::monitor-off)
	(monitor-stats)
	(sfs::monitor-stats)
	)
      )

    ;;; Return the value in the RESULTS register.
    *R-RESULTS*
    )
  )



;;;; -------------  OPCODE processing  -----------------

(defun activate-closures (clause rule-type)
  "Activates if-added or if-needed rules."

  (let ((frame          (frame clause))
        (slot           (slot clause))
        (closures       NIL)
        )
    
    (when (>= *trace-level* 2)
      (format *trace-output* "~%Activating ~a rules from ~S." rule-type clause))
      
    (setq closures 
          (mapcan #'(lambda (closure)
                      (try-to-instantiate closure clause))
                  (retrieve-rules-for frame slot rule-type (negated-p clause))))

    (when (>= *trace-level* 3)
      (format T "~%Found ~D ~A closures."
              (length closures) rule-type))
    
    (dolist (closure closures)

	;; Filter out some extraneous results that were returned.
	(when (rule-closure-p closure)
	  (when (>= *trace-level* 2)
	    (format *trace-output* "~%Activating ~S - ~S" 
		    (rule-closure-rule closure)
		    (rule-closure-bindings closure)))
	  (add-to-action-queue 
	   (make-activation  :clause   NIL
			     :index    (cl-user::rule-index
					(rule-closure-rule closure))
			     :bindings (rule-closure-bindings closure)
			     :rule     (rule-closure-rule     closure)
			     :key      (rule-closure-key      closure)
                           
			     ;; The code can be an if-needed rule iff
			     ;; it is a continuation on the first clause.
			     :code     (if (if-needed-rule-p
					    (rule-closure-rule closure))
                                       
					   (find-if-needed-rule-code
					    (rule-closure-rule closure)
					    (rule-closure-lookup-vars closure))

					 ;;ELSE
					 (CL-USER::rule-code
					  (rule-closure-rule closure)))
			     ))))
      closures
    )
  )


(defun :activate-ia ()

  (with-aam-monitoring :ACTIVATE-IA
  
    (unless (null *R-ACTIVATE-IA*)
      (activate-closures *R-ACTIVATE-IA* :IF-ADDED)
      (setq *R-ACTIVATE-IA* NIL)
      )
    )
  )


(defun :activate-in ()

  (with-aam-monitoring :ACTIVATE-IN
  
    ;; (VARIABLE-P frame) is only true when the slot is NAME.
    ;; We don't activate if-needed rules for NAME.
    (unless (or (variable-p (frame *R-ACTIVATE-IN*))
                (SFS:slot-full-p (frame *R-ACTIVATE-IN*) (slot *R-ACTIVATE-IN*)
				 (facet *R-ACTIVATE-IN*))
		(query-previously-made-p *R-ACTIVATE-IN*)
		)
      
      (activate-closures *R-ACTIVATE-IN* :IF-NEEDED)
      (add-query-to-history *R-ACTIVATE-IN*)
      (setq *R-ACTIVATE-IN* NIL)
      )
    )
  )


#| old version
(defun :activate-ia ()
  "Activates if-added rules."

  (with-aam-monitoring :ACTIVATE-IA
  
    (unless (null *R-ACTIVATE-IA*)

      (when (>= *trace-level* 2)
	(format *trace-output*
		"~%Activating if-added rules from ~S." *R-ACTIVATE-IA*))
      
      ;; Results accumulate here.
      (setq *CLOSURES* NIL)

      (let* ((clause         *R-ACTIVATE-IA*)
             (set-rule-facet  (if (negated-p clause) *n-if-added
				                     *if-added))
	     (slot-rule-facet (if (negated-p clause) *slot-n-if-added
				                     *slot-if-added))
	     (self-rule-facet (if (negated-p clause) *self-n-if-added
				                     *self-if-added))
             (frame          (frame clause))
             (slot           (slot  clause))
	     (new-closures   NIL)
             )


        ;; AAM-PROCESS function is re-entrant

	;; (1) Rules from the sets the frame belongs to
	(aam-process *activate-rules-code-1*
		     `((((cl-user::?frame  . ,frame)
			 (cl-user::?slot   . ,slot)
			 (cl-user::?facet  . ,set-rule-facet)
			 (cl-user::?clause . ,clause)
			 )))
		     :RULE :IA-RULE-ACTIVATION)

	;; (2) Rules from the slot
	(aam-process *activate-rules-code-2* 
		     `((((cl-user::?frame  . ,slot)
			 (cl-user::?slot   . ,slot)
			 (cl-user::?facet  . ,slot-rule-facet)
			 (cl-user::?clause . ,clause)
			 )))
		     :RULE :IA-RULE-ACTIVATION)


	;; (3) Rules from the sets the slot belongs to
	(aam-process *activate-rules-code-1*
		     `((((cl-user::?frame  . ,slot)
			 (cl-user::?slot   . ,slot)
			 (cl-user::?facet  . ,slot-rule-facet)
			 (cl-user::?clause . ,clause)
			 )))
		     :RULE :IA-RULE-ACTIVATION)

	;; (4) Generic rules from the sets the frame belongs to
	(aam-process *activate-rules-code-1* 
		     `((((cl-user::?frame  . ,frame)
			 (cl-user::?slot   .  cl-user::generic)
			 (cl-user::?facet  . ,set-rule-facet)
			 (cl-user::?clause . ,clause)
			 )))
		     :RULE :IA-RULE-ACTIVATION)

	;; (5) Generic rules from the slot
	(aam-process *activate-rules-code-2* 
		     `((((cl-user::?frame  . ,slot)
			 (cl-user::?slot   .  cl-user::generic)
			 (cl-user::?facet  . ,slot-rule-facet)
			 (cl-user::?clause . ,clause)
			 )))
		     :RULE :IA-RULE-ACTIVATION)

	;; (6) Generic rules from the sets the slot belongs to
	(aam-process *activate-rules-code-1* 
		     `((((cl-user::?frame  . ,slot)
			 (cl-user::?slot   .  cl-user::generic)
			 (cl-user::?facet  . ,slot-rule-facet)
			 (cl-user::?clause . ,clause)
			 )))
		     :RULE :IA-RULE-ACTIVATION)

	;; (7) rule completions from self
	(aam-process *activate-rules-code-2* 
		     `((((cl-user::?frame  . ,frame)
			 (cl-user::?slot   . ,slot)
			 (cl-user::?facet  . ,self-rule-facet)
			 (cl-user::?clause . ,clause)
			 )))
		     :RULE :IA-RULE-ACTIVATION)
          
        (setq new-closures (retrieve-rules frame slot (if (negated-p clause)
							 *n-value *value)
                                           :IF-ADDED))
	(when (/= (length *closures*)
		  (length new-closures))
	  (format t "~%~D /= ~D"
		  (length *closures*)
		  (length new-closures))
          (setq new-closures (mapcan #'(lambda (closure)
                                         (:instantiate-new closure clause))
                                     new-closures))
	  (if (> (length *closures*)
		   (length new-closures))
	    (break "More rules?"))
          ;;else
          (format t "...OK")
	  )

	(when (>= *trace-level* 3)
	  (format T "~%Activate-IA found ~D closures." (length *CLOSURES*)))
	
        (dolist (closure *CLOSURES*)
	  ;; Filter out some extraneous results that were returned.
          (when (rule-closure-p closure)
	    (when (>= *trace-level* 2)
	      (format *trace-output* "~%Activating ~S - ~S" 
		      (rule-closure-rule closure)
		      (rule-closure-bindings closure)))
	    (add-to-action-queue 
             (make-activation  :clause   NIL
		               :index    (cl-user::rule-index
					  (rule-closure-rule closure))
		               :bindings (rule-closure-bindings closure)
                               :rule     (rule-closure-rule     closure)

			       ;; The code can be an if-needed rule iff
			       ;; it is a continuation on the first clause.
		               :code     (if (if-needed-rule-p
					      (rule-closure-rule closure))
					     
					     (find-if-needed-rule-code
					      (rule-closure-rule closure)
					      (mapcar #'car
						      (rule-closure-bindings
						       closure)))
					   ;;ELSE
					     (CL-USER::rule-code
					      (rule-closure-rule closure)))
                               )))
	  )
	(setq *R-ACTIVATE-IA* NIL)
	*CLOSURES*
        )
      )
    )
  )
|#


#|
;;; old version
(defun :activate-in ()
  "Activates if-needed rules."

  (with-aam-monitoring :ACTIVATE-IN
  
    ;; Don't activate rules unless there is an activation clause
    ;; and the slot is not already full.

    (unless (or (SFS:slot-full-p (frame *R-ACTIVATE-IN*) (slot *R-ACTIVATE-IN*)
				 (facet *R-ACTIVATE-IN*))
		(if-needed-rule-already-active-for *R-ACTIVATE-IN*))
      
      ;; Activated rules accumulate here.
      (setq *CLOSURES* NIL)
      
      (let* ((clause          *R-ACTIVATE-IN*)
	     (set-rule-facet  (if (negated-p clause) *n-if-needed
			          *if-needed))
	     (slot-rule-facet (if (negated-p clause) *slot-n-if-needed
			          *slot-if-needed))
	     (frame          (frame clause))
	     (slot           (slot  clause))
	     (new-closures   NIL)
	     )
        

        ;; This can only happen when the slot is NAME.
        ;; We don't activate if-needed rules for names.
        (unless (variable-p frame)
          
	  (when (>= *trace-level* 2)
	    (format *trace-output* "~%Activating if-needed rules from ~S."
		    clause))
          
          
	  ;; AAM-PROCESS function is re-entrant
          
	  ;; (1) Rules from the sets the frame belongs to
	  (when (sfs:kb-frame-p frame)
	    (aam-process *activate-rules-code-1*
			 `((((cl-user::?frame  . ,frame)
			     (cl-user::?slot   . ,slot)
			     (cl-user::?facet  . ,set-rule-facet)
			     (cl-user::?clause . ,clause)
			     )))
		     :RULE :IN-RULE-ACTIVATION))
          
	  ;; (2) Rules from the slot
	  (aam-process *activate-rules-code-2* 
		       `((((cl-user::?frame  . ,slot)
			   (cl-user::?slot   . ,slot)
			   (cl-user::?facet  . ,slot-rule-facet)
			   (cl-user::?clause . ,clause)
			   )))
		       :RULE :IN-RULE-ACTIVATION)

	  
          
          
	  ;; (3) Rules from the sets the slot belongs to
	  (aam-process *activate-rules-code-1*
		       `((((cl-user::?frame  . ,slot)
			   (cl-user::?slot   . ,slot)
			   (cl-user::?facet  . ,slot-rule-facet)
			   (cl-user::?clause . ,clause)
			   )))
		       :RULE :IN-RULE-ACTIVATION)

          
	  ;; (4) Generic rules from the sets the frame belongs to
	  (when (sfs:kb-frame-p frame)
	    (aam-process *activate-rules-code-1* 
			 `((((cl-user::?frame  . ,frame)
			     (cl-user::?slot   .  cl-user::generic)
			     (cl-user::?facet  . ,set-rule-facet)
			     (cl-user::?clause . ,clause)
			     )))
			 :RULE :IN-RULE-ACTIVATION))

          
	  ;; (5) Generic rules from the slot
	  (aam-process *activate-rules-code-2* 
		       `((((cl-user::?frame  . ,slot)
			   (cl-user::?slot   .  cl-user::generic)
			   (cl-user::?facet  . ,slot-rule-facet)
			   (cl-user::?clause . ,clause)
			   )))
		       :RULE :IN-RULE-ACTIVATION)

	
	
	  ;; (6) Generic rules from the sets the slot belongs to
	  (aam-process *activate-rules-code-1* 
		       `((((cl-user::?frame  . ,slot)
			   (cl-user::?slot   .  cl-user::generic)
			   (cl-user::?facet  . ,slot-rule-facet)
			   (cl-user::?clause . ,clause)
			   )))
		       :RULE :IN-RULE-ACTIVATION)

          
        (setq new-closures (retrieve-rules frame slot (if (negated-p clause)
							 *n-value *value)
                                           :IF-NEEDED))
	(when (/= (length *closures*)
		  (length new-closures))
	  (format t "~%~D /= ~D"
		  (length *closures*)
		  (length new-closures))
          (setq new-closures (mapcan #'(lambda (closure)
                                         (:instantiate-new closure clause))
                                     new-closures))
	  (if (> (length *closures*)
		   (length new-closures))
	    (break "IN: More rules?"))
          ;;else
          (format t "...OK")
	  )

	  (when (>= *trace-level* 3)
	    (format T "~%Activate-IN found ~D closures." (length *CLOSURES*)))
	
	  (dolist (closure *CLOSURES*)
	    ;; Filter out some extraneous results that were returned.
	    (when (rule-closure-p closure)
	      (when (>= *trace-level* 2)
	        (format *trace-output* "~%Activating ~S - ~S" 
		        (rule-closure-rule closure)
		        (rule-closure-bindings closure)))
	      (add-to-action-queue 
	       (make-activation  :clause   NIL
			         :index    (cl-user::rule-index
					    (rule-closure-rule closure))
			         :bindings (rule-closure-bindings closure)
				 :activate-ia-clause  NIL
				 :activate-in-clause  NIL
			         :rule     (rule-closure-rule     closure)
			         :code     (find-if-needed-rule-code
					     (rule-closure-rule closure)
					     (mapcar #'car
						     (rule-closure-bindings
						      closure)))
			         )))
	    )
	  *CLOSURES*
	  )
        )
      )
    )
  )
|#


;;; If there is more than one binding list in the topmost binding-set
;;; of the *BINDINGS* register, pick one at random and throw the rest
;;; away.

(defun :any ()

  ;; Do it truly randomly
  ;; We know that the binding-set is non-empty because 
  ;; the preceding opcode was a check for failure.

  (let ((rand   (random (length (car *R-BINDINGS*)))))

    (push (list (nth rand (pop *R-BINDINGS*))) *R-BINDINGS*)
    )
  )


;; Copied directly from Algernon v2.0's (algy-ask) function.
;; Modified to not assert from within the function.
;; The AAM compiles code to do the assert afterward.

(defun :ask ()

  ;; If the user answers 'NO', fail.
  ;; If the user answers 'YES', succeed with T.
  ;; If the user enters a value, succeed with a binding set.

  (with-aam-monitoring :ASK
    (let* ((clause (second *R-CLAUSE*))
	   (frame  (frame clause))
	   (slot   (slot  clause))
	   (facet  (facet clause))
	   )

      (cond ((not (ground-clause-p clause))    ;; NON-ground clause

	     (let* ((var      (find-if #'variable-p (arguments clause)))
		    (domains  (sfs:kb-get-slot-domains slot))
		    (set      (when domains
				(elt domains (1- (position var clause)))))
		    (possible-values
		     (unless (constantp set) ; nil or keyword
		       (mapcar #'car	;We know 'member' is a
					;2-arity relation
			       (sfs:kb-get-values set 'CL-USER::member *value))))
		    (value nil)
		    )

	       (format *query-io*
		       "~% Give me a value for ~(~s~) in ~(~s~): " var clause)
	       (when possible-values
		 (format *query-io* "~% [possible values:")
		 (format *query-io* ": "))

	       ;; Loop 'till get an acceptable value:
	       (loop
		 ;; Print out list of possible values for var (if known).
		 (when possible-values
		   (mapc #'(lambda (x)
			     (format *query-io* " ~(~s~)" x))
			 possible-values)
		   (format *query-io* "] "))
	       
		 (setq value (read *query-io* nil nil))
		 (cond (possible-values
			(if (find value possible-values)
			    (return)
			  (format *query-io* "~% [Enter a value in:")))

		       ((sfs:kb-frame-p value) (return))

		       ((keywordp set)

			(if (typep value (find-symbol (string set)))
			    (return)
			  (format *query-io*
				  "~% [Enter a value of type ~s: " set)))

		       (t
			(format *query-io*
				"~% ~(~s~) is not a known frame." value)
			(cond ((y-or-n-p "Do you want to create it ?")
			       (sfs:kb-def-frame value)
			       (return))
			      (t
			       (format *query-io*
				       "~% Give me a value for ~(~s~) in ~(~s~)" var clause)
			       (if possible-values
				   (format *query-io* " [possible values:")
				 (format *query-io* ": ")))))))


	       (push `(((,var . ,value))) *R-BINDINGS*)

	       ))

	    ;; A ground clause
	    (T
	     (let ((currently-known
		    (member (arguments clause)
			    (sfs:kb-get-values frame slot facet)
			    :test *equalp-test*))
		   (negative-currently-known
		    (member (arguments clause)
			    (sfs:kb-get-values frame slot
					       (facet (negate clause)))
			    :test *equalp-test*))
		   )

	       (cond
		(currently-known           (push T   *R-BINDINGS*))
		(negative-currently-known  (push NIL *R-BINDINGS*))

		((yes-or-no-p " Is it true that ~(~s~)? " clause)
		 (push T *R-BINDINGS*))

		(t			; User has responded "no".
		 (push NIL *R-BINDINGS*))
		)
	       )
	     )
	    )
      )
    )
  )


(defun :assert ()
  "Error checking should be performed by the machine before
this opcode is executed."

;;;Time estimate:
;;; 

  (with-aam-monitoring :ASSERT
  
    (let* ((clause   *R-CLAUSE*)
	   (slot     (slot  clause))
	   (frame    (frame clause))
	   (facet    (facet clause))
	   (result   NIL)
	   )

      (when (>= *TRACE-LEVEL* 3)  (format *trace-output* "~S" clause))

      ;; The FRAME may not be a frame if the argument is a LISP type
      ;; example is (car ?list ?car ?rest).

      (when (or (sfs:kb-frame-p frame)
		(variable-p     frame))    ;; variable when have NAME clause.
      
	;; Check slot domains.
	;; Removed - done at compile time now.
;;	(slot-domains-match-clause clause)

	;; Check for non-ground clause
	(unless (ground-clause-p (arguments clause))
	  (aam-error :NON-GROUND-ASSERTION clause))

	(setq result (sfs:kb-put-value frame slot facet (arguments clause)))

	(push result *R-BINDINGS*)
	  
	(when (eq result T)            ;; Store this for If-Added activation.
	  (setq *R-ACTIVATE-IA* clause)
	  (when (eq slot *IMP-SUPERSET)
	    (propagate-rules-from-class-to-subclass (frame clause) (car (arguments clause)))
	    ))

	(when (>= *TRACE-LEVEL* 3)
	  (ccase result
	    ((T)          (format *trace-output* "~%  Assert succeeded."))
	    (:KNOWN       (format *trace-output* "~%  Assertion was known."))
	    ((NIL)        (format *trace-output* "~%  Assert failed."))
	    )
	  )
	)
      )
    )
  )


(defun :assume ()
  "Error checking should be performed by the machine before
this opcode is executed."

  (with-aam-monitoring :ASSUME
  
    (format *trace-output*
	    "~%WARNING: ASSUME called, but Not Yet Implemented.")
    )
  )


;; :BIND Modified (mh) 27 Mar 1997
;; - Changed behavior when given just one variable.
;;   Now it always unifies to the given value.

(defun :bind ()
  "The :BIND operator in Algernon."

  ;;; The topmost clause has the form  (:BIND vars expr)

  (with-aam-monitoring :BIND
    (let* ((clause       *R-CLAUSE*)
	   (form         (third clause))
	   (result       NIL)
	   )

      (when (consp form)
	(cond ((eq (car form)  :VALUES)
	       (setq form `(sfs::kb-get-values ',(second form) ',(third form) *value)))

	      ((eq (car form)  :NON-VALUES)
	       (setq form `(sfs::kb-get-values ',(second form) ',(third form) *n-value)))
	      )
	)

      (setq result (eval form))


      (let* ((vars         (second clause))
	     (binding-list (algy-unify vars result)) )
      
	(next-clause)
	(if (eq binding-list :FAILED)
	    (format *error-output*
		    "~%Warning: :BIND form: ~S did not bind anything." clause)
	  ;;ELSE
	  (push (list binding-list) *R-BINDINGS*))
	)
      )
    )
  )
    

(defun :boundp ()
  "Checks the variable in the clause and sets result to T if it is bound,
where it is checked by the :FAIL? operator.  Don't need :BRANCH :SELECT 
after this one."
  
  (with-aam-monitoring :BOUNDP
  
    (let ((var       (frame *R-CLAUSE*))
	  )

      (if (variable-p var)
	  (push NIL *R-BINDINGS*)
	;;else
	  (push T   *R-BINDINGS*))
      )
    )
  )


(defun :branch-user ()
  "The :BRANCH operator in Algernon."

  ;;; The topmost clause has the form  (:BRANCH vars expr)

  (with-aam-monitoring :BRANCH-USER

    (let* ((clause       *R-CLAUSE*)
	   (results      (eval (third clause)))
	   (vars         (second clause))
	   (binding-set  (mapcan #'(lambda (result)
				     (let ((bindings (algy-unify vars result)))
				       (unless (eq bindings :FAILED)
					 (list bindings)) ))
				 results))
	   )
      (next-clause)

      (if (null binding-set)
	  (format *error-output* "Warning: :BRANCH form: ~S did not bind anything."
		  clause)
	;;ELSE
	(push binding-set *R-BINDINGS*)
	)
      )
    )
  )


(defun :branch (&OPTIONAL (save-registers NIL))
  "If there are multiple binding lists in the binding set in the
binding register, processing needs to branch."
  
  ;; The optional argument allows this to be called by another function.

  ;; Creates a set of activations, one for each binding in the
  ;; current binding set, and adds them to the Actions register.
  
  ;; Note that if the binding set contains only one binding list,
  ;; and if the reasoning is performing depth-first-search,
  ;; the sequence of operations (:BRANCH :SELECT) will be a no-op.

  (with-aam-monitoring :BRANCH
  
    (setq *R-ACTIVATE-IN* *R-CLAUSE*)     ;; This is for :activate-in.

    (if (null (car *R-BINDINGS*))
	(add-to-action-queue (make-aam-activation :CODE (if save-registers
							    :SELECT
							  ;;else
							    :REST)))
      ;;ELSE
      (dolist (binding-list (car *R-BINDINGS*))
	(add-to-action-queue (make-aam-activation :BL binding-list
						  :CODE (if save-registers
							    :SELECT
							  ;;else
							    :REST)))
	)
      )

    ;; This line of reasoning will continue if and when selected
    ;; from the queue, so we get rid of everything unless the
    ;; caller wants to save the registers, which is what happens
    ;; in an :save() call.

    (unless save-registers
      (setq *R-CLAUSE*      NIL
	    *R-BINDINGS*    NIL
	    *R-CODE*        NIL
	    *R-RULE*        NIL
	    *R-INDEX*       NIL
	    *R-ACTIVATE-IA* NIL
	    *R-ACTIVATE-IN* NIL
	    *R-CONTEXT*     NIL
	    ))
    )
  )


(defun :clause ()
  
  (with-aam-monitoring :CLAUSE
    (setq *R-CLAUSE* (pop *R-CODE*))))


(defun :clear-slot ()
  "Clears a slot of a frame."

  ;; There is no FAIL? after this one, so we must call
  ;; (next-clause) ourselves.

  (with-aam-monitoring :CLEAR-SLOT
  
    (let* ((clause *R-CLAUSE*)
	   (frame  (frame clause))
	   (slot   (car (arguments clause)))
	   )

      (next-clause)
      (sfs:kb-delete-values frame slot)
      
      ;; also clear the query history for this slot.
      (clear-query-history frame slot)
      )
    )
  )


;; Conj-branch  added 8 Apr 97 (mh)
;; the :BRANCH operator in the AAM is an implicit OR branch.  The
;; path succeeds if any branch succeeds.  The :ALL-PATHS command
;; in Algernon needs a conjunctive branch - it succeeds iff *all*
;; branches succeed.  This command does a conjunctive branch.

(defun :conj-branch ()
  "If there are multiple binding lists in the binding set in the
binding register, processing needs to branch."
  
  ;; Creates a set of activations, one for each binding in the
  ;; current binding set, and adds them to the Actions register.
  ;; Each activation is a member of a conjunctive branch.
  ;; The code here is identical to :BRANCH, except for adding the
  ;; conjunctive id.
  
  ;; Note that if the binding set contains only one binding list,
  ;; and if the reasoning is performing depth-first search,
  ;; the sequence of operations (:CONJ-BRANCH :SELECT) will be a no-op.

  (with-aam-monitoring :CONJ-BRANCH
  
    (let ((conj-id  (pop *R-CODE*)))
      (setq *R-ACTIVATE-IN* *R-CLAUSE*)     ;; This is for :activate-in.
    
      (if (null (car *R-BINDINGS*))
	  (add-to-action-queue (make-aam-activation :CODE    :REST
						    :CONJ-ID conj-id))
	;;ELSE
	(dolist (binding-list (car *R-BINDINGS*))
	  (add-to-action-queue (make-aam-activation :BL      binding-list
						    :CODE    :REST
						    :CONJ-ID conj-id))
	  )
	)

      ;; This line of reasoning will continue if and when selected
      ;; from the queue, so we get rid of everything unless the
      ;; caller wants to save the registers, which is what happens
      ;; in an :save() call.

      (setq *R-CLAUSE*      NIL
	    *R-BINDINGS*    NIL
	    *R-CODE*        NIL
	    *R-RULE*        NIL
	    *R-INDEX*       NIL
	    *R-ACTIVATE-IA* NIL
	    *R-ACTIVATE-IN* NIL
	    *R-CONTEXT*     NIL
	    )
      )
    )
  )


(defun :conj-clear ()
  "Clears the action queue of any branches of the same conjunctive set
when any branch fails."

  ;; :ALL-PATHS is compiled as a SUBR now, so we can just kill the whole
  ;; set of pending actions.
  ;;  (setq *R-ACTIONS* NIL)
  
  ;; Set the RESULTS register to NIL.  Since this clause failed,
  ;; the path in progress will fail too, so we can clear it.
  ;;  (setq *R-RESULTS* NIL)

  ;; The CLEAR comes after the SUBR ends, so we need to selectively delete
  ;; elements from the queue.
  
  (let ((conj-id (pop *R-CODE*)))
    (cond ((eq conj-id :ALL)    (setq *R-ACTIONS* NIL)
				(setq *R-RESULTS* NIL))
	  (T                    (setq *R-ACTIONS*
				  (delete conj-id *R-ACTIONS* :key #'activation-conj-id)))
	  )
    )
  )


(defun :continuation ()
  "Creates a rule continuation after a query or assert."
  
  ;; Make a continuation for every binding list in the binding set.
  ;; Store it in a slot of the frame mentioned in the topmost
  ;; clause, which should already have its variables substituted
  ;; by values.

  ;; Frame and slot of the clause are guaranteed to be bound,
  ;; otherwise the compiler would have generated a path-restriction error.


  (with-aam-monitoring :CONTINUATION
  
    (let (clause frame slot facet
	  (rule (pop *R-CODE*))
	  )
      (setq clause *R-CLAUSE*)

      (setq frame     (frame clause))
      (setq slot      (slot clause))
      (setq facet     (if (negated-p clause) *self-n-if-added
			*self-if-added))

      ;; KB-PUT-VALUE checks for duplicate values using #'equalp,
      ;; so we don't need to check here.

      (dolist (binding-list (car *R-BINDINGS*))
	
	(sfs:kb-put-value frame slot facet
			  (list
			   (make-aam-closure
			    :rule      rule
			    :bindings  binding-list))))
      )
    )
  )


(defun :delete ()
  "Clears a frame-slot-facet of a frame."

  ;; There is no FAIL? after this one, so we must call
  ;; (next-clause) ourselves.

  (with-aam-monitoring :DELETE
    
    (let* ((clause  *R-CLAUSE*)
	   (frame   (frame clause))
	   (slot    (slot  clause))
	   (facet   (facet clause))
	   (value   (arguments clause))
	   )
      (next-clause)
      (sfs:kb-delete-value frame slot facet value)
      
      ;; also clear the query history for this slot.
      (clear-query-history frame slot)
      )
    )
  )


(defun :eval ()
  "Calls LISP to eval the form.  No return value is used
or placed on the stack."
  
  (with-aam-monitoring :EVAL
  
    (eval (second *R-CLAUSE*))
    (next-clause)
    )
  )


(defun :fail? ()
  "Checks the top of the BINDINGS stack for a failed
ASSERT, VERIFY or QUERY  (= value NIL).
If so, it skips to the given :LABEL statement."

  ;;; This is called after an ASSERT, QUERY or VERIFY
  ;;; when compiled in *fail-is-good* mode.  If the
  ;;; clause failed (which is good) it skips to the
  ;;; given label.

  ;;; If the operation succeeded, simply pops the topmost clause
  ;;; from the PATH register.
  


  (with-aam-monitoring :FAIL?
  
    (let ((label  (pop *R-CODE*)))
      
      (cond ((null (car *R-BINDINGS*)) ;; Operation failed.  Skip ahead.

	     (when (>= *trace-level* 3)
               (format *trace-output* "~%   * failed"))

	     (pop *R-BINDINGS*)        ;; Pop the NIL result
	     
	     (:skip label)	       ;; Jump to the label

	     (pop *R-CODE*)            ;; Get rid of :LABEL
	     (pop *R-CODE*)            ;; Get rid of the label

	     )

	    ;; Else - clause succeeded
	    ;; get rid of return value if it's not a set of bindings.
	    ((not (consp (car *R-BINDINGS*)))
	     (next-clause)   
	     (pop *R-BINDINGS*)
	     )

	    (T
	     (next-clause))
	    )
      )
    )
  )


(defun :known? ()
  "Checks the top of the BINDINGS stack for a :KNOWN
ASSERT.  If so, it skips to the given :LABEL statement."

  (with-aam-monitoring :KNOWN?
  
    (let ((label  (pop *R-CODE*)))
      
      (when (eq (car *R-BINDINGS*) :KNOWN) ;; value was known.

	(when (>= *trace-level* 3)
	  (format *trace-output*
		  "~%   * assertion of ~S was already known."
		  *R-CLAUSE*))
	  
	(pop *R-BINDINGS*)
	     
	(:skip label)	       ;; Jump to the label
	
	(next-clause)
	(pop *R-CODE*)            ;; Get rid of :LABEL
	(pop *R-CODE*)            ;; Get rid of the label
	
	)
      )
    )
  )


(defun :label ()

  ;;; A no-op.  Pop the next argument from the CODE register.
  
  (with-aam-monitoring :LABEL
    (pop *R-CODE*)
    )
  )


(defun :merge-bindings ()

  ;;; The BINDINGS register contains two binding sets.
  ;;; Merge them into one binding set.

  (with-aam-monitoring :MERGE-BINDINGS
  
    (push (merge-binding-sets (pop *R-BINDINGS*) (pop *R-BINDINGS*))
	  *R-BINDINGS*)
    )
  )


(defun :neq ()
  "Calls LISP to check equality of the two arguments.
Return value (T or NIL) is placed on the stack."
  
  (with-aam-monitoring :NEQ
  
    (let ((clause *R-CLAUSE*))
      (push (not (eq (second clause) (third clause)))
	    *R-BINDINGS*)
      )
    )
  )


(defun :pop ()
  "Removes the topmost item of the given register."
  
  (with-aam-monitoring :POP
  
    (let ((register (cdr (assoc (pop *R-CODE*) *AAM-REGISTERS*))))
      
      (eval `(pop ,register))
      )
    )
  )

  
(defun :push ()
  "Inserts an item onto the top of the given register."
  
  (with-aam-monitoring :PUSH
  
    (let ((register (cdr (assoc (pop *R-CODE*) *AAM-REGISTERS*)))
	  (value    (pop *R-CODE*)))

      (eval `(push ',value ,register))
      )
    )
  )


(defun :query ()
  "There is a non-ground clause in the CLAUSE register."

  (with-aam-monitoring :QUERY
  
    (let* ((clause      *R-CLAUSE*)
	   (slot        (slot  clause))
	   (frame       (frame clause))
	   (args        (arguments clause))
	   (facet       (facet clause))
	   (values      NIL)
	   (results     NIL)
	   )


      (when (>= *TRACE-LEVEL* 3)  (format *trace-output* "~S" clause))
	

      ;; The FRAME may not be a frame if the argument is a LISP type
      ;; example is (car ?list ?car ?rest).
      ;; But allow it if it is a variable, because of the (name ?x "name") form.

      (when (or (sfs:kb-frame-p frame)
		(variable-p     frame))

	;; NAME slot gets special handling
	;; Can have two forms:
	;;   (name ?x "name")  or
	;;   (name something ?name)

	(when (and (eq slot 'CL-USER::NAME)
		   (variable-p (frame clause)))
;;	  (setq frame (first (arguments clause)))
	  (setq frame (string (first (arguments clause))))
	  )

	;; Get the values from the KB

	;; Check slot domains.
	;; Removed - done at compile time now.
;;	(slot-domains-match-clause clause)

	(setq values (sfs:kb-get-values frame slot facet args))

	;; Place a binding set on the BINDINGS stack.

	(cond ((and (eq slot 'CL-USER::NAME)
		    (variable-p (frame clause)))
	       (push (make-binding-set (list (frame clause)) values)
		     *R-BINDINGS*))

	      (T
	       ;; It looks like the call to algy-unify cannot fail,
	       ;; since kb-get-values picks up instances of the pattern ARGS.
	       ;; Change this when we are sure.
	       (setq results 
		 (mapcan #'(lambda (value)
			     (let ((result (algy-unify args value)))
			       (unless (eq result :FAILED)
				 (list result)) ))
			 values))
	       (push results *R-BINDINGS*)
	       )
	      )

	(when (>= *TRACE-LEVEL* 3)
	  (if (car *R-BINDINGS*)
	      (format *trace-output* "~%  Query succeeded.")
	    ;;ELSE
	    (format *trace-output* "~%  Query failed")
	    )
	  )
	)
      )
    )
  )


(defun :return ()

  ;; A line of reasoning has finished.
  ;; If successful, push the binding set onto the RESULTS register.
  ;; Otherwise, halt the line of reasoning.

  (with-aam-monitoring :RETURN
  
    (let ((return-type  (pop *R-CODE*)))

      (case return-type

	(:SUCCESS    ;; Return the topmost Binding set (should only be one).
	         (monitor :REASONING-SUCC 0)
		 (if (car *R-BINDINGS*)
		     (dolist (binding-list (car *R-BINDINGS*))
		       (push (cons T (list binding-list)) *R-RESULTS*)
		       )
		   ;;else
		     (push (cons T NIL) *R-RESULTS*)))

	(:FAILURE
	         (monitor :REASONING-FAIL 0)
	         (pop  *R-BINDINGS*)   ;; That set of bindings failed; dump it.
		 )

	(:NOTHING  ;; Don't do anything, just return.
 	         (monitor :REASONING-OK 0)
		 ) 

	(T
	   (break "INTERNAL ERROR: Unknown option to :RETURN!"))
	)

      (setq *R-CODE* NIL)   ;; Don't need to continue this branch.
      )
    )
  )


(defun :rules ()
  "Defines rule(s) using AAM:DEF-RULE."

  (with-aam-monitoring :RULES
  
    ;; Form of 'clause' is (:RULES class rule1 rule2 ...)
    (let* ((clause *R-CLAUSE*)
           (class  (second clause))
           (rules  (cddr   clause)))

      (if (symbolp class)
	  (dolist (rule rules)
	    (AAM:DEF-RULE  'CL-USER::CLASS class rule))
	;;else
	(progn
	  (format *error-output* "~2%*** ERROR: :RULES clause has no class name.")
	  (format *error-output* "~%***    correct format: (:RULES class rule1 rule2 ...)")
	  (format *error-output* "~%***    the input: ~A" (list :RULES (second clause)))
	  )
	)
      )

    (next-clause)    ;; Since we don't do a :FAIL?, we need to do this.
    )
  )


(defun :save ()
  "Like :BRANCH, but doesn't clear the CODE register so that
processing can continue.  Used before an :ACTIVATE-IN instruction."

  ;; Note: (17 Sep 1997, mh)
  ;; ------------------
  ;; in depth-first mode, :SAVE should be performed before
  ;; :ACTIVATE-IN, but in breadth-first mode :SAVE should
  ;; be performed after.  I am working on resolving the problem.

  (:branch :SAVE-REGISTERS)
  )


(defun :select ()
  "Selects an operation from the ACTIONS register."

  ;;; Can generalize this to do rankings, etc.
  ;;; But built-in functions will handle Depth-first and
  ;;; Breadth-first by always making the topmost action the next one.

  (with-aam-monitoring :SELECT
  
    (let ((next-action  (pop *R-ACTIONS*)))

      ;; *last-command* may be obviated by *R-RULE*   (may be...)
      ;; (mh) 11 Sep 1996

      ;; Print the rule that is executing, but make it
      ;; print nicely.

      (when (>= *trace-level* 2)
	(let ((rule  (activation-rule next-action)))
	  (if (eq rule *last-command*)
	      (princ "." *trace-output*)
	    ;;else - print the rule.
	    (format *trace-output* "~%Executing ~S" rule))
	  (setq *last-command* rule)
	  )
	)


      (when next-action
        (setq *R-CLAUSE*       (activation-clause               next-action))
        (setq *R-BINDINGS*     (list (list (activation-bindings next-action))))
        (setq *R-INDEX*        (activation-index                next-action))
        (setq *R-ACTIVATE-IA*  (activation-activate-ia-clause   next-action))
        (setq *R-ACTIVATE-IN*  (activation-activate-in-clause   next-action))
        (setq *R-RULE*         (activation-rule                 next-action))
        (setq *R-CODE*         (activation-code                 next-action))
        (setq *R-KEY*          (activation-key                  next-action))
        )
      )
    )
  )


(defun :show ()
  "Displays the form on the screen.  No return value is placed on the stack."
  
  (with-aam-monitoring :SHOW
  
    (sfs::kb-print-no-rules (second *R-CLAUSE*))
    (next-clause)
    )
  )


(defun :skip (&OPTIONAL (to-label NIL))
  "Pops the CODE register until reaching the label."

  ;; Optional argument allows this to be called from within the program.

  ;; Maybe we should call this function "op-skip-and-jump" !
  
  (let ((label (or to-label (pop *R-CODE*)))
	)

;;  (when (eq label :FAIL)
;;    (format t "~%Failed on ~S" *R-CLAUSE*))

    (when (>= *trace-level* 3)
      (format *trace-output* "~%Skipping to label ~S" label))

    (#-GCL loop #+GCL sloop:sloop
	until (or (and (eq (first  *R-CODE*) :LABEL)
		       (eq (second *R-CODE*) label))
		  (null *R-CODE*))
	do    (pop *R-CODE*))
    
    (when (null *R-CODE*)
      (break (format NIL "INTERNAL AAM ERROR: Couldn't find the label '~S'."
		     label)))
    )
  )


(defun :slot ()
  "Defines a slot using GFP:CREATE-SLOT."

  ;;; We support slot definitions of the form:
  ;;; (:SLOT name domains :CARDINALITY n)

  (with-aam-monitoring :SLOT
  
    ;; Form of 'clause' is (:SLOT name domains [:cardinality n])
    (let ((clause *R-CLAUSE*))
      (GFP:create-slot (second clause) (third clause))
      (CL-USER::tell `((,*ISA ,(second clause) CL-USER::SLOTS)))

      (when (SFS::kb-slot-p 'CL-USER::ARITY)
	(CL-USER::tell `((CL-USER::arity ,(second clause) ,(length (third clause))))))

      ;; Optionally assert cardinality
      (when (eq (fourth clause) :CARDINALITY)
        (CL-USER::tell
	 `((CL-USER::cardinality ,(second clause) ,(fifth clause))))
        )
      )

    (next-clause)    ;; Since we don't do a :FAIL?, we need to do this.
    )
  )


(defun :srules ()
  "Defines rule(s) using AAM:DEF-RULE."

  (with-aam-monitoring :SRULES
  
    ;; Form of 'clause' is (:RULES rule1 rule2 ...)
    (let* ((clause *R-CLAUSE*)
           (slot   (second clause))
           (rules  (cddr   clause)))

      (dolist (rule rules)
        (AAM:DEF-RULE  'CL-USER::SLOT slot rule))
      )

    (next-clause)    ;; Since we don't do a :FAIL?, we need to do this.
    )
  )


(defun :stop ()

  (when (>= *trace-level* 3)
    (format *trace-output* "~%:STOP called - halting execution."))

  (throw *TOP-LEVEL-EXIT-TAG* :DONE)
  )


(defun :subr ()

  ;; A subroutine makes a recursive call to the AAM processor
  ;; and places the bindings returned in the BINDINGS register.

  (with-AAM-monitoring  :SUBR
    (let ((var          (pop *R-CODE*))
	  (code         (pop *R-CODE*))
	  (binding-sets NIL)
	  (binding-set  NIL)
	  (results      NIL)
	  (result       NIL)
	  )

      (when (null code)   ;; Compiler should optimize this case out
	  (return-from :SUBR NIL))

      
      (setq result
	(aam-process code (list (car *R-BINDINGS*))
		     :RULE *R-RULE*))

	(setq binding-sets (mapcar #'cdr result))
	(setq results      (mapcar #'car result))

	(setq binding-set (car binding-sets))
	(do ((bset     (cdr binding-sets) (cdr bset)))
	    ((null bset))
;;	  (setq binding-set (merge-binding-sets binding-set (car bset)))
	  (setq binding-set (append binding-set (car bset)))
	  )

;;  (format t "~%SUBR: process returned ~S and ~S and ~S" results binding-sets binding-set)

      ;; We only want to save bindings for the 'var' of the form.
      ;; filter them before returning a result.

      ;; If there was a variable given,
      (cond ((every #'null results)  (setq results nil))

	    ((null var)       (setq results (or binding-set (car results))))  ;; to handle :OR

	    (T  (setq results
		  (or (delete NIL
			      (mapcar #'(lambda (binding-list)
					  (delete-if-not
					   #'(lambda (binding)
					       (eq var (car binding)))
					   binding-list))
				      binding-set))
		      T)))
	    )

;;      (format t "~%SUBR: returning ~S" results)
      
      ;; If there was a variable given, as in the :THE form,
      ;; we only return its bindings, and it is a new variable,
      ;; so we push its bindings on the list.
      ;;
      ;; But if we are returning all variables, they are a superset
      ;; of what is already on the bindings list, so we just replace
      ;; the bindings list.
      ;;
      ;; And if results are NIL, we push a NIL result on the bindings list.

      (cond ((null results)   (push NIL *R-BINDINGS*))
	    ((null var)       (setq *R-BINDINGS* (list results)))
	    (T                (push results *R-BINDINGS*))
	    )
      )
    )
  )


(defun :subst ()

  (with-aam-monitoring :SUBST
  
    (let ((clause   *R-CLAUSE*)
	  (bindings (caar *R-BINDINGS*))
	  )
      
      (setq *R-CLAUSE* (my-nsubstitute clause bindings))

      (when (>= *trace-level* 3)
	(format *trace-output* " -> ~S" *R-CLAUSE*))
      )
    )
  )


(defun :succeed? ()
  "Checks the top of the BINDINGS stack for a successful
ASSERT, VERIFY or QUERY  (value <> NIL).
If so, it skips to the given :LABEL statement."

  ;;; This is called after an ASSERT, QUERY or VERIFY
  ;;; when compiled in *fail-is-good* mode.  If the
  ;;; clause succeeded (which is good) it skips to the
  ;;; given label.

  ;;; If the operation failed, simply pops the following :LABEL
  ;;; instruction from the CODE register.  This instruction
  ;;; sometimes precedes a :LABEL instruction.
  

  (with-aam-monitoring :SUCCEED?
  
    (let ((label  (pop *R-CODE*)))
      
;;  (format t "~%:SUCCEED?:  ~S" (car *R-BINDINGS*))

      (cond ((not (null (car *R-BINDINGS*))) ;; Operation succeeded.  Skip

	     (when (>= *trace-level* 3)
               (format *trace-output*
       "~%   * operation on clause ~D succeeded (but wasn't supposed to): ~S"
		     *R-INDEX* *R-CLAUSE*))

	     (:skip label)

	     (when (eq (car *R-CODE*) :LABEL)
	       (pop *R-CODE*)            ;; Pop the label
	       (pop *R-CODE*)            ;; Pop the label
	       )
	     (next-clause)             ;; Do this *after* error message.

	     (when (not (consp (car *R-BINDINGS*)))   ;; Get rid of the 'T', if any.
	       (pop *R-BINDINGS*))
	     )

	    ;; Else - clause failed, get rid of return value
	    (T
	     (pop *R-BINDINGS*)
	     (when (eq (car *R-CODE*) :LABEL)
	       (pop *R-CODE*)
	       (pop *R-CODE*))           ;; Get rid of :LABEL
	     (next-clause)   
	     )
	    )
      )
    )
  )


(defun :unboundp ()
  "Checks the variable in the clause and sets result to NIL if it is bound,
where it is checked by the :FAIL? operator.  Don't need :BRANCH :SELECT 
after this one."
  
  (with-aam-monitoring :UNBOUNDP
  
    (let ((var       (frame *R-CLAUSE*))
	  )

      (if (variable-p var)
	  (push T *R-BINDINGS*)
	;;else
	  (push NIL   *R-BINDINGS*))
      )
    )
  )


(defun :unique? ()
  "Like :succeed? except that it only aborts the computation
if there are multiple bindings for the given variable."


  ;;; This is called after querying a path in a :THE command.
  ;;; We know that the query succeeded.  We want to fail
  ;;; if the path succeeded with more than one binding.
  
  ;;; The next item in the CODE register is a variable to check.
  ;;; After that is a label to jump to if it succeeded.

  ;;; To fail, simply pop the topmost clause from the PATH register.
  
  ;;; If the operation succeeded *with multiple bindings for the
  ;;; variable* , prints an error message, pops the failure and 
  ;;; pops the topmost clause.

  ;;; If the operation succeeded with one binding, jump to
  ;;; the success label.

  (with-aam-monitoring :UNIQUE?
  
    (let ((clause         *R-CLAUSE*)
	  (var            (pop *R-CODE*))
	  (success-label  (pop *R-CODE*))
	  )

      (cond ((and (not (null (car *R-BINDINGS*))) ;; Operation failed if multi
		  (multiple-bindings-p var (car *R-BINDINGS*)))
	     
	     (pop *R-BINDINGS*)
	     (when (>= *trace-level* 3)
               (format *trace-output*
		       "~%   * instantiation of clause ~D wasn't supposed to succeed with multiple bindings: ~S"
		     *R-INDEX* clause))

	     (pop  *R-BINDINGS*)       ;; That set of bindings failed; dump it.
	     (setq *R-CODE* NIL)       ;; Don't need to follow this branch.
	     )

	    ;;; The path must have succeeded with just one binding.
	    ;;; So we are done (successful).
	    ;;; Go to the success label (bypass intervening code)

	    (T

	     (:skip success-label)

	     (pop *R-CODE*)   ;; Get rid of the label
	     (pop *R-CODE*)   ;; Get rid of the label

	     ;; get rid of the return value if it's not a set of bindings.

	     (when (not (consp (car *R-BINDINGS*)))
	       (pop *R-BINDINGS*))
	     )
	    )
      )
    )
  )


;;;; ------- TAXONOMY --------------

(defun :taxonomy (&OPTIONAL (taxonomy NIL))
  "Create sets and instances."

  ;;; The optional argument allows it to be called from
  ;;; within the program, as well as when processing
  ;;; the :TAXONOMY opcode.

  ;;; If (deftaxonomy ...) is used, we must compile the form
  ;;; at runtime.  This could be bad because we might not have
  ;;; the compiler available at runtime.

  ;; :TAXONOMY never has variables, so
  ;; there will never be any bindings to pass into aam-process.

  (with-aam-monitoring :TAXONOMY

    (if taxonomy
      ;; Must do a runtime compilation
      
      (aam-process
       (aam-compile
        (make-taxonomy (or taxonomy *R-CLAUSE*))
        :ASSERT
        NIL)
       NIL
       :RULE    *R-RULE*
       )

      ;;else - compiled already
      (progn
	(aam-process (pop *R-CODE*) NIL :RULE *R-RULE*)
	(next-clause)
	)
      )
    )
  )



;;; Modified MAKE-TAXA to produce rules too.  The rules ensure that 
;;; every set is disjoint from its sibling sets.
;;;
;;; ((disjoint ?set1 ?set2)
;;;  ->
;;;  (:rules ?set1
;;;    ((isa ?x set1) -> (not (isa ?x set2)))))


(defun make-taxonomy (tree)
  
  (nconc
   (make-taxa tree)
   (taxa-disjoint tree)
   )
  )
	      

(defun make-taxa (tree)
  (mapcan #'(lambda (x)
	      (if (atom x)
		  (taxa-member x (car tree))
		(taxa-superset x (car tree))))
	  (cdr tree))
  )


(defun taxa-member (new-member set)
  (if (eq new-member :complete)
      `((CL-USER::complete ,set CL-USER::true))
      (let ((var (new-var)))
	`((:the ,var (CL-USER::name ,var ,(string new-member)))
	  (,*member ,set ,var)
	  (,*isa    ,var ,set)))))

(defun taxa-superset (new-set superset)
  (let ((new-set-name (car new-set))
	(new-set-description (cdr new-set))
	(var (new-var)))
    (append `((:the ,var (,*name ,var ,(string new-set-name)))
	      (,*isa ,var CL-USER::Sets)
	      (,*imp-superset ,var ,superset))
	    (make-taxa (cons var new-set-description)))))


(defun taxa-disjoint (tree)
  "Hey! No taxa dis joint!"
  
  (if (atom tree)
      NIL
    (append
     (disjoint-relations
      (member-and-siblings
       (mapcan #'(lambda (elt)
		   (and (consp elt)
			(list (car elt))))
	       (cdr tree))))

     (mapcan #'(lambda (elt)
		 (and (consp elt)
		      (taxa-disjoint elt)))
	     (cdr tree))
     )
    )
  )


(defun member-and-siblings (set)
  
  (unless (atom set)
    (mapcar #'(lambda (member)
		(cons member (remove member set)))
	    set)
    )
  )


(defun disjoint-relations (sibling-sets)

  (mapcan #'(lambda (member-and-siblings)
	      (unless (null (cdr member-and-siblings))
		(cons
		 (disjoint-rule member-and-siblings)
		 (mapcar #'(lambda (sibling)
			     `(CL-USER::disjoint ,(car member-and-siblings) ,sibling))
			 (cdr member-and-siblings)))))
	  sibling-sets)
  )

(defun disjoint-rule (member-and-siblings)

  `(:rules ,(car member-and-siblings)
	   ((,*isa CL-user::?x ,(car member-and-siblings))
	    ,*forward
	    ,@(mapcar #'(lambda (sibling)
			  `(CL-user::not (,*isa CL-USER::?x ,sibling)))
		      (cdr member-and-siblings))))
  )
  


;;;; ------- end of TAXONOMY --------------


(defun :test ()
  "Calls LISP to eval the form.  The return value is tested
by the :FAIL? operator.  Don't need :BRANCH :SELECT after this one."
  
  (with-aam-monitoring :TEST
  
    (let ((result   (eval (second *R-CLAUSE*))))
      (push (not (null result)) *R-BINDINGS*)
      )
    )
  )


(defun :verify ()
  "Error checking should be performed by the machine before
this opcode is executed."

  (with-aam-monitoring :VERIFY
  
    (let* ((clause   *R-CLAUSE*)
	   (slot     (slot  clause))
	   (frame    (frame clause))
	   (facet    (facet clause))
	   (values   NIL)
	   )

      ;; Check slot domains.
      ;; Removed - done at compile time now.
;;      (slot-domains-match-clause clause)

      (setq values (sfs:kb-get-values frame slot facet))

      (cond ((member (arguments clause) values :test *equalp-test*)
	     (when (>= *TRACE-LEVEL* 3) (format *trace-output* "~%  Verify confirmed."))
	     (push T *R-BINDINGS*))
	    (T
	     (when (>= *TRACE-LEVEL* 3) (format *trace-output* "~%  Verify not confirmed."))
	     (push NIL *R-BINDINGS*)
	     )
	    )
      )
    )
  )

;;; --------------  LOWER-LEVEL OPCODES  --------------------
;;;
;;; These are not generated by the compiler.  These are used
;;; in the hand-written routines to do rule activation.

(defun op-clauses ()
  (format t "Hey! someone called op-clauses!~%")
  (break)
  )



(defun :kb-def-frame ()

  ;;; The next item in the CODE register is a variable name to 
  ;;; which the result should be bound. 
  ;;;
  ;;; The variable and the frame's name are pushed onto the bindings register.

  (with-aam-monitoring :KB-DEF-FRAME
    (let* ((var         (pop *R-CODE*))
	   (new-frame   (sfs:kb-def-frame
			 (or (find-name var (copy-list
					     (cddr *R-CLAUSE*))
					(car *R-BINDINGS*))
			     'CL-USER::FRAME)))
	   )
      (next-clause)  ;; 

      (push (list (list (cons var new-frame))) *R-BINDINGS*)
      )
    )
  )


(defun op-kb-retrieve ()

  ;;; The CLAUSE register is a form containing (frame slot facet)
  ;;; 
  ;;; The next item in the CODE register is a variable name to 
  ;;; which the results should be bound.
  ;;;
  ;;; The value(s) are retrieved from the KB and a set of binding lists
  ;;; is pushed onto the BINDINGS register.

  (with-aam-monitoring :KB-RETRIEVE
    (let* ((retrieve-info  *R-CLAUSE*)
           (frame          (first   retrieve-info))
           (slot           (second  retrieve-info))
           (facet          (third   retrieve-info))
           (results        (sfs:kb-get-values frame slot facet))
           (binding-set    (make-binding-set (list (pop *R-CODE*))
                                              results))
           )
      
      (push binding-set *R-BINDINGS*)
      )
    )
  )



(defun try-to-instantiate (closure clause)
  "Attempts to instantiate the closure from the clause."

  ;;; This operation instantiates the key of the closure's
  ;;; rule against the clause.  Instantiation does not need to be
  ;;; complete (i.e. all variables bound) - just consistent.
  ;;; 
  ;;;nm-Obsolete comment.
  ;;; If the rule is instantiated, it is 
  ;;; modified 11 Nov 96 (mh) to also return a Parameter Indirection Table.
  
  (with-aam-monitoring :INSTANTIATE

    (let* ((rule-bindings     (rule-closure-bindings closure))
           (new-bindings      NIL)
	   (rule-clause       NIL)
	   (rule-lookup-vars  NIL)
	   )
      
      ;; Apply the bindings in the closure.
      (setq rule-clause
            (sublis rule-bindings (CL-USER::rule-key (rule-closure-rule closure))))
      
      ;; Attempt to unify with the clause.
      (monitor :UNIFY-INST 0)
      (setq new-bindings (algy-unify rule-clause clause))

      ;; Save these bindings for rule lookup
      (unless (eq new-bindings :FAILED)
        (setq rule-lookup-vars (mapcar #'car new-bindings)))

      (if (not (eq new-bindings :FAILED))
        (progn
	  (monitor :INSTANTIATE-SUCC 0)
	  (when (cl-user::rule-root (rule-closure-rule closure))
	    (monitor :INSTANTIATE-CONT 0))
          (list (make-aam-closure :rule        (rule-closure-rule closure)
				  :bindings    (append new-bindings rule-bindings)
				  :lookup-vars rule-lookup-vars
				  :key         rule-clause))
          )
        ;;else
        NIL
        )
      )
    )
  )

;;; --------------  BINDING LIST MANAGEMENT  ----------------


#|
Input:  BLSet  = (((?x . 3) (?y . 4) (?z . 5))
                  ((?x . 3) (?y . 4) (?z . 6)))
        vars   = (?a ?b)
	values = ((7 8) (14 16))

	Result = (((?x .  3) (?y .  4) (?z . 5) (?a .  7) (?b .  8))
		  ((?x .  3) (?y .  4) (?z . 5) (?a . 14) (?b . 16))
		  ((?x .  3) (?y .  4) (?z . 6) (?a .  7) (?b .  8))
		  ((?x .  3) (?y .  4) (?z . 6) (?a . 14) (?b . 16)))

 = merge-binding-list-into-set(make-binding-list(vars, values), BLSet);

|#

(defun make-binding-set (vars list-of-values)
  "VARS is a list of variables.
LIST-OF-VALUES is a list of sets of values.  Each set has one
binding for each variable."

  ;; The NREVERSE is just to make it look pretty.
  ;; The 'pairlis' call reverses the order of the vars.
  
  (mapcar #'(lambda (values) (nreverse (pairlis vars values)))
	  list-of-values)
  )


(defun merge-binding-sets (new-blset BLSet)
  "Returns a set of binding lists.  Each input is a set
of binding lists."

  ;;; It is guaranteed that the two sets will have
  ;;; non-overlapping sets of variables.  So we need
  ;;; to do a sort of cross-product.

  (cond ((null new-blset)  BLSet)
	((null BLSet)      new-blset)
	(T
	   (mapcan #'(lambda (bl1)
		       (mapcar #'(lambda (bl2)
				   (append bl1 (copy-list bl2)))
			       new-blset)
		       )
		   BLSet))
	)

  )
  

;;;; ---- Counting number of bindings  -------


(defun number-of-bindings (var binding-set)
  "Returns T if there are multiple bindings for the var in the set."

  ;;; The binding set contains multiple binding lists.
  ;;; Query will place a binding in each binding list
  ;;; if there are multiples.  So we have to search each
  ;;; binding list for the variable, and count the number
  ;;; of bindings retrieved.
  
  (count-if #'(lambda (binding-list)
		(assoc var binding-list))
	    binding-set)
  )



(defun single-binding-p (var binding-set)
  "Returns T if there are multiple bindings for the var in the set."

  (= (number-of-bindings var binding-set) 1)
  )


(defun multiple-bindings-p (var binding-set)
  "Returns T if there are multiple bindings for the var in the set."

  (> (number-of-bindings var binding-set) 1)
  )


;;; ---------------  NAME GENERATION  -------------------

;;; This is straight from Algernon
;;; Find-name
;;;  Tries to find a mnemonic "address" for the frame for var.
;;;  Path is path about to be asserted about var
;;;  (under substitutions in result).
;;;  The address of a frame is the symbol on whose property list
;;;  the frame lives.
;;;
(defun find-name (var path binding-set)

  (dolist (binding-list binding-set)
    (setq path (my-nsubstitute path binding-list)))
  
  (or

   ;; Try to base address on the name of the new frame:
   (funcall #'(lambda (clause)
		(if clause (intern (string-upcase (first (arguments clause)))
				   :CL-USER)))
	    (car (member-if #'(lambda (clause)
			   (and (eql (slot clause) 'CL-USER::name)
				(eql var (frame clause))
				(not (variable-p
				      (first (arguments clause)))))
			   )
		       path)))

    ;; Try to base address on binary relation to other frames:
   (funcall #'(lambda (clause)
		(if clause
		    (intern (format nil "~a.~a"
				    (frame clause) (slot clause))
			    :CL-USER)))
	    (car (member-if #'(lambda (clause)
			   (and (= 1 (length (arguments clause)))
				(eql var (first (arguments clause)))
				(not (variable-p (frame clause)))
				(not (variable-p (slot clause)))
				(not (eql (slot clause) *ISA))))
		       path)))

    ;; Try to base address on non-binary relation to other frames:
   (funcall #'(lambda (clause)
		(if clause 
		    (intern (format nil "~a.~a"
				    (frame clause) (slot clause))
			    :CL-USER)))
	    (member-if #'(lambda (clause)
			   (and (member var (arguments clause))
				(not (variable-p (frame clause)))
				(not (variable-p (slot clause)))))
		       path))

   ))



;;;---------------  Slot type check/assert ------------

;;; Slot domain matching was moved to the compiler
;;; 13 Nov 1996 (mh)

;;; New slot domain matcher
;;; From mtg with Ben, Spencer, Mike  on  7 Oct 96

;;; 1) Definition of taxonomy extended so that sibling classes are disjoint.
;;;    This is enforced by rules.
;;;
;;; 2) slot-domains-match-clause turns into an error checker.
;;;    (isa <x> <y>)
;;;         =>  T      if "<x> isa <y>" is known.
;;;         =>  error  if "not: <x> isa <y>" is known.
;;;         =>  error  if "<x> isa <y>" is unknown.

(defun slot-domains-match-clause (clause)
  "Checks the clause against the definition of the slot."
  
  (break "Somebody called SLOT-DOMAINS-MATCH-CLAUSE")

  (let ((slot  (slot clause)))

    (unless (or (eq slot *isa)
		(eq slot *name)
		(not CL-USER::*check-slot-domains*))

      (let ((domains   (SFS::kb-get-slot-domains (slot clause)))
	    (args      (cons (frame clause) (arguments clause)))
	    )

	;; (1) Check for simple error - length mismatch

	(when (and domains (/= (length domains) (length args)))
	  (if (> (length domains) (length args))
	      (aam-error :SLOT-ARGS-TOO-FEW clause)
	    ;;ELSE
	      (aam-error :SLOT-ARGS-TOO-MANY clause)))

	;; (2) Check each argument.

	(do ((d-type domains   (cdr d-type))
	     (arg    args      (cdr arg))
	     )
	    ((null d-type) T)

	    (cond

	   ;; CASE 1: TYPE is NIL (which means any type is okay) or
	   ;;         the argument is a variable.  Don't check it.
	   ((or (null (car d-type)) (variable-p (car arg))))

	   ;; CASE 2: TYPE is a LISP type.
	   ((member (car d-type) *lisp-type-domains*)
	    (or (typep (car arg) (find-symbol (string (car d-type))))
		(aam-error :DOMAIN-MISMATCH clause (car arg) (car d-type))))

	   ;; CASE 3: TYPE is a frame
	   ((SFS::kb-frame-p (car d-type))
	    (cond

	     ;; Same frame is okay.
	     ((eq (car arg) (car d-type)))

	     ;; Arg not a frame is not okay.
	     ((not (sfs::kb-frame-p (car arg)))
	      (aam-error :DOMAIN-MISMATCH clause (car arg) (car d-type)))

	     ;; Domain not a frame is not okay.
	     ((not (sfs::kb-frame-p (car d-type)))
	      (aam-error :ILLEGAL-DOMAIN clause (car d-type)))

	     ;; We check to see whether the ISA relation is known:
	     ((member (car d-type) (mapcar #'car (sfs::kb-get-values (car arg) *isa *value)))
	      T)

	     ;; check to see whether the ISA relation is known to be false:
	     ((member (car d-type)
		      (mapcar #'car (sfs::kb-get-values (car arg) *isa *n-value)))

	      (aam-error :UNACCEPTABLE-DOMAIN clause (car arg) (car d-type)))

	     ;; Is it provable that it ISA the type?
	     ((isa-p (car arg) (car d-type))
	      T)

	     ;; Otherwise, an error
	     (T
	      (aam-error :UNKNOWN-DOMAIN-MATCH clause (car arg) (car d-type))))
	    )
	   )
	    )				;do
	)
      )
    )
  T
  )



#|  ;; this comment is now out of date.
Spencer says that two queries are incomparable if 
at least one condition from set A and one from set B is true:

 A) a.1:  the pattern has some variables bound to constants.
    a.2:  the pattern has some (non-singleton) equivalence sets.

 B) b.1:  the expression has some variables bound to constants.
    b.2:  the expression has some (non-singleton) equivalence sets.

otherwise, if A.2 and B.2 are false, then the pattern
is more general than the expression.
|#


(defun has-bindings-to-constants-p (binding-list)

  (some #'(lambda (binding) (not (variable-p (cdr binding))))
	binding-list)
  )

(defun has-non-singleton-class-p (classes)

  ;; True if some class has more than one element.
  (some #'cdr classes))

#|  ;; no longer used
(defun more-general (pred1 pred2)
  "The clause with (more) variables goes in the first argument and the
ground clause goes in the second argument.  Returns T if the first
argument is more general than the second argument."

  ;; 12 Nov 1996
  ;; Two predicates can be orderable, or they may be incomparable.
  ;; If they each unify to each other, then they are incomparable.

  (monitor :UNIFY-MORE-GEN 0)

  (let (p-bindings p-classes e-bindings e-classes)

    (multiple-value-setq
	(p-bindings p-classes e-bindings e-classes)
        (algy-unify pred1 pred2))

    ;; Check for failure
    (if (eq (car p-bindings) :FAILED)
	NIL

      ;;else check for various conditions:

      (let ((test-A1 (has-bindings-to-constants-p (cdr p-bindings)))
	    (test-A2 (has-non-singleton-class-p   p-classes))
	    (test-B1 (has-bindings-to-constants-p e-bindings))
	    (test-B2 (has-non-singleton-class-p   e-classes))
	    value
	    )

	(setq value
	      (cond ((and (or test-A1 test-A2) (or test-B1 test-B2))
		     :INCOMPARABLE)
		    ((and (not test-B1) (not test-B2))  :MORE-GENERAL-OR-EQ)
		    (T                                  :LESS-GENERAL)))

	(eq value :MORE-GENERAL-OR-EQ)
	)
      )
    )
  )


(defun less-general (pred1 pred2)
  "The clause with (more) variables goes in the first argument and the
ground clause goes in the second argument."

  (more-general pred2 pred1)
  )
|#


(defun filter-activations (activations)
  "Removes activations not created by rules."

  (remove-if #'(lambda (activation)
		 (or (member (activation-rule activation) '(:USER-COMMAND NIL))
		     (not (if-needed-rule-p (activation-rule activation))))
		)
	     activations))



;;; -----------------  Miscellaneous Routines  ----------------


(defun find-all-rules ()
  "Prints the location of all rules in the KB."

  (let ((num-rules   0))
    (dolist (frame (SFS:KB-get-all-frames))
      (dolist (slot-info (get frame 'SFS::slot-info))
	(dolist (facet-info (cdr slot-info))
	  (cond ((member (car facet-info) 
			 '(cl-user::n-if-added        cl-user::if-added
			   cl-user::n-if-needed       cl-user::if-needed
			   cl-user::slot-n-if-added   cl-user::slot-if-added
			   cl-user::slot-n-if-needed  cl-user::slot-if-needed
			   cl-user::self-if-added     cl-user::self-n-if-added
			   ))
 
		 (incf num-rules (length (cdr facet-info)))
		 
		 (format *trace-output* "~%~S/~S/~S     ~S"
			 frame (car slot-info) (car facet-info)
			 (mapcar #'(lambda (value)
				     (rule-closure-rule (car value)))
				 (cdr facet-info))))))))

    (format *trace-output* "~%~D instantiated rules or continuations found."
	    num-rules)
    )
  )




(defun next-clause ()
  "increments the clause number."

  ;; This used to do more when we processed paths, rather than clauses.

  (incf *R-INDEX*)
  )


(defun new-var ()

  ;;; Generates a new variable name.
  (gentemp "?$" (find-package :CL-USER)))


(defun show-registers ()

  (format *trace-output* "~%~%REGISTERS:")
  (format *trace-output* "~%~12@A: ~S" 'CLAUSE    *R-CLAUSE*)
  (format *trace-output* "~%~12@A: ~S" 'BINDINGS  *R-BINDINGS*)
  (format *trace-output* "~%~12@A: ~S" 'ACTIONS   *R-ACTIONS*)
  (format *trace-output* "~%~12@A: ~S" 'RESULTS   *R-RESULTS*)
  (format *trace-output* "~%~12@A: ~S" 'CODE      *R-CODE*)
  (format *trace-output* "~%~12@A: ~D" 'INDEX     *R-INDEX*)
  (format *trace-output* "~%~12@A: ~D" 'RULE      *R-RULE*)
  (format *trace-output* "~%~12@A: ~S" 'IA-CLAUSE *R-ACTIVATE-IA*)
  (format *trace-output* "~%~12@A: ~S" 'IN-CLAUSE *R-ACTIVATE-IN*)
  (format *trace-output* "~%~12@A: ~S" 'CONTEXT   *R-CONTEXT*)
  (format *trace-output* "~%------------------------------~%")
  )


(defun aam-error (error-type clause &REST other-args)
  "Prints a message about the error and breaks."

  (ccase error-type
    (:PATH-RESTRICTION  
     (format *error-output*
             "~%;;*AAM COMPILE ERROR*  Access limitation error in ~S"
             clause)
     )
    
    (:NON-GROUND-ASSERTION
     (format *error-output*
	     "~%;;*AAM ERROR* Attempting to assert the non-ground clause ~S"
	     clause))

    (:SLOT-ARGS-TOO-FEW
     (format *error-output*
             "~%;;*AAM COMPILE ERROR*  Too few args supplied for slot ~S in ~S"
             (slot clause) clause))
        
    (:SLOT-ARGS-TOO-MANY
     (format *error-output*
             "~%;;*AAM COMPILE ERROR*  Too many args supplied for slot ~S in ~S"
             (slot clause) clause))
    
    (:DOMAIN-MISMATCH
     (format *error-output*
             "~%;;*AAM ERROR*  Slot argument ~S is not of type  ~S in ~S" 
             (first other-args) (second other-args) clause))
    
    (:ILLEGAL-DOMAIN
     (format *error-output*
             "~%;;*AAM ERROR*  Unknown domain type ~S for slot ~S"
	     (first other-args) (slot clause)))

    (:UNACCEPTABLE-DOMAIN
     (format *error-output*
             "~%;; *AAM COMPILE ERROR*~%;; The frame ~S is not of type ~S, in ~S."
             (first other-args) (second other-args) clause))
    
    (:UNKNOWN-DOMAIN-MATCH
     (format *error-output*
             "~%;; *AAM COMPILE ERROR*~%;; Whether ~S is of type ~S is unknown in ~S." 
             (first other-args) (second other-args) clause))
    )
  (break "To ignore error, continue")
  T
  )

