;;;; -*- Mode:Lisp; Package:AAM; Syntax:COMMON-LISP; Base:10 -*-

(in-package :AAM)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;         Rule storage and lookup for the AAM
;;;
;;;  Rule caches are indexed by set-of-classes.  For example,
;;;  if "Rover" isa (Dogs Mammals Physical-Objects Things),
;;;  the set of classes will be the index to the cache.
;;;
;;;  Caches are frames whose value is on the 'rule-cache'
;;;  slot.  Frames have a '*rule-cache' link to their rule cache, 
;;;  while classes have a 'member' link to every cache they 
;;;  are a member of.
;;;
;;;  This design is by Ben Kuipers, based on a design by
;;;  Spencer Bishop, which resulted from a design by Mike Hewett.
;;;
;;;
;;;
;;;  (retrieve-rules-for <frame> <slot> <value-type> <rule-type>) 
;;;
;;;  (propagate-rule-if-necessary ...)
;;;  (propagate-rules-from-class-to-subclass class subclass)
;;;
;;;  Note: 6 April 1997 (mh)
;;;    - the original design has duplicate lists of rules stored
;;;      in different caches.  The space overhead was reduced
;;;      by using the same list for all the duplicates.  But this
;;;      is a problem because rules can be copied from various
;;;      places and the lists get intermixed.  We have a bug now
;;;      where supposedly duplicate lists in the different places are
;;;      not the same lists.  I'm going to construct new lists,
;;;      in order to overcome the bug.  It's possible that this
;;;      can be remedied by a better design.
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

;; Cache facets, where rules are stored.


(defparameter *rule-caching* T
  "Set to NIL to inhibit rule caching during KB bootstrap.")


(defparameter *rule-facets* (list *if-added      *n-if-added      *if-needed      *n-if-needed
				  *slot-if-added *slot-n-if-added *slot-if-needed *slot-n-if-needed))

(defparameter *non-rule-facets* (list *value *n-value *queries *self-if-added *self-n-if-added)) 


(defun rule-facets (facets)
  "Filters facets to remove all non-rule facets."

  (nset-difference facets *non-rule-facets*))


(defun cached-rules-from (frame slot type direction negated)
  "type is CLASS, SLOT or SELF,
direction is :IF-ADDED or :IF-NEEDED,
negated is T or NIL."

  (let ((facet  (cache-facet type direction negated)))

    (mapcar #'car  (sfs:kb-get-values frame slot facet))
    )
  )


(defun direct-rules-from (frame slot type direction negated)
  "type is *class, *slot or *self,
direction is :IF-ADDED or :IF-NEEDED,
negated is T or NIL."

  (let ((facet (rule-facet type direction negated))
	)
    (mapcar #'car (sfs:kb-get-values frame slot facet))
    )
  )


(defun retrieve-rules-for (frame slot direction negated)
  "(retrieve-rules 'Rover 'length :IF-ADDED NIL)"

  (with-aam-monitoring :RETRIEVE-RULES
    
    ;; Frame may not be a frame when the slot has domains that
    ;; are not frames (such as (CAR) which takes a :LIST in
    ;; the frame argument.

    (let* ((frame-isa-classes   (and (SFS::kb-frame-p frame) (classes-of-frame frame)))
	   (slot-isa-classes    (classes-of-frame slot))
	   (continuation-facet  (rule-facet *self :IF-ADDED negated))
	   )

	;; 1) Class rules from sets the frame belongs to
	;; 2) Slot rules from the slot and the sets it belongs to.
	;; 3) (if-added only) Continuations from the frame.
    
	(append
	 (delete-duplicates 
	  (append
     
	   ;; (1)
	   ;; Cached rules
	   (mapcan #'(lambda (class)
		       (cached-rules-from class slot *class direction negated))
		   frame-isa-classes)

	   ;; Cached generic rules
	   (mapcan #'(lambda (class)
		       (cached-rules-from class *generic *class direction negated))
		   frame-isa-classes)

	   ;; rules stored directly on the class
	   (mapcan #'(lambda (class)
		       (direct-rules-from class slot *class direction negated))
		   frame-isa-classes)

	   ;; generic rules stored directly on the class
	   (mapcan #'(lambda (class)
		       (direct-rules-from class *generic *class direction negated))
		   frame-isa-classes)

	   ;; (2)

	   ;; slot rules stored directly on the slot
	   (direct-rules-from slot slot     *slot direction negated)

	   ;; generic slot rules stored directly on the slot
	   (direct-rules-from slot *generic *slot direction negated)

	   ;; cached slot rules stored on the superclasses of slot
	   (mapcan #'(lambda (class)
		       (cached-rules-from class slot *slot direction negated))
		   slot-isa-classes)

	   ;; cached generic slot rules stored on the superclasses of slot
	   (mapcan #'(lambda (class)
		       (cached-rules-from class *generic *slot direction negated))
		   slot-isa-classes)

	   ;; slot rules stored directly on the superclasses of slot
	   (mapcan #'(lambda (class)
		       (direct-rules-from class slot *slot direction negated))
		   slot-isa-classes)

	   ;; generic slot rules stored directly on the superclasses of slot
	   (mapcan #'(lambda (class)
		       (direct-rules-from class *generic *slot direction negated))
		   slot-isa-classes)
	   )
	  :test #'eq
	  :key  #'rule-closure-rule     ;; for delete-duplicates
	  )
		    
	 ;; (3)
	 (when (eq direction :IF-ADDED)
	   (mapcar #'car (sfs:kb-get-values frame slot continuation-facet)))
	 )       ;; end of append
	)
    )
  )


(defun cache-rule (rule frame slot type direction negated)
  "type is *class, *slot or *self,
direction is :IF-ADDED or :IF-NEEDED."

  ;; This is only called when a new rule has been added on a new facet.

  (let* ((facet       (cache-facet type direction negated))
	 (rules       (sfs:kb-get-values frame slot facet))
	 )

    (unless (listp rule)
      (break "CACHE-RULE:  Got a non-list rule from somewhere."))

    (if rules
	(unless (member (rule-closure-id (car rule)) rules
			:key #'(lambda (x) (rule-closure-id (car x))))
	  (nconc rules (list rule)))
      ;;else
      (sfs:kb-put-value frame slot facet rule))

;; DEBUGGING
;;    (format t "~%Cache for ~S, ~S, ~S is now: ~S"
;;	    frame slot facet (sfs::kb-get-values frame slot facet))
    )
  )


;;;; Higher-level storage functions.

#|
1) A rule or continuation is added to a class or slot.
2) A rule is deleted.
3) A imp-superset link is added from a sub-class to a class
4) A slot is added to a frame.

The cache is a hash table, indexed by the pair (frame . slot).


Implementation of the four cases:

1) The new rule is stored in every cache automatically since 
   we use the same list and values are destructively appended
   to the value list.  We just have to handle the case where
   it was the first rule of that type added.

2) How do we delete rules?   (Happens extremely rarely).

3) All rules on all slots are propagated downward.

4) 

|#

;;; Don't we need *n-generic???
;;; MH:  3 Oct 1996


;;; Type 1 - new rule

(defun propagate-rule-if-necessary (class slot facet)
  "Propagate a new rule to the rule caches."

  (when *rule-caching*
    
#|
;;; Modified 6 Apr 1997 (mh) to not destructively append,
;;; so we need to always propagate the value, not just
;;; when there is only one value.
|#    
;; if the facet contains only one value, it needs to be propagated.

;;;    (let ((values   (sfs::kb-get-values class slot facet)))
;;;      (when (= (length values) 1)

      (let ((rules      (sfs::kb-get-values class slot facet))
	    (subsets    (subsets-of-class class))
	    (type       (rule-facet-type facet))
	    (direction  (rule-facet-direction facet))
	    (negated    (rule-facet-negated facet))
	    )
	(dolist (subset subsets)
	  (dolist (rule rules)
	    (cache-rule rule subset slot type direction	negated)
	  )
	)
      )
    )
  )



;;; Type 2 - delete rule

(defun delete-rule-from-caches (rule class)
  "Delete a rule from all caches."

  ;; this requires that the format of the command be: 
  ;; (:DELETE-RULE <class> rule-name)

  (format *trace-output*
	  "~%;;; *** Algernon doesn't know how to delete rule ~S from ~S yet." rule class)
  )


;;; Type 3 - a new IMP-SUPERSET link has been asserted.

(defun propagate-rules-from-class-to-subclass (class subclass)
  "Propagate all rules."

  (when *rule-caching*
    
    (let* ((class-slots     (sfs:all-slots-of-frame class))

	   type direction negated)

    ;;; Propagate all rules from all slots
      (dolist (slot class-slots)
	(dolist (facet (rule-facets (SFS:all-facets-of-slot class slot)))
	  (setq type      (rule-facet-type      facet))
	  (setq direction (rule-facet-direction facet))
	  (setq negated   (rule-facet-negated   facet))

	  (dolist (rule (sfs:kb-get-values class slot facet))
	    (cache-rule rule subclass slot type direction negated)
	    )
	  )
	)
      )
    )
  )


