;;;; -*- Mode:Lisp; Package:CL-USER; Syntax:COMMON-LISP; Base:10 -*-

(in-package "CL-USER")

;;; The functions in this file are in the AAM package,
;;; but the data constants are in the CL-USER package, so we just
;;; put the whole file in the user package and specify
;;; the AAM package when necessary.

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1995-1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  Definitions of functions for parsing rules in the 
;;;  Algernon Abstract Machine
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

(defmacro rule-antecedent (rule)
  `(first (sfs::kb-get-values ,rule 'antecedent aam::*value)))

(defmacro rule-consequent (rule)
  `(first (sfs::kb-get-values ,rule 'consequent aam::*value)))

(defmacro rule-key (rule)
  `(first (sfs::kb-get-values ,rule 'key aam::*value)))

(defmacro rule-direction (rule)
  `(caar (sfs::kb-get-values ,rule 'direction aam::*value)))

(defmacro rule-index (rule)
  `(caar (sfs::kb-get-values ,rule 'index aam::*value)))

(defmacro rule-root (rule)
  `(caar (sfs::kb-get-values ,rule 'root aam::*value)))

(defmacro rule-code (rule)
  `(car (sfs::kb-get-values ,rule 'code aam::*value)))


;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 


(defun AAM::make-rule (&KEY direction key antecedent consequent (root 'rule)
		            (name 'rule) (index 1))

  (let ((new-rule (GFP::create-instance name 'rules))
	)
    (if (null new-rule)
	(cerror "Return NIL"
		"Unable to create a rule.")
      ;;ELSE
        (progn
	  (SFS::kb-declare-rule         new-rule)
	  (SFS::kb-declare-continuation new-rule)

	  (GFP:add-facet-value new-rule 'direction  'value  direction)
	  (GFP:add-facet-value new-rule 'key        'value  key)
	  (GFP:add-facet-value new-rule 'antecedent 'value  antecedent)
	  (GFP:add-facet-value new-rule 'consequent 'value  consequent)
	  (GFP:add-facet-value new-rule 'index      'value  index)
	  (GFP:add-facet-value new-rule 'root       'value  root)
	  )
	)
    
    new-rule
    )
  )



(defun AAM::def-rule (type frame rule)
  (let* 
      ((direction      (car (or (member '-> rule)
				(member '<- rule))))
       (key-form       (car rule))
       (slot           (if (AAM::variable-p (AAM::slot key-form))
		           AAM::*generic
			 ;;else
			   (AAM::slot key-form)))

       (facet	       (AAM::rule-facet type direction
					(AAM::negated-p key-form)))

       (antecedent     (AAM::rule-head rule direction))
       (consequent     (AAM::rule-tail rule direction))
       (new-rule       NIL)
       (new-rules      NIL)
       (bound-vars     NIL)
       (name           NIL)
       (save-arg-types NIL)
       )


    (when (eq direction '<-);; need to swap ante and conse
      (psetq antecedent consequent consequent antecedent))


    ;; The compiler type-checks each clause.  Because we generate
    ;; several continuation rules from the user-defined rule,
    ;; we need the types of all the variables so that the
    ;; type-checking is done correctly.  (The continuations
    ;; are made from shorter and shorter antecedents,
    ;; so later on the earlier clauses aren't available to
    ;; deduce types from.
    (setq AAM::*ARGUMENT-TYPES* NIL)
    
    ;; For backward chaining rules, do the consequent first.
    (dolist (clause (if (eq direction '<-) consequent antecedent))
      (AAM::validate-argument-types (AAM::slot            clause)
				    (cons (AAM::frame     clause)
					  (AAM::arguments clause))))

    (dolist (clause (if (eq direction '<-) antecedent consequent))
      (AAM::validate-argument-types (AAM::slot            clause)
				    (cons (AAM::frame     clause)
					  (AAM::arguments clause))))

    (setq save-arg-types AAM::*argument-types*)


    ;; Generate the rules and the continuation rules.

    (if (null antecedent)
	(progn 
	  (setq new-rule
	    (AAM::make-rule :key        key-form
			    :direction  direction
			    :antecedent antecedent
			    :consequent consequent
			    :root       name
			    :name       (or name 'rule)
			    :index      1))

	  ;; Store the rule, save the name
	  (setq name new-rule)
	  (GFP:add-facet-value frame slot facet
			       (aam::make-aam-closure :rule new-rule
						      :bindings NIL
						      :key      NIL))
	  (aam::propagate-rule-if-necessary frame slot facet)

	  ;; Compile the rule.
	  (setq AAM::*argument-types* save-arg-types)
	  (sfs::kb-put-value new-rule 'code  aam::*value
			     (aam:aam-compile-rule new-rule bound-vars))
	  
	  (push new-rule new-rules)
	  )

      ;;else - normal rule
      (dotimes (index (length antecedent))
	(when (= index 1) (setq direction '->))  ;; Continuations are if-added.
	(setq new-rule  (AAM::make-rule :key        key-form
					:direction  direction
					:antecedent antecedent
					:consequent consequent
					:root       name
					:name  (if (= index 0) 'rule
						 (aam::continuation-rule-name
						      name (1+ index)))
					:index      (1+ index)))

	;; Store the rule, save the name
	;; This will also propagate the rule to caches it belongs to, since
	;; KB value appending is destructive
	(when (zerop index)
	  (setq name new-rule)

	  (GFP:add-facet-value frame slot facet
			       (aam::make-aam-closure :rule     new-rule
						      :bindings NIL
						      :key      NIL))
	  (AAM::propagate-rule-if-necessary frame slot facet)
	  )

	
	;; Compile the rule.
	(setq AAM::*argument-types* save-arg-types)
	(sfs::kb-put-value new-rule 'code  aam::*value
			   (aam:aam-compile-rule new-rule bound-vars))
	
	(push new-rule new-rules)
	
	;; On backward-chaining rules, make a continuation for 
	;; the first antecedent.
	(when (eq direction '<-)
	  (setq direction '->)
	  (setq key-form (car antecedent))
	  (setq new-rule  (AAM::make-rule :key        key-form
					  :direction  direction
					  :antecedent antecedent
					  :consequent consequent
					  :root       name
					  :name  (aam::continuation-rule-name
						    name (1+ index))
					  :index      (1+ index)))
	  (push new-rule new-rules)

	  ;; We don't know what arguments will be bound by
	  ;; an if-needed rule, but at least this one will be.
	  (push (aam::frame (car consequent)) bound-vars)

	  (setq AAM::*argument-types* save-arg-types)
	  (sfs::kb-put-value new-rule 'code  aam::*value
			     (aam:aam-compile-rule new-rule bound-vars))
	  )

	(setq bound-vars (append bound-vars
				 (aam::variable-arguments (car antecedent))))

	(setq antecedent (cdr antecedent))
	(setq key-form (car antecedent))
	)
      )  ;; IF

    (nreverse new-rules)
    )
  )


;;;; ------------------  Useful accessor fns  -----------------

(defun AAM::rule-head (rule direction)
  (subst nil (member direction rule) rule))


(defun AAM::rule-tail (rule direction)
  (copy-list (cdr (member direction rule))))


(defun AAM::rule-facet (type direction negated)
  "type is CLASS, SLOT or SELF.
direction is :IF-ADDED or :IF-NEEDED, or -> or <-,
negated is T or NIL."

  (if (eq direction :if-added)
      (setq direction '->)
    ;;else
    (if (eq direction :if-needed)
      (setq direction '<-)))

  (case type
    (class
     (ccase direction
       (-> (if negated aam::*n-if-added  aam::*if-added))
       (<- (if negated aam::*n-if-needed aam::*if-needed))
       ))
    
    (slot 
     (ccase direction
       (-> (if negated aam::*slot-n-if-added   aam::*slot-if-added))
       (<- (if negated aam::*slot-n-if-needed  aam::*slot-if-needed))
       ))

    (self  (if negated aam::*self-n-if-added   aam::*self-if-added))
    )
  )


;;; These are the opposite of the above.
(defun AAM::rule-facet-type (facet)

  (if (member facet (list aam::*if-added aam::*n-if-added aam::*if-needed aam::*n-if-needed))
      AAM::*class
    ;;else
    (if (member facet (list aam::*slot-if-added aam::*slot-n-if-added
			    aam::*slot-if-needed aam::*slot-n-if-needed))
	AAM::*slot
      ;;else
        AAM::*self)
    )
  )

(defun AAM::rule-facet-direction (facet)
  (if (member facet (list aam::*if-added aam::*n-if-added aam::*slot-if-added aam::*slot-n-if-added
			  aam::*self-if-added aam::*self-n-if-added))
      :IF-ADDED
    ;;else
      :IF-NEEDED)
  )

(defun AAM::rule-facet-negated (facet)
  (member facet (list aam::*n-if-added aam::*n-if-needed aam::*slot-n-if-added
		      aam::*slot-n-if-needed aam::*self-n-if-added)
	  :test #'eq)
  )


(defun AAM::cache-facet (type direction negated)
  "type is CLASS or SLOT.
direction is :IF-ADDED or :IF-NEEDED.
negated is T or NIL."

  (case type
    (class
     (case direction
       (:IF-ADDED
	(if negated aam::*cache-n-if-added   aam::*cache-if-added))
       (:IF-NEEDED
	(if negated aam::*cache-n-if-needed  aam::*cache-if-needed))
       ))

    (slot
     (case direction
       (:IF-ADDED
	(if negated aam::*cache-slot-n-if-added   aam::*cache-slot-if-added))
       (:IF-NEEDED
	(if negated aam::*cache-slot-n-if-needed  aam::*cache-slot-if-needed))
       )
     )

    (self  (if negated aam::*self-n-if-added   aam::*self-if-added))
    )
  )


;;;; ------------------  Rule Activation and Storage  -----------------

;;;  Given a rule, a place to store it, and its type, create the rule frame
;;;  and store it.


;;; Check whether a rule is already stored, before creating and storing it.
;;;   This implementation depends on rule variables being represented by
;;;   symbols, and only detects identical source code, not renamed variables
;;;   or equivalent effect.

(defun RULE-ALREADY-PRESENT (frame slot facet arrow key ante conse)
  (let ((value-list (sfs::kb-get-values frame slot facet))
	)
    (#-GCL loop #+GCL sloop:sloop
      for Rule-and-stuff in value-list
          when (and (eql   arrow  (rule-direction  Rule-and-stuff))
                    (equal ante   (rule-antecedent Rule-and-stuff))
                    (equal conse  (rule-consequent Rule-and-stuff))
                    (equal key    (rule-key        Rule-and-stuff)))
            do (return T)
          finally (return nil))))


(defun aam::slot-full-p (clause)
  "Returns T if the max cardinality of the frame/slot is met."

  (let* ((frame        (aam::frame clause))
         (slot         (aam::slot  clause))
         (cardinality  (caar (sfs::kb-get-values slot 'cardinality 'value)))
         (num-values   (length 
                        (if (aam::negated-p clause)
                          (sfs::kb-get-values frame slot 'n-value)
                          (sfs::kb-get-values frame slot 'value))))
         )
    (if (null cardinality)
        NIL
        ;;else
        (< num-values cardinality)
        )
    )
  )
