;;;; -*- Mode:Lisp; Package:SFS; Syntax:COMMON-LISP; Base:10 -*-

(in-package :SFS)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  Performance monitoring for the SFS system.
;;;
;;;  (monitor-reset)  -- resets all statistical counters and 
;;;                          turns monitoring on.
;;;  (monitor-on)         -- Turns monitoring on.
;;;  (monitor-off)        -- Turns monitoring off.
;;;  (monitor-stats)      -- Shows results.
;;;
;;;  (monitor <op>)       -- Increment the counter for this operation.
;;;  (monitor-frame <frame>)  -- record a touch of a frame.
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; 30 Oct 1997 (nm)      copy *sfs-stats* in monitor-stats so that
;;;                       sort doesn't destroy the original


;;; This is an a-list of (op #calls time) tuples

(defparameter *MONITOR-CATEGORIES*
  '(:FRAME-P :SLOT-P :FACET-P :RETRIEVE     :SELECT     :STORE :STORE-KNOWN
             :SLOT-FULL-P     :RULE-P       :CONTINUATION-P
	     :DELETE-VALUE    :ALL-CLAUSES  :STORE-NAME :RETRIEVE-NAME
	     :NEW-FRAME       :DELETE-FRAME :CLEAR-SLOT))


(defparameter *SFS-STATS*  NIL)

(defparameter *SFS-MONITORING*  NIL)     ;; Toggle for monitoring


(defun MONITOR (op)   ;; Increments the count.
  (when *SFS-MONITORING*
    (let ((record (assoc op *SFS-STATS*)))
      (incf (second record))
      )
    )
  )


(defun MONITOR-FRAME (frame)   ;; Increments the count.
  (when *SFS-MONITORING*
    (incf (get frame 'touch))
    )
  )


#|
;; Functions are faster...?

(defmacro MONITOR (op)   ;; Increments the count.
  `(when *SFS-MONITORING*
     (let ((record (assoc ,op *SFS-STATS*)))
       (incf (second record))
       )
     )
  )
(defmacro MONITOR-FRAME (frame)   ;; Increments the count.
  `(when *SFS-MONITORING*
     (incf (get ,frame 'touch))
     )
  )
|#


(defmacro with-sfs-monitoring (op &REST body)

  `(progn
     (SFS::monitor ,op)
     ,@body
     )
  )


(defun monitor-reset ()
  
  (setq *SFS-STATS*
    (sort (append
	   (mapcar #'(lambda (category)
		       (list category 0))
		   *MONITOR-CATEGORIES*))
	  #'string-lessp
	  :key #'car
	  )
    )

  (monitor-on)
  )


(defun monitoring-p () *sfs-monitoring*)



(defun monitor-on ()

  (setq *sfs-monitoring* T)
  (format T "~%SFS performance monitoring is now ON.")
  )


(defun monitor-off ()

  (setq *sfs-monitoring* NIL)
  (format T "~%SFS performance monitoring is now OFF.")
  )


(defun monitor-stats ()

  (format T "~%-------------------------------------")
  (format T "~%     SFS Performance Monitoring")
  (format T "~%-------------------------------------")
  (format T "~%   instruction                 calls ")
  (format T "~%   ----------------------------------")

;;; This prints out the info in sorted order, eliminating zeroes.
  (let ((total-calls  0))

    (dolist (category-info (sort (copy-list *SFS-STATS*) #'> :key #'second))
      (unless (zerop (second category-info))
	(format T "~%   ~24S ~8D"
		(first category-info) (second category-info))
	(incf total-calls (second category-info))
	)
      )

    (format T "~%-------------------------------------")
    (format T "~%   Total Calls:             ~8D" total-calls)
    )

  (frame-stats)

  (format T "~2%")

  )


(defun frame-stats ()

  (let ((frames-touched 0)
	(num-touches    0)
	(total-frames   0))

    ;; FRAMES TOUCHED
    (dolist (frame *frame-list*)
      (incf total-frames)
      (when (> (get frame 'touch) 0)
	(incf frames-touched)
	(incf num-touches (get frame 'touch))))

    (format *trace-output*
	    "~2%  ~D of ~D frames touched, ~D total touches."
	    frames-touched total-frames num-touches)

    ;; RULES TOUCHED
    (setq frames-touched 0
	  num-touches    0
	  total-frames   0)

    (dolist (frame *frame-list*)
      (when (kb-rule-p frame)
	(incf total-frames)
	(when (> (get frame 'touch) 0)
	  (incf frames-touched)
	  (incf num-touches (get frame 'touch)))))

    (format *trace-output*
	    "~2%  ~D of ~D rules touched, ~D total touches."
	    frames-touched total-frames num-touches)

    ;; CONTINUATIONS TOUCHED
    (setq frames-touched 0
	  num-touches    0
	  total-frames   0)

    (dolist (frame *frame-list*)
      (when (kb-rule-p frame)
	(when (kb-continuation-p frame)
	  (incf total-frames)
	  (when (> (get frame 'touch) 0)
	    (incf frames-touched)
	    (incf num-touches (get frame 'touch))))))
      
    (format *trace-output*
	    "~2%  ~D of ~D continuations touched, ~D total touches."
	    frames-touched total-frames num-touches)

    )
  )
