;;;; -*- Mode:Lisp; Package:SFS; Syntax:COMMON-LISP; Base:10 -*-

(in-package :SFS)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  A Simple Frame System for use with the Algernon Abstract Machine
;;;  Note - the code is not necessarily optimal in terms of either
;;;         speed or functionality.  But it does provide the basics
;;;         of a frame/slot/facet knowledge representation system.
;;;
;;;  This code is adapted from Algernon's "AKBINT-ALGY.LISP",
;;;  so it's probably under the original copyright by Kuipers, 1992-1996.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

;;; *------- Frame access via names
;;;
;;; Frames are named by a symbol, which is their 'truename'. 
;;; If a suggested name is in use when a frame is to be defined,
;;; the namer will create a new name by appending a number
;;; to the name.
;;;
;;; 20 Aug 1996 (mh)  v1.1  public name indexing
;;; 22 Aug 1996 (mh)  v1.2  added KB-GET-SLOT-DOMAINS
;;; 16 Sep 1996 (mh)  v1.3  added KB-ALL-CLAUSES
;;; 24 Sep 1996 (mh)  v1.4  added monitor-frame and KB-RULE-P
;;; 27 Sep 1996 (mh)  v1.5  added ALL-SLOTS-OF-FRAME
;;;  8 Oct 1996 (mh)  v1.6  added ALL-FACETS-OF-SLOT
;;; 15 Oct 1996 (mh)  v1.7  modified KB-GET-VALUES to do 'join'.
;;; 12 Nov 1996 (mh)  v1.8  modified 'join' to do 'unify'.
;;; 29 Oct 1997 (nm)  v1.8  modified 'join' to do 'instance-p'.

;;;-------------------------------------------------
;;;--- List of interface functions - implemented
;;;-------------------------------------------------

;;; *------- KB clear/reset
;;; KB-RESET

;;; *------- Retrieve
;;; KB-GET-ALL-FRAMES
;;; KB-GET-VALUES               frame slot facet   ; common operation
;;; KB-ALL-CLAUSES  frame
;;; KB-GET-SLOT-DOMAINS slot
;;; ALL-SLOTS-OF-FRAME  frame
;;; ALL-FACETS-OF-SLOT  frame slot

;;; *------- Store
;;; KB-PUT-VALUE frame slot facet value            ; common operation

;;; *------- Definition
;;; KB-DEF-FACET facet-name
;;; KB-DEF-FRAME suggested-name
;;; KB-DEF-SLOT  slot-name domains
;;; KB-DECLARE-RULE frame-name
;;; KB-DECLARE-CONTINUATION frame-name

;;; *------- Delete
;;; KB-DELETE-FRAME    frame
;;; KB-DELETE-VALUE    frame slot facet value       ; :DELETE
;;; KB-DELETE-VALUES   frame slot &OPTIONAL facet   ; :CLEAR-SLOT

;;; *------- Type checking
;;; KB-FACET-P
;;; KB-FRAME-P
;;; KB-RULE-P
;;; KB-SLOT-P
;;; SLOT-FULL-P

;;; *------- Print
;;; KB-PRINT       frame &OPTIONAL stream
;;; KB-PRINT-FACET frame &OPTIONAL stream
;;; KB-PRINT-FRAME frame &OPTIONAL stream
;;; KB-PRINT-FRAME-NO-RULES frame &OPTIONAL stream
;;; KB-PRINT-SLOT  frame &OPTIONAL stream



;;;-------------------------------------------------
;;;--- List of interface functions - not implemented
;;;-------------------------------------------------
;;;
;;;
;;; KB-GET-FRAME-BY-NAME        namestring
;;; KB-GET-ALL-SLOTS-OF-FRAME   frame
;;; KB-GET-ALL-FACETS-OF-SLOT   frame slot
;;;
;;; KB-MERGE-FRAMES frame1 frame2
;;;
;;; KB-RULE-P
;;;
;;; KB-LOAD-SNAPSHOT stream
;;; KB-SAVE-SNAPSHOT stream



#+GCL
(export '(sfs::KB-ALL-CLAUSES
	  sfs::KB-CONTINUATION-P
          sfs::KB-DECLARE-RULE
          sfs::KB-DECLARE-CONTINUATION
	  sfs::KB-DEF-FACET
	  sfs::KB-DEF-FRAME
	  sfs::KB-DEF-SLOT
	  sfs::KB-DELETE-FRAME
	  sfs::KB-DELETE-VALUE
	  sfs::KB-DELETE-VALUES
	  sfs::KB-FACET-P
	  sfs::KB-FRAME-P
	  sfs::KB-GET-ALL-FRAMES
	  sfs::KB-GET-SLOT-DOMAINS
	  sfs::KB-GET-VALUES
	  sfs::KB-PRINT
	  sfs::KB-PRINT-FACET
	  sfs::KB-PRINT-FRAME
          sfs::KB-PRINT-NO-RULES
	  sfs::KB-PRINT-SLOT
	  sfs::KB-PUT-VALUE
	  sfs::KB-RESET
          sfs::KB-RULE-P
	  sfs::KB-SLOT-P
	  sfs::SLOT-FULL-P
	  sfs::ALL-SLOTS-OF-FRAME
	  sfs::ALL-FACETS-OF-SLOT
            
	  sfs::DELETE-NAME
	  sfs::GET-ALL-NAMES
	  sfs::NAMES-RESET
	  sfs::OBJECTS-NAMED
	  sfs::STORE-NAME
	  )
	(find-package :SFS)
	)



;;;; ------  GLOBALS  -----

;;; This prevents us from accidentally messing up
;;; the data structures in the file.
(eval-when (load)
  (when (fboundp 'kb-reset)
    (kb-reset)))
  

(defparameter *FRAME-LIST*  NIL  "Holds names of all frames.")
(defparameter *SLOT-LIST*   NIL  "Holds names of slots and their templates.")
(defparameter *FACET-LIST*  NIL  "Holds names of all facets")

(defconstant  *value        'CL-USER::value)
(defconstant  *n-value      'CL-USER::n-value)

(defparameter *user-package* :CL-USER)



;;;; ------  ACCESSORS  -----

(defun frame-slot-info (frame)
  "Returns implementation-dependent slot info.  No error-checking."
  
  (get frame 'slot-info)
  )


(defun frame-slot-facet-info (frame slot)
  "Returns implementation-dependent slot info.  No error-checking."
  
  (cdr (assoc slot (frame-slot-info frame)))
  )

;;;; ------  predicates  -----

(defmacro variable-p (thing)
  "Returns T if 'thing' (a symbol) is a variable."
  
  `(and (symbolp ,thing)
       (char= (elt (string ,thing) 0) #\?)))


;;;; ------  Some handy error routines  -----

(defmacro frame-error (frame continue-msg)
 `(cerror ,continue-msg "~S is not a frame" ,frame))

(defmacro slot-error (slot continue-msg)
 `(cerror ,continue-msg "~S is not a slot" ,slot))

(defmacro facet-error (facet continue-msg)
 `(cerror ,continue-msg "~S is not a facet" ,facet))

(defmacro slot-full-error (frame slot continue-msg)
 `(cerror ,continue-msg "KB-PUT: The ~S slot of ~S is already full" ,slot ,frame))




;;; *------- KB clear/reset -----------------------------------

(defun frame-initialize (frame)

  (setf (get frame 'count)           0)   ;;; Used for name construction
  (setf (get frame 'slot-info)     NIL)   ;;; Used for slot info
  (setf (get frame 'type)          NIL)   ;;; Not a frame, slot, or facet any more
  (setf (get frame 'touch)           0)   ;;; Used to count touches.
  (setf (get frame 'rulep)         NIL)   ;;; Is this a rule?
  (setf (get frame 'continuationp) NIL)   ;;; Is this a rule continuation?
  )


;;; KB-RESET
(defun kb-reset ()
  "Reset (clear) the existing knowledge base.  The resulting state
should be the same as when the program starts."

  ;; FRAMES
  (dolist (frame (kb-get-all-frames))
    (frame-initialize frame)
    )

  (setq *FRAME-LIST*  NIL)
  (setq *SLOT-LIST*   NIL)
  (setq *FACET-LIST*  NIL)

  (names-reset)

  (when *SFS-MONITORING*
    (monitor-reset))

  (format *trace-output* "~%KB cleared")

  :RESET
  )



;;; *------- Retrieve -----------------------------------------

;;; KB-GET-ALL-FRAMES
(defun kb-get-all-frames ()
  "Return a list of the truenames of all existing frames."
  
  *FRAME-LIST*
  )


;;; KB-GET-VALUES                               ; common operation
(defun kb-get-values (frame slot facet &OPTIONAL (pattern NIL))
  "Returns all values of the frame-slot-facet, or optionally just
those that match a pattern.  Since the arity of the slot is unknown, 
the values will have a list-of-lists form."

  (monitor :RETRIEVE)

  ;; User can use a shortcut notation involving the string name
  ;; of a frame.  Use this if necessary

  (cond ((and (eq slot 'CL-USER::NAME)   ;; Special handling
	      (stringp frame)
	      (eq facet *value))
	 (let ((frames (objects-named frame)))
	   (dolist (fr frames)
	     (monitor-frame fr))
	   (mapcar #'list frames)))
	  
	((kb-frame-p frame)
	 (monitor-frame frame)
	 (cond ((kb-slot-p slot)
		(cond ((kb-facet-p facet)
		       (let ((values
			      (cdr (assoc facet
					  (frame-slot-facet-info frame slot))))
			     )
			 (if pattern
			     (join pattern values)
			   ;;else
			     values)))
		      (T (facet-error facet "Return NIL")
			 NIL)
		      ))

	       (T (slot-error slot "Return NIL") NIL)
	       )
	 )
	  
	(T    (frame-error frame "Return NIL")
	      NIL)
	)
  )


(defun join (pattern values)
  "Not a user-level function.  Returns the members of the VALUES list
that match the pattern."
  
  ;; Removes those values whose values don't match the values in the pattern.
  
  (remove-if-not #'(lambda (value) (instance-p value pattern)) values)
  )


(defun kb-all-clauses (frame)

  (monitor-frame frame)
  (monitor :ALL-CLAUSES)

  ;;; Returns all clauses involving all slots on this frame
  (#-GCL loop #+GCL sloop:sloop
    for (slot . facet-info) in (frame-slot-info frame)
    nconc
	(nconc (mapcar #'(lambda (facet-value)
			   (cons slot
				 (cons frame facet-value)))
		       (cdr (assoc *value facet-info)))
	       (mapcar #'(lambda (facet-value)
			   (list 'NOT (cons slot
					    (cons frame facet-value))))
		       (cdr (assoc *n-value facet-info)))
	       ))
  )


(defun kb-get-slot-domains (slot)

  (cdr (assoc slot *SLOT-LIST*))
  )


(defun all-slots-of-frame (frame)
  "Returns a list of the slots of the frame."

  (mapcar #'car (frame-slot-info frame)))


(defun all-facets-of-slot (frame slot)
  "Returns a list of the facets of the slot"

  (mapcar #'car (frame-slot-facet-info frame slot)))


;;; *------- Store -----------------------------------------
;;; KB-PUT-VALUE                                ; common operation
(defun KB-PUT-VALUE (frame slot facet value)
  "Stores a value in a facet, if the value is not there already.
Returns T if the value was added, NIL if not, :KNOWN if the
value was already known."
  

  (monitor :STORE)

  (let* ((fs-info    (frame-slot-info frame))
	 (fsf-info   (cdr (assoc slot fs-info)))
	 (facet-info (assoc facet fsf-info))
	 )

    ;; Temporary hack to implement :QUOTE, which
    ;; doesn't seem to do anything in Algernon.
    ;; (mh) 10 Sep 1996

    (when (member (car value) '(:QUOTE QUOTE))
      (setq value (second value)))
      
    ;; Check to see whether the value is already there.
    (when (member value (cdr facet-info) :test *equalp-test*)
      (monitor :STORE-KNOWN)
      (return-from KB-PUT-VALUE :KNOWN))

    (cond ((kb-frame-p frame)
	   (monitor-frame frame)
	   (cond ((kb-slot-p slot)
		  (cond ((kb-facet-p facet)

			 (cond ((not (slot-full-p frame slot facet))
				(cond ((null fs-info)	  ;;; First slot added?
				       (setf (get frame 'slot-info)
					 `((,slot . ((,facet . (,value)))))))
                                        
				      ((null fsf-info)	  ;;; new slot?
				       (rplacd (last fs-info)
					       `((,slot . ((,facet . (,value)))))))
                                        
				      ((null facet-info) ;;; new facet?
				       (rplacd (last fsf-info)
					       `((,facet . (,value)))))
                                        
				      (T		    ;;; Just another value
				       (rplacd (last (cdr facet-info))
					       `(,value))))

				;; If it's a name, put it in the NAME kb.
				(when (and (eq slot 'CL-USER::NAME)
					   (eq facet *value))
				  (sfs:store-name frame (car value)))
                                  
				T  ;; Return T
				)
			       (T (slot-full-error frame slot "Return NIL")
				  NIL)
			       ))
			(T (facet-error facet "Return NIL")
			   NIL)
			))

		 (T (slot-error slot "Return NIL") NIL)
		 )
	   )

	  (T    (frame-error frame "Return NIL")
		NIL)
	  )
    )
  )


;;; *------- Definition -----------------------------------------

;;; KB-DEF-FRAME
(defun KB-DEF-FRAME (name &KEY (create T))
  "If the frame already exists, will create a new frame (by making
up a new name) if :CREATE is T.  Otherwise will return the name
of the existing frame."

  (when (kb-frame-p name)
    (cond ((not create)
	   (format *trace-output*
		   "~%Warning: Frame '~S' redefined." name)
	   (return-from KB-DEF-FRAME name))

	  (t  
	   ;; Make a new frame name.
	   ;; This is a common case, for example the user wants
	   ;; a 'rule' and SFS will generate the unique name 'rule5'
	   (incf (get name 'count))

	   ;; Modified 27 Oct 1997 (mh)
	   ;; This was not working correctly with atoms like
	   ;; |MILES STANDISH|.  The second and subsequent
	   ;; names had embedded '|' characters.  
	   ;;(setq name (intern (format nil "~S~D" name (get name 'count))
	   (setq name (intern (format nil "~A~D" (symbol-name name) (get name 'count))
			      *user-package*))
	   ))
    )

  ;;; Initialize the frame.

  (monitor :NEW-FRAME)
  (frame-initialize name)
  (setf (get name 'type) :FRAME)
  (push name *FRAME-LIST*)
  (store-name name (string name))
  name
  )


;;; KB-DEF-FACET facet-name
(defun kb-def-facet (facet-name)
  "Creates a new facet definition."
  
  (let ((new-frame NIL))
    
    (if (member facet-name *FACET-LIST*)
	(format *trace-output* "~%Warning: Facet '~A' redefined." facet-name)
      ;;ELSE
      (progn
	(push facet-name *FACET-LIST*)
	(setq new-frame (kb-def-frame facet-name))
	(setf (get new-frame 'TYPE) :FACET)
	)
      )
    new-frame
    )
  )


;;; KB-DEF-SLOT  slot-name domains
(defun kb-def-slot (slot-name domains)
  "Creates a new slot definition.  If the slot exists, the new
domains replace the old domains."

  (let ((slot-info (assoc slot-name *SLOT-LIST*))
	(new-frame NIL)
	)

    (when slot-info
      (format *trace-output* "~%Warning: Slot '~A' redefined." slot-name)
      (kb-delete-frame slot-name)
      (setq *SLOT-LIST* (delete slot-name *SLOT-LIST* :key #'car))
      )

    (push (cons slot-name domains) *SLOT-LIST*)
    (setq new-frame (kb-def-frame slot-name))
    (setf (get new-frame 'TYPE) :SLOT)

    ;; Return
    new-frame
    )
  )


;;; KB-DECLARE-RULE
(defun kb-declare-rule (frame-name)
  "Declare that frame-name is a rule.  The KB takes note
so that KB-RULE-P works correctly."

  (setf (get frame-name 'RULEP) T)
  )


;;; KB-DECLARE-CONTINUATION
(defun kb-declare-continuation (frame-name)
  "Declare that frame-name is a rule.  The KB takes note
so that KB-CONTINUATION-P works correctly."

  (setf (get frame-name 'CONTINUATIONP) T)
  )



;;; *------- Delete -----------------------------------------

;;; KB-DELETE-FRAME
(defun kb-delete-frame (name)
  "Deletes the frame, but not links to it.  Not very well implemented yet."
  
  (let ((names  (cons (list (string name))
                      (kb-get-values name 'CL-USER::NAME *value)))
	)

    (monitor-frame name)
    (monitor :DELETE-FRAME)
    (dolist (public-name names)
      (delete-name name (car public-name)))

    (frame-initialize name)
    (setq *FRAME-LIST* (delete name *FRAME-LIST*))
    )
  )


;;; KB-DELETE-VALUE                                  ; needed for :DELETE
(defun kb-delete-value (frame slot facet value)
  "Deletes one value from a facet's entry.
Returns T if value was deleted, NIL otherwise."


  (monitor-frame frame)
  (monitor :DELETE-VALUE)

  (when (eq slot 'CL-USER::NAME)
    (delete-name frame (car value)))  ;; Value is a list of arguments

  (let* ((fs-info    NIL)
	 (fsf-info   (frame-slot-facet-info frame slot))
	 (facet-info (assoc facet fsf-info))
	 )

    (unless (member value (cdr facet-info) :test *equalp-test*)
      (return-from KB-DELETE-VALUE NIL))

    (setf (cdr facet-info)
      (delete value (cdr facet-info) :test *equalp-test*))

    ;; Remove the facet
    (when (null (cdr facet-info))
      (setq fsf-info (assoc slot (frame-slot-info frame)))
      (setf (cdr fsf-info)
	(delete facet (cdr fsf-info) :test #'eq :key #'car))

      ;; Possibly remove the whole slot.
      (when (null (cdr fsf-info))
	(setq fs-info (frame-slot-info frame))
	(setf (get frame 'slot-info)
	  (delete slot fs-info :test #'eq :key #'car))
	)
      )
    T
    )
  )



;;; 'delete-all-values' is not a user-level function.
(defun delete-all-values  (frame slot facet)
  
  (let ((slot-info   (assoc slot (frame-slot-info frame)))
	fs-info
	)
    (when slot-info
      (setf (cdr slot-info)
	(delete facet (cdr slot-info) :test #'eq :key #'car))
      )

    ;; Possibly remove the whole slot.
    (when (null (cdr slot-info))
      (setq fs-info (frame-slot-info frame))
      (setf (get frame 'slot-info)
	(delete slot fs-info :test #'eq :key #'car))
      )
    )
  )


;;; KB-DELETE-VALUES                                       ; :CLEAR-SLOT
(defun kb-delete-values (frame slot &OPTIONAL (facet NIL))
  "If facet is specified, clears the facet.  Otherwise,
clears the value and n-value facets.  Returns T if successful."

  (monitor-frame frame)
  (monitor :CLEAR-SLOT)
  (if facet
      (delete-all-values frame slot facet)

    ;ELSE - clear just the value and n-value facets

    (progn
      (delete-all-values frame slot *value)
      (delete-all-values frame slot *n-value)
      )
    )
  )
      


;;; *------- Type predicates

;;; KB-CONTINUATION-P
(defun kb-continuation-p (frame-name)
  "Returns T if the frame is a rule continuation, otherwise NIL.
Frame-name must be a symbol."

  (monitor :CONTINUATION-P)
  (and (symbolp frame-name)
       (get frame-name 'CONTINUATIONP))
  )



;;; KB-FACET-P
(defun kb-facet-p (facet-name)
  "Returns T if the facet exists, otherwise NIL.
Facet-name must be a symbol."

  (monitor :FACET-P)
  (if (and (symbolp facet-name)
	   (eq (get facet-name 'type) :FACET))
      T
    ;;else
      NIL
      )
  )


;;; KB-FRAME-P
(defun kb-frame-p (frame-name)
  "Returns T if the frame exists, otherwise NIL.
Frame-name must be a symbol."

  (monitor :FRAME-P)
  (if (and (symbolp frame-name)
	   (member (get frame-name 'type) '(:FRAME :SLOT :FACET) :test #'eq))
      T
    ;;else
      NIL)
  )


;;; KB-RULE-P
(defun kb-rule-p (frame-name)
  "Returns T if the frame is a rule, otherwise NIL.
Frame-name must be a symbol."

  (monitor :RULE-P)
  (and (symbolp frame-name)
       (get frame-name 'RULEP))
  )


;;; KB-SLOT-P
(defun kb-slot-p (slot-name)
  "Returns T if the slot exists, otherwise NIL.
Slot-name must be a symbol."

  (monitor :SLOT-P)
  (and (symbolp slot-name)
       (eq (get slot-name 'type) :SLOT))
  )


;;; SLOT-FULL-P
(defun slot-full-p (frame slot facet)
  "Returns T if the slot is full (i.e. number of current contents = slot cardinality), otherwise NIL."

  ;; SLOT-FULL-P only applies to the 'value' facet.

  ;; Sometimes this will be called in the situation where the
  ;; frame is not yet a frame.  Make sure to check for existence.

  (monitor :SLOT-FULL-P)
  (when (and (eq facet *value)
	     (kb-frame-p frame))
    (let ((cardinality  (and (kb-slot-p 'CL-USER::CARDINALITY)
                             (caar (kb-get-values slot 'CL-USER::CARDINALITY *value))))
          )

      (if cardinality
	  (<= cardinality (length (kb-get-values frame slot facet)))
        ;;else
        NIL)
      )
    )
  )



;;; *------- Print

;;; KB-PRINT
(defun kb-print (thing &optional (stream *standard-output*))
  
  (if (kb-facet-p thing)
      (kb-print-facet thing stream)
    ;;ELSE
    (if (kb-slot-p thing)
	(kb-print-slot thing stream)
      ;;ELSE
      (if (kb-rule-p thing)
	  (kb-print-rule thing stream)
	;;ELSE
	(if (kb-frame-p thing)
	    (kb-print-frame thing stream)
	  ;;ELSE
	  (format *error-output* "~%~S unknown." thing)
	  ))))
  )


(defun kb-print-no-rules (thing &optional (stream *standard-output*))
  
  (if (kb-facet-p thing)
      (kb-print-facet thing stream)
    ;;ELSE
    (if (kb-slot-p thing)
	(kb-print-slot thing stream)
      ;;ELSE
      (if (kb-rule-p thing)
	  (kb-print-rule thing stream)
	;;ELSE
	(if (kb-frame-p thing)
	    (kb-print-frame-no-rules thing stream)
	  ;;ELSE
	  (format *error-output* "~%~S unknown." thing)
	  ))))
  )

;;; KB-PRINT-FACET
(defun kb-print-facet (facet &OPTIONAL (stream *standard-output*))

  (format stream "~%~A is a facet." facet)
  )


;;; KB-PRINT-RULE
(defun kb-print-rule (frame &OPTIONAL (stream *standard-output*))

    (monitor-frame frame)

  ;; Print antecedent and consequent

  (let ((ante       (car  (kb-get-values frame 'CL-USER::ANTECEDENT *value)))
	(conse      (car  (kb-get-values frame 'CL-USER::CONSEQUENT *value)))
	(key        (car  (kb-get-values frame 'CL-USER::KEY        *value)))
	(direction  (caar (kb-get-values frame 'CL-USER::DIRECTION  *value)))
	(*print-pretty*  T)
	)

    (cond ((eq direction 'CL-USER::->)
	   (format stream "~%(~A" (car ante))
	   (when (cdr ante)
	     (format stream "~{~% ~A~}" (cdr ante)))
	   (format stream "~% ->")
	   (format stream "~{~% ~A~}~% )" conse)
	   (format stream "~2% KEY: ~A" key))

	  (T
	   (format stream "~%(~A" (car conse))
	   (when (cdr conse)
	     (format stream "~{~% ~A~}" (cdr conse)))
	   (format stream "~% <-")
	   (format stream "~{~% ~A~}~% )" ante)
	   (format stream "~2% KEY: ~A" key))
	   )
    )

  frame
  )


;;; KB-PRINT-FRAME
(defun kb-print-frame (frame &OPTIONAL (stream *standard-output*))

  (monitor-frame frame)
  (format stream "~%~A:" frame)

  (let ((value-facet     NIL)
	(*print-pretty*  T))
    
    (dolist (slot-info (get frame 'slot-info))

      ;; Don't print slot names that start with a '*'
      (unless (char= #\* (elt (string (car slot-info)) 0))
	(format stream "~%~15@A:" (car slot-info))   ;; Slot name

	;; Special handling for the VALUE facet.
	(when (setq value-facet (assoc *value (cdr slot-info)))
	  (format stream " ~{ ~S~}" (cdr value-facet))
	  )

	;; Special handling for the N-VALUE facet.
	(when (setq value-facet (assoc 'CL-USER::n-value (cdr slot-info)))
	  (format stream "~%~15@A:" (format nil "NOT-~A" (car slot-info)))   ;; Slot name
	  (format stream " ~{ ~S~}" (cdr value-facet))
	  )

	(dolist (facet-info (cdr slot-info))
	  ;; Special handling for rule slots to get them to print nicer.

	  (cond ((member (car facet-info) '(CL-USER::value CL-USER::n-value))    NIL)

		((member (car facet-info) 
			 '(cl-user::n-if-added        cl-user::if-added
			   cl-user::n-if-needed       cl-user::if-needed
			   cl-user::slot-n-if-added   cl-user::slot-if-added
			   cl-user::slot-n-if-needed  cl-user::slot-if-needed
			   cl-user::self-if-added     cl-user::self-n-if-added
			   ))
 
		 (format stream "~%~18@S: ~{ ~A~%~}" (car facet-info)
			 (cdr facet-info)))

		(T
		 (format stream "~%~18@S: ~{ ~A~}" (car facet-info)
			 (cdr facet-info)))
		)
	  )
	)
      )
    )
  )



;;; KB-PRINT-FRAME-NO-RULES
(defun kb-print-frame-no-rules (frame &OPTIONAL (stream *standard-output*))

  (format stream "~2%~A:" frame)

  (let ((value-facet NIL))
    (dolist (slot-info (get frame 'slot-info))

      ;; Don't print slot names that start with a '*'
      (unless (char= #\* (elt (string (car slot-info)) 0))

	(when (or (assoc *value   (cdr slot-info))   ;; Only print if values there.
		  (assoc *n-value (cdr slot-info)))
	  (format stream "~%~15@A:" (car slot-info))   ;; Slot name

	  ;; Special handling for the VALUE facet.
	  (when (setq value-facet (assoc *value (cdr slot-info)))
	    (format stream " ~{ ~S~}" (cdr value-facet))
	    )

	  ;; Special handling for the N-VALUE facet.
	  (when (setq value-facet (assoc 'CL-USER::n-value (cdr slot-info)))
	    (format stream "~%~15@A:" (format nil "NOT-~A" (car slot-info)))   ;; Slot name
	    (format stream " ~{ ~S~}" (cdr value-facet))
	    )
	  )
	)
      )
    )
  )


;;; KB-PRINT-SLOT
(defun kb-print-slot (slot &OPTIONAL (stream *standard-output*))

  (format stream "~%~A is a slot with domains ~A~%  and slots:"
	  slot (cdr (assoc slot *SLOT-LIST*)))

  (let ((value-facet  NIL))
    (dolist (slot-info (get slot 'slot-info))
      (format stream "~%~15@A:" (car slot-info))
      
      ;; Special handling for the VALUE facet.
      (when (setq value-facet (assoc *value (cdr slot-info)))
	(format stream " ~{ ~S~}" (cdr value-facet))
	)

      ;; Special handling for the N-VALUE facet.
      (when (setq value-facet (assoc 'CL-USER::n-value (cdr slot-info)))
	(format stream "~%~15@A:" (format nil "NOT-~A" (car slot-info)))   ;; Slot name
	(format stream " ~{ ~S~}" (cdr value-facet))
	)

      (dolist (facet-info (cdr slot-info))
	(unless (member (car facet-info) '(cl-user::value cl-user::n-value))
	  (format stream "~%~18@S: ~{ ~A~}" (car facet-info) (cdr facet-info))
	  )
	)
      )
    )
  )


(defun kb-test ()
  "Tests functionality of KB interface functions.  This function 
should work for all implementations of the KB interface."

  (kb-reset)

;;  (meter-on)

  
  (kb-def-frame  'Things)
  (kb-def-frame  'physical-object)
  (kb-def-frame  'person)
  (kb-def-frame  'man)
  (kb-def-frame  'woman)
  (kb-def-frame  'color)
  (kb-def-frame  'truck)
  (kb-def-frame  'engine)

  (kb-def-slot  'colour   '(physical-object color))
  (kb-def-slot  'cars     '(person truck))
  (kb-def-slot  'engine   '(truck engine))
  (kb-def-slot  'capacity '(engine :number))

  (kb-def-facet *value)
  (kb-def-facet *n-value)

  ;; Where do we put a rule?
  ;; (kb-put-value 'rectangle 'some-slot 'some-facet <rule>)

  (kb-put-value 'truck    'colour    *value  '(brown))
  (kb-put-value 'engine 'capacity  *value  '(1.6))
  (kb-put-value 'man    'cars      *value  '(woman))

  (kb-print-frame   'car)
  (kb-delete-value  'car 'colour *value '(brown))
  (kb-print-frame   'car)

  (kb-print-frame   'engine)
  (kb-delete-values 'engine 'capacity)
  (kb-print-frame   'engine)

;;  (meter-stats)

  (dolist (f (kb-get-all-frames))
    (kb-print-frame f)
    )

;;  (meter-stats)

  )



(defun t1 () 
  
  (print (sfs::kb-def-frame 'truck))
  (print (sfs::kb-def-frame 'brown))
  (print (sfs::kb-def-slot 'color '(physical-objects colors)))
  (print (sfs::kb-def-facet 'value))
  (print (sfs::kb-def-facet 'n-value))
  
  (sfs::kb-print-frame 'truck)
  (sfs::kb-put-value 'truck 'color 'value '(brown))
  (sfs::kb-print-frame 'truck)
  (sfs::kb-delete-value 'truck 'color 'value '(brown))
  (sfs::kb-print-frame 'truck)
  (sfs::kb-put-value 'truck 'color 'value '(brown))
  (sfs::kb-print-frame 'truck)
  (sfs::kb-put-value 'truck 'color 'value '(green))
  (sfs::kb-print-frame 'truck)
  )
