;;;; -*- Mode:Lisp; Package:AAM; Syntax:COMMON-LISP; Base:10 -*-

(in-package :AAM)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  Trace functions for the AAM
;;;
;;;  *trace-level*         - 0 to 4, level of tracing detail (4 is highest detail)
;;;  (trace-aam)           - Turns monitoring on to a certin level
;;;  (untrace-aam)         - Turns monitoring off (level 0).
;;;  (with-aam-silent)     - A macro that binds trace-level to 0.
;;;  (with-normal-output)  - A macro that binds trace-level to 1.
;;;
;;;  *rules-traced*
;;;  (trace-rule ...)      - Trace at level 4 only when the rule is executing.
;;;                          This is mostly useful for debugging the AAM.
;;;  (untrace-rule ...)
;;;  (rule-traced-p ...)
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 



;;; This is an a-list of (op #calls time) tuples

(defparameter *trace-level*  0
  "0: no tracing
1: activation and execution info
2: activation, execution and timing info
3: activation, execution, timing, and register-level info")


(defun trace-aam (&optional (level 1))
  "0: no tracing
1: activation and execution info
2: activation, execution and timing info
3: activation, execution, timing, and register-level info"

  (setq *trace-level* level)
  (format t "~%Tracing is now at level ~d." *trace-level*)
  )


(defun untrace-aam ()
  (setq *trace-level* 0)
  (format t "~%Tracing is now off.")
  )


(defmacro with-aam-silent (&REST body)

  `(let ((aam::*trace-level*  0)
	 )
     ,@body
     )
  )


(defmacro with-normal-output (&REST body)

  `(let ((aam::*trace-level*  1)
	 )
     ,@body
     )
  )



(defmacro CL-USER::with-aam-silent (&REST body)

  `(let ((aam::*trace-level*  0)
	 )
     ,@body
     )
  )


(defmacro CL-USER::with-normal-output (&REST body)

  `(let ((aam::*trace-level*  1)
	 )
     ,@body
     )
  )


;; Used only when bootstrapping the KB
(defmacro CL-USER::with-no-rule-caching (&REST body)

  `(let ((aam::*rule-caching*  NIL)
	 )
     ,@body
     )
  )



;;; ------- Rule Tracing ------
(defparameter *rules-traced*  NIL
  "An entry for the rule means that it's execution should be traced in detail.")

(defun trace-rule (&REST rules)

  (dolist (rule rules)
    (pushnew rule *rules-traced*)
    (format *trace-output* "~%Now tracing ~S." rule)
    )
  (not (null rules))
  )


(defun untrace-rule (&REST rules)

  (dolist (rule rules)

    (cond ((rule-traced-p rule)
           (setq *rules-traced* (delete rule *rules-traced* :test #'eq))
           (format *trace-output* "~%No longer tracing ~S." rule)
           T)
          
          (T
           (format *trace-output* "~%~S was not being traced." rule)
           NIL)
          )
    )
  )


(defun untrace-rules ()

  (setq *rules-traced* NIL)
  (format *trace-output* "~%No longer tracing rules.")
  T)


(defun rule-traced-p (rule)

  (member rule *rules-traced*))
