;;;; -*- Mode:Lisp; Package:AAM; Syntax:COMMON-LISP; Base:10 -*-

(in-package :AAM)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  This is the unification mechanism used by Algernon.
;;;  Copied from Algernon v2.0, with changes to eliminate
;;;  variable-variable matching.  
;;;
;;;  This is simpler than general unification because it
;;;  only matches variables to symbols.  (General unification
;;;  will match variables to expressions.)  
;;;
;;;  The last two comments are now out of date: 29 Oct, 1997.
;;;
;;;  23 Jan 1998 (nm)  Simplified ALGY-UNIFY.
;;;
;;;  29 Oct 1997 (nm)  Rewrote ALGY-UNIFY and added the function
;;;                    INSTANCE-P, which does one way pattern
;;;                    matching.  
;;;  30 Oct 1997 (nm)  Added monitoring to ALGY-UNIFY and INSTANCE-P.
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (ALGY-UNIFY PATTERN EXPRESSION)
;;;    => :FAILED or [binding] (my notation for a list of bindings)
;;;
;;; side-effects: the global variables *pat-env* and *exp-env* may be
;;; destructively modified to include new bindings.
;;;   
;;; *PAT-ENV* is initially (:PAT)
;;; *EXP-ENV* is initially (:EXP)
;;;
;;; A binding is a pair (VAR . TERM).  Any variables in TERM belong
;;; the same enviroment as VAR.  (In Algernon, TERM will either be
;;; a variable, or it will be ground.)
;;; 
;;; ALGY-UNIFY calls ALGY-UNIFY-INTERNAL with the arguments shown.
;;;
;;; (ALGY-UNIFY-INTERNAL PATTERN *PAT-ENV* EXPRESSION *EXP-ENV*)
;;;
;;; (Using separate environments for the PATTERN and the EXPRESSION
;;; makes it unecessary to rename variables.)
;;;
;;; The result of this call is T or NIL.   If it is NIL, ALGY-UNIFY 
;;; returns :FAILED.  If it is T, then ALGY-UNIFY returns a binding
;;; list, the CDR of *PAT-ENV*. 
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

(defparameter *pat-env* (list :pat))
(defparameter *exp-env* (list :exp))

(defmacro make-binding (var term)
  `(cons ,var ,term))

(defmacro binding-var (binding) 
  `(car ,binding))
(defmacro binding-term (binding) 
  `(cdr ,binding))

(defmacro bind (var var-env term)
  `(progn (push (make-binding ,var ,term) (cdr ,var-env))
          t))

;; Note: This may cause x to be re-evaluated.  Use with care!
(defmacro my-variable-p (x)
  `(and (symbolp ,x) (char= #\? (char (symbol-name ,x) 0))))

(defun algy-unify (pat exp)
  "Determines whether PAT and EXP unify, and returns either :FAILED or 
a list of bindings.  The list of bindings is only one half of the mgu,
namely, the bindings for variables in PAT."
  
  (with-aam-monitoring :UNIFY

    (setf (cdr *pat-env*) nil
	  (cdr *exp-env*) nil)
    (let ((result (algy-unify-internal pat *pat-env* exp *exp-env*)))
      (if (null result)
	  :FAILED
	(cdr *pat-env*)) )) )

(defun algy-unify-internal (pat pat-env exp exp-env)
  "Returns T or NIL depending on whether PAT and EXP are unifiable
given the ss-bindings in their respective environments.  *PAT-ENV* 
and *EXP-ENV* may be destructively changed."

  (setq pat (dereference-pat pat pat-env))
  (multiple-value-setq (exp exp-env)
    (dereference-exp exp exp-env pat-env))
  (cond ((my-variable-p exp)
         (if (my-variable-p pat)
	     (if (and (eq pat exp)
		      (eq pat-env exp-env))
	         t
	       ;; Given a choice, we always bind the variable in EXP.
	       ;; Note that if EXP dereferenced to a variable in PAT,
	       ;; this could create a variable to variable binding in
	       ;; *PAT-ENV*.
	       (bind exp exp-env pat))  
	   (bind exp exp-env pat)) )
        ((my-variable-p pat)
	 (bind pat pat-env exp))
        ((consp pat)
	 (if (consp exp)
	     (do ((pat-list pat (cdr pat-list))
		  (exp-list exp (cdr exp-list)))
		 ((or (null pat-list) 
		      (null exp-list))
		  (and (null pat-list) 
		       (null exp-list)) )
	       (unless (algy-unify-internal (car pat-list) pat-env 
					    (car exp-list) exp-env)
	         (return nil)) )
	   nil))
	((consp exp) nil)
	((equalp pat exp) t)
	
	;; I would like to drop the next two cases.
	((and (stringp pat) (not (stringp exp)))
	 (format *error-output* "~%Unifying string ~S to non-string ~S" 
		 pat exp)
	 (equalp (read-from-string pat) exp))
	((and (not (stringp pat)) (stringp exp))
	 (format *error-output* "~%Unifying non-string ~S to string ~S"
		 pat exp)
	 (equalp pat (read-from-string exp)))

	(t nil)	))

;;; Whenever there is a binding *to* a variable, that variable is in
;;; PAT-ENV.  If the term in the binding is not a variable, it
;;; doesn't matter what environment we return.

(defun dereference-exp (term env pat-env)
  "Follows bindings to a non-variable or unbound variable.  Returns
the final term and its environment."

  (if (my-variable-p term)
      (let ((binding (assoc term (cdr env))))
        (if binding
	    (values (dereference-pat (binding-term binding) pat-env)
		    pat-env)  ;; see remark above.
	  (values term env) ))
    (values term env)) )

(defun dereference-pat (pat pat-env)
  "Follows bindings to a non-variable or unbound variable.  Returns
the final term.  (We know that the term's environment is still 
*PAT-ENV*, if it is a variable, because no variable in *PAT-ENV* is
ever bound to a variable in *EXP-ENV*.)"

  (if (my-variable-p pat)
      (let ((binding (assoc pat (cdr pat-env))))
        (if binding
	    (dereference-pat (binding-term binding) pat-env)
	  pat))
    pat) )

(defun my-instantiate (term env)
  (cond ((my-variable-p term)
         (let ((binding (assoc term (cdr env))))
	   (if binding
	       (my-instantiate (binding-term binding) env)
	     term) ))
        ((consp term)
	 (let ((new-car (my-instantiate (car term) env))
	       (new-cdr (my-instantiate (cdr term) env)) )
	   (if (and (eq (car term) new-car)
		    (eq (cdr term) new-cdr))
	       term
	     (cons new-car new-cdr)) ))
        (t term) ))

(defun instance-p (exp pat)
  "Determines whether EXP is an instance of PAT, returning T or NIL.
*PAT-ENV* may be destructively changed.  This function is similar
to ALGY-UNIFY except that variables in EXP are treated as constants."
  (labels ((instance-internal (exp pat)
	     (cond ((my-variable-p pat)
		    (let ((binding (assoc pat (cdr *pat-env*))))
		      (if binding
			  (equalp (binding-term binding) exp)
			(progn (push (make-binding pat exp) (cdr *pat-env*))
			       t)) ))
		   ((consp pat)
		    (if (consp exp)
			(do ((pat-list pat (cdr pat-list))
			     (exp-list exp (cdr exp-list)))
			    ((or (null pat-list) 
				 (null exp-list))
			     (and (null pat-list) 
				  (null exp-list)) )
			  (unless (instance-internal (car exp-list) 
						     (car pat-list))
			    (return nil)) )
		      nil))
		   ((consp exp) nil)
		   ((equalp pat exp) t)
		   ) ))

    (with-aam-monitoring :INSTANCE-P

      (setf (cdr *pat-env*) nil)
      (instance-internal exp pat)) ))


(defun test-unify (&OPTIONAL (length 5))

  (let ((p-vars     '(?x ?y ?z ?u ?v ?w g h i j k l m n o p))
	(e-vars     '(?a ?b ?c g h i j k l m n o p))
	(pattern    NIL)
	(expression NIL)
	)

    (dotimes (i length)
      (push (nth (random (length p-vars)) p-vars) pattern)
      )

    (dolist (p-var pattern)
      (if (aam::variable-p p-var)
	  (push (nth (random (length e-vars)) e-vars) expression)
	;;else
	  (push p-var expression)))

    (setq expression (nreverse expression))

    (format t "~%(aam::algy-unify '~S '~S)" pattern expression)

    (aam::algy-unify pattern expression)
    )
  )


