;;;; -*- Mode:Lisp; Package:CL-USER; Syntax:COMMON-LISP; Base:10 -*-

(in-package :CL-USER)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  User-level interaction (TELL and ASK) for the Algernon
;;;  Abstract Machine
;;;
;;;  (tell)
;;;  (ask)
;;;
;;;  (silently)
;;;  (with-no-continuations)
;;;
;;;  (acom-reset)
;;;  (a-assert)
;;;  (a-query)
;;;  (show-bindings)
;;;  (deftaxonomy taxonomy)
;;;
;;;  (find-rule <key>)
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

;;; user-settable parameters

(defparameter *check-slot-domains*  T
  "Set to NIL to inhibit checking slot domains (after the application is
debugged).  The resulting application will run much faster.")



(defun TELL (path &KEY retrieve eval collect comment)
  "Asserts the clauses in the path.  Analyzes the results."

  (let (values-and-bindings)

    (when comment
      (terpri *trace-output*)
      (format *trace-output* comment))

    (when (>= aam::*trace-level* 2)
      (format t "~2%Tell: ~S" path))
    
    (let ((aam::*backward-chain* (not retrieve)))
      (setq values-and-bindings
	(aam:aam-process (aam:aam-compile path :ASSERT NIL) NIL))
      )


    (when (>= aam::*trace-level* 1)
      (aam::pp-output :TELL values-and-bindings))
    
    (aam::post-op   :TELL values-and-bindings :eval eval :collect collect)
    )
  )
	   

(defun ASK (path &KEY retrieve eval collect comment)
  "Queries the clauses in the path.  Analyzes the results."

  (let (values-and-bindings)

    (when comment
      (terpri *trace-output*)
      (format *trace-output* comment))

    (when (>= aam::*trace-level* 2)
      (format t "~2%Ask: ~S" path))
    
    (let ((aam::*backward-chain* (not retrieve)))
      (setq values-and-bindings
	(aam:aam-process (aam:aam-compile path :QUERY NIL) NIL))
      )

;;    (setq return-value (not (null values-and-bindings)))

    (when (>= aam::*trace-level* 1)
      (aam::pp-output :ASK (copy-tree values-and-bindings))
      )
    
    (aam::post-op   :ASK values-and-bindings :eval eval :collect collect)
    )
  )


(defun VERIFY (path &KEY retrieve eval collect comment)
  "Queries the clauses in the path.  Analyzes the results."

  (let (values-and-bindings
	(return-value NIL))

    (when comment
      (terpri *trace-output*)
      (format *trace-output* comment))

    (when (>= aam::*trace-level* 2)
      (format t "~2%VERIFY: ~S" path))
    
    (let ((aam::*backward-chain* (not retrieve)))
      (setq values-and-bindings
	(aam:aam-process (aam:aam-compile path :VERIFY NIL) NIL))
      )

    (setq return-value (not (null values-and-bindings)))

    (when (>= aam::*trace-level* 1)
      (aam::pp-output :VERIFY values-and-bindings)
      (aam::post-op   :VERIFY values-and-bindings :eval eval :collect collect))

    return-value
    )
  )



  
(defun show-bindings (binding-set)

  (dolist (binding-list binding-set)
    (format t "~2%   Bindings:")
    (dolist (binding binding-list)
      (format t "~%~12@A = ~A" (car binding) (cdr binding))
      ))
  )



;;; ---------  Handling KBs - loading, reloading, resetting  ---------

(defparameter *ALGY-LOADED-KBS*  NIL
  "Records all KBs loaded in.")
    
(defun algy-note-loaded-kb (kb)
  
  ;;; Must insert them in the same order as loaded.
  (unless (member kb *ALGY-LOADED-KBS*)
    (if (null *ALGY-LOADED-KBS*)
	(setq *ALGY-LOADED-KBS* (list kb))
      ;;ELSE
        (nconc *ALGY-LOADED-KBS* (list kb)))))


;;; ---------  USEFUL Algy user-level functions  ---------

;;; This is a commonly-called function in Algernon systems.
;;; This should keep a list of loaded KBs and load all of them.

(defun acom-reset ()

  (setq aam::*TOP-LEVEL-EXIT-TAG* NIL)    ;; Reset Catch/Throw mechanism.

  (sfs::kb-reset)

  (dolist (kb *ALGY-LOADED-KBS*)
    (case kb
      (:CORE        (load-algy-file "algy-core-kb"))
      (:COMMON      (load-algy-file "algy-common-kb"))
      (:SET-THEORY  (load-algy-file "algy-set-theory-kb"))
      )
    )
  )



(defun A-ASSERT (sentence predicates &key collect)
  (with-normal-output
    (tell predicates :comment sentence :collect collect)))

(defun A-QUERY (sentence predicates &key collect)
  (with-normal-output
    (ask predicates :comment sentence :collect collect)))



(defmacro WITH-NO-CONTINUATIONS (&REST body)

  `(let ((aam::*make-continuation* NIL))
     ,@body)
  )


(defmacro deftaxonomy (taxonomy)
  "Same as the :TAXONOMY special form in Algernon."

  `(aam::op-taxonomy ',taxonomy))


;;;--------- User-level access functions

;; Modified 06 Feb 1998 (mh) to return a list of rules
;; found rather than printing them out.  This lets
;; the function be used programmatically.
(defun find-rule (key)
  "Given the 'key' clause, locates the rule or rules."

  (let ((rules-found NIL))
    (dolist (rule (mapcar #'car (sfs::kb-get-values 'rules 'member 'value)))
      (when (equal (car (sfs::kb-get-values rule 'key 'value)) key)
	(push rule rules-found)
	;; (format *standard-output* "~%~S" rule))
	)
      )
    rules-found
    )
  )


(defun algy-trace (&OPTIONAL (level 1))
  "0: no tracing
1: normal operation success/failure
2: extended trace - shows rules being activated and executed
3: additionally shows internal registers."

  (aam::trace-aam level)
  )


(defmacro SILENTLY (&REST body)
  "Sets trace level to 0 while executing the body."

  `(with-aam-silent
     ,@body
     )
  )


(defun algy-trace-rule (rule-name)
  "Sets trace level to 2 when the given rule is being executed."

  (aam::trace-rule rule-name)
  )


(defun algy-untrace-rule (rule-name)
  "No tracing for the given rule."

  (aam::untrace-rule rule-name)
  )


(defun algy-untrace ()
  "Removes all rule tracing and sets the trace level to 1."

  (aam::untrace-rules)
  (algy-trace 1)
  )
