;;;; -*- Mode:Lisp; Package:AAM; Syntax:COMMON-LISP; Base:10 -*-

(in-package :AAM)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1996 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;;
;;;  Definitions of functions for the Algernon Abstract Machine
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 


#+GCL
(export '(aam::DEF-RULE
	  aam::MAKE-RULE
	  aam::AAM-COMPILE
	  aam::AAM-COMPILE-RULE
	  aam::AAM-PROCESS
	  aam::PC
	  aam::RULE-PC
	  aam::TRACE-AAM
	  aam::UNTRACE-AAM
	  aam::TRACE-RULE
	  aam::UNTRACE-RULE
	  aam::UNTRACE-RULES
	  aam::WITH-AAM-SILENT
	  )
	(find-package :AAM)
	)


;;;; ----------  Various useful predicates  ------------


(defconstant  *class             'CL-USER::class)
(defconstant  *slot              'CL-USER::slot)
(defconstant  *self              'CL-USER::self)

(defconstant  *value             'CL-USER::value)
(defconstant  *n-value           'CL-USER::n-value)

(defconstant  *isa               'CL-USER::isa)
(defconstant  *member            'CL-USER::member)
(defconstant  *imp-superset      'CL-USER::imp-superset)
(defconstant  *subset            'CL-USER::subset)
(defconstant  *name              'CL-USER::name)
(defconstant  *disjoint          'CL-USER::disjoint)

(defconstant  *generic           'CL-USER::generic)

(defconstant  *if-added          'CL-USER::if-added)
(defconstant  *if-needed         'CL-USER::if-needed)
(defconstant  *n-if-added        'CL-USER::n-if-added)
(defconstant  *n-if-needed       'CL-USER::n-if-needed)

(defconstant  *slot-if-added     'CL-USER::slot-if-added)
(defconstant  *slot-if-needed    'CL-USER::slot-if-needed)
(defconstant  *slot-n-if-added   'CL-USER::slot-n-if-added)
(defconstant  *slot-n-if-needed  'CL-USER::slot-n-if-needed)

(defconstant  *self-if-added     'CL-USER::self-if-added)
(defconstant  *self-n-if-added   'CL-USER::self-n-if-added)

(defconstant *cache-if-added    'CL-USER::cache-if-added)
(defconstant *cache-if-needed   'CL-USER::cache-if-needed)
(defconstant *cache-n-if-added  'CL-USER::cache-n-if-added)
(defconstant *cache-n-if-needed 'CL-USER::cache-n-if-needed)

(defconstant *cache-slot-if-added    'CL-USER::cache-slot-if-added)
(defconstant *cache-slot-if-needed   'CL-USER::cache-slot-if-needed)
(defconstant *cache-slot-n-if-added  'CL-USER::cache-slot-n-if-added)
(defconstant *cache-slot-n-if-needed 'CL-USER::cache-slot-n-if-needed)

(defconstant  *forward   'CL-USER::->)
(defconstant  *backward  'CL-USER::<-)

(defconstant  *queries   'CL-USER::queries)    ;; Facet for query history






(defun negated-p (clause)
  "Returns T if the clause begins with a NOT."
  
  (eq 'CL-USER::not (first clause))
  )


(defun variable-p (thing)
  "Returns T if 'thing' (a symbol) is a variable."
  
  (and (symbolp thing)
       (char= (char (symbol-name thing) 0) #\?)))


;;; ACCESSOR FUNCTIONS

(defun negate (clause)

  (if (negated-p clause)
      (second clause)
    ;;
    (list 'CL-USER::NOT clause)
    )
  )


(defun slot (clause)

;;;Time estimate:
;;;  3 LISP units (if not negated)
;;;  6 LISP units (if negated)

  (if (negated-p clause)
      (slot (second clause))
    ;;ELSE
      (first clause)
      )
  )


(defun frame (clause)

  (if (eq (slot clause) :RETRIEVE)    ;; Handle special form
    (frame (second clause))
    ;;ELSE
    (if (negated-p clause)
      (frame (second clause))
      ;;ELSE
      (second clause)
      )
    )
  )


(defun facet (clause)

;;;Time estimate:
;;;  2 LISP units

  (if (negated-p clause)
      *n-value
    ;;ELSE
      *value)
  )


(defun arguments (clause)

;;;Time estimate:
;;;  3 LISP units (if not negated)
;;;  6 LISP units (if negated)

  (if (negated-p clause)
      (arguments (second clause))
    ;;ELSE
      (cddr clause)
      )
  )


(defun variable-arguments (clause)

  (cond ((not (consp clause))    NIL)

	((variable-p clause)     (list clause))
    
	(T
	 (case (slot clause)

	   ((:A :BIND :BRANCH)
	    (append
	     (if (listp (frame clause)) (list (car (frame clause))) (list (frame clause)))
	     (mapcan #'variable-arguments (arguments clause))))

	   (:ANY (mapcan #'variable-arguments (cdr clause)))

	   (:ALL-PATHS
	    (append (mapcan #'variable-arguments
			    (if (negated-p clause)
				(cddr (second clause)) (cddr clause)))))

	   ((:ASK :DELETE :RETRIEVE)  (variable-arguments (second clause)))

	   ((:BOUNDP :UNBOUNDP)
	    (list (frame clause)))

	   ((:EVAL :RULES :SHOW :SLOT :SRULES :TAXONOMY :TEST)
	    NIL)    ;; these don't bind anything

	   ((:FAIL :UNP :NO-COMPLETION :NO-CONTINUATION)
	    (mapcan #'variable-arguments (cdr clause)))

	   ((:FORC :THE)
	    (if (listp (frame clause)) (list (car (frame clause))) (list (frame clause))))

	   (:OR	  ;;; (:OR <path> <path> ...))
	    (mapcan #'(lambda (path)
			(mapcan #'variable-arguments path))
		    (cdr clause)))

	   ;; :CLEAR-SLOT
	   (T  (remove-if-not #'variable-p 
			      (cons (frame clause) (arguments clause))))
	   )
	 )
	)
  )



(defun if-needed-rule-p (rule)

  (eq (caar (sfs::kb-get-values rule 'CL-USER::direction *value))
      'CL-USER::<-)
  )



(defun continuation-rule-name (rule-name rule-index)

  (intern (format nil "~S-~3,'0D" rule-name rule-index) :CL-USER)
  )


(defun my-nsubstitute (pat alist)
  (cond ((variable-p pat)
         (nlookup pat alist))
        ((consp pat)
         (cons (my-nsubstitute (car pat) alist)
	       (my-nsubstitute (cdr pat) alist)))
        (t
         pat)))

(defun nlookup (var alist)
  (let ((b (assoc var alist :test #'eq)))
    (if b
	(if (variable-p (cdr b))
	    (nlookup (cdr b) alist)
	  ;;else
	    (cdr b))
      ;;else
	var)))

#|
;;; I guess this version is destructively modifying
;;; the 'alist' to increase its speed on future 
;;; indirect lookups.  However, we don't always want to
;;; destructively modify the alist.  (mh) 18 Nov 1997

(defun nlookup (var alist)
  (let ((b (assoc var alist :test #'eq)))
    (if b
	(if (variable-p (cdr b))
	    (setf (cdr b)
		  (nlookup (cdr b) alist))
	    (cdr b))
	var)))
|#


;;; -----  Lookup routines for Algernon's hierarchy  -------------


(defun imp-supersets-of-frame (frame)
  (mapcar #'car (sfs:kb-get-values frame *imp-superset *value))
  )

(defun classes-of-frame (frame)
  (mapcar #'car (sfs:kb-get-values frame *isa *value))
  )

(defun subsets-of-class (class)
  (mapcar #'car (sfs:kb-get-values class *subset *value))
  )



(defun isa-p (frame type)
  "Returns T if it is provable that frame is of type TYPE."

  ;; This needs to be efficient - it is called many times.

  ;; We know that it isn't directly an ISA relationship
  ;; because SLOT-DOMAINS-MATCH-CLAUSE already checked that.
  ;; So we see whether it is inheritable via the IMP-SUPERSET relation.

  ;; However, IMP-SUPERSET is transitive, so we only need to look
  ;; up one level in the hierarchy.

  (or (member type (imp-supersets-of-frame frame))
      (some #'(lambda (class)
		(member type (imp-supersets-of-frame class)))
	    (classes-of-frame frame)))
  )


(defun show-query-history ()

  (let ((values NIL))

    (dolist (frame (SFS::kb-get-all-frames))
      (dolist (slot (sfs::all-slots-of-frame frame))
	(setq values (sfs::kb-get-values frame slot *queries))
	(when values
	  (format t "~%~20A~20A ~S  " frame slot values)
	  )
	)
      )
    )
  )

  


(defun find-all-continuations ()

  ;; Finds and prints all continuations stored on frames and slots in the KB.

  (format t "~%      KB continuations:~%")
  (format t "~%FRAME               SLOT                CONTINUATION")
  (format t "~%-----               ----                ------------~%")
    
  (dolist (frame (SFS::kb-get-all-frames))
    (dolist (slot (sfs::all-slots-of-frame frame))
      (dolist (rule-inst (sfs::kb-get-values frame slot *self-if-added))
	(format t "~%~20A~20AIA  " frame slot)
	(print-rule-instance (car rule-inst))
	)
      (dolist (rule-inst (sfs::kb-get-values frame slot *self-n-if-added))
	(format t "~%~20A~20ANIA " frame slot)
	(print-rule-instance (car rule-inst))
	)
      )
    )
  )


(defun print-rule-instance (ri)
  "Condenses and prints out the RI."

  ;; Need the rule name and also the binding values
  ;; We can't fit in the binding vars, unfortunately.

  (format t "~A ~{~A ~}"
	  (rule-closure-rule ri)
	  (mapcar #'cdr (rule-closure-bindings ri)))
  )
