// ===================================================================
// Copyright (c) 1997, All rights reserved
//
// This software is free for educational and non-profit use.
// Any for-profit use must be governed by a license available
// from the author, who can be contacted via email at 
// "hewett@cs.stanford.edu"
//
//  This software was written for the Algernon project
//  at the University of Texas at Austin

// Micheal Hewett    (hewett@cs.utexas.edu)
// Spencer Bishop    (sbishop@cs.utexas.edu)
//
// ===================================================================
//  The Amaze GUI talks to Algernon via a socket connection.  Commands are
//  sent from Amaze and information is sent back from Algernon.  It should
//  be possible to perform all Algernon operations from Amaze.
//
// ===================================================================
//
//  Amaze.java  - A GUI  for Algernon version 3.0
//
//  15 Nov 1996 (mh)  Added menu item to load and run an example.
//  21 Nov 1996 (mh)  Re-configured so that LISP opens the connection.
//  10 Feb 1997 (mh)  Changed name from AlgyGui to Maze and 
//                    re-did the Gui interface using SpecJava.
//  17 Feb 1997 (mh)  Changed name from Maze to Amaze
//
// -------------------------------------------------------------------

package amaze;

import java.applet.*;
import java.awt.*;
import java.io.*;
import java.net.*;
import java.util.*;

import amaze.algy.*;      // The Algernon KB interface
import amaze.browser.*;   // The main browser panel
import lib.display.*;     // For the TextWindow     class
import lib.dynatype.*;    // For the LispValue      class
import lib.menu.*;        // For the ActiveMenuItem class
import lib.net.*;         // For the Connection     class


/**
 * This is a class that embodies a browser for the
 * Algernon system.  (Algernon is a mouse, so a 
 * maze is appropriate for viewing it, right.)
 * It communicates with Algernon (or any such 
 * system via a socket.
 * @author  Micheal S. Hewett    hewett@cs.utexas.edu
 * @author  Spencer Bishop       sbishop@cs.utexas.edu
 * @date    Wed Feb 12 14:18:58 1997
 * @version 1.0
 *
 */
public class Amaze extends Applet
{

/* ------------------  PUBLIC Variables   ------------------------------ */
  /**
   * The root path to the Algernon directory, as in:
   * <code>"/u/qr/algy3/"</CODE>.  Passed in as an
   * argument to the Amaze program.
   *
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Mar  5 18:41:42 1997
   * @version 1.0
   * 
   */
  public static String ALGY_DIR = "/u/qr/algy3/";

  /**
   * The path to the amaze directory.
   */
  public static String AMAZE_DIR = "/u/qr/algy3/gui/amaze/";


  /**
   * A pointer to the current KB_Manager.
   * Use it to send and receive messages to and from the external KB.
   * 
   * @see amaze.algy.KBManager
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 08:58:05 1997
   * @version 1.0
   * 
   */
  public static KBManager        KB_MGR;

  /**
   * Contains the port number via which this instance of Amaze is
   * talking to LISP.
   * 
   * @see Amaze#makeServerSocket
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 27 16:55:31 1997
   * @version 1.0
   * 
   */
  int connPort;   // The port to which this Amaze is listening.


  /**
   * A pointer to the top-level display frame used by Amaze.
   * 
   * @see AmazeFrame
   * @see amaze.browser.AlgyBrowser
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 09:00:52 1997
   * @version 1.0
   * 
   */
  public static AmazeFrame       DISPLAY;  // The top-level frame.


/* ------------------  PRIVATE variables   ------------------------------ */

  ServerSocket  listenSocket;  
  static int    stackWindowCount = 0;
  

/* ------------------  CONSTRUCTOR   ------------------------------ */

  public Amaze(int thePort)
  {
    super();

    connPort = makeServerSocket(thePort);   // Find an open port.

    System.err.println("\n\n(Amaze is listening on port " + connPort + ")");
  }


/* ------------------  PUBLIC methods   ------------------------------ */

  /**
   * makeServerSocket tries the given port number, and if it
   * is in use, it calls itself recursively with the port number
   * incremented.  This is only useful for general-purpose
   * socket connections, above 5000.  Obviously it won't work
   * for assigned ports, such as the WWW or Telnet ports.
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 27 16:45:29 1997
   * @version 1.0
   * 
   */
  public int makeServerSocket(int port)
  {
    boolean retry = false;

    try { listenSocket = new ServerSocket(port); }
    catch (SocketException e)
    {
      System.err.print("\nPort " + port + " is busy...trying another");
      retry = true;
    }
    catch (IOException e)
    {
      System.err.print("\nIO Exception while trying port " + port);
      System.exit(1);
    }

    if (retry)
      return makeServerSocket(port + 1);
    else
      return port;
  }


  /**
   * The arguments are the port number and the path
   * to the main Algernon directory.
   */
  public static void main(String args[])
  {
    if (args.length == 1)
    {
      int portNum = -99;

      // Try to find the port number.
      try { portNum = Integer.parseInt(args[0]); }
      catch (NumberFormatException e)
      {
	System.err.println("Amaze: port number '" + args[0] + "' must be an integer.");
	System.exit(1);
      }
      finally {}

      if (portNum < 5000)
      {
	System.err.println("Amaze: port number '" + args[0] + "' must be >= 5000.");
	System.exit(1);
      }

      // Initialize the path to the Algernon directory
      ALGY_DIR  = System.getProperty("ALGY.HOME");
      if (ALGY_DIR == null)
      {
	System.err.print("-DALGY.HOME not specified.  Using default path: ");
	ALGY_DIR = "/u/qr/algy3/";
	System.err.println(ALGY_DIR);
      }
      AMAZE_DIR = ALGY_DIR + File.separator + "gui" + File.separator + "amaze" + File.separator;

      // Start the applet.
      Amaze applet  = new Amaze(portNum);
      applet.init();	
      DISPLAY       = new AmazeFrame("Amaze for Algernon v3", applet, 500, 400);

      applet.start();
    }
    else
    {
      System.err.println("usage: amaze.Amaze <port> [algernon-root-path]");
      System.exit(1);
    }
  }


  public void init()
  {
    try
    {
      // Tell LISP that we are ready.
      System.out.println("(:OPEN " + connPort + ")\n");

      // Accept the connection.
      Socket clientSocket = listenSocket.accept();
      AmazeConnection conn = new AmazeConnection(clientSocket);
      KB_MGR = new KBManager(conn);
    }

    catch (IOException e)
    { System.err.println("Amaze: exception while listening for connections: " + e); 
      System.exit(1);
    }
  }


  public void start()
  {
    // Fix up all the required fields now that everything has been initialized.
    DISPLAY.initIO(KB_MGR);
  }

  public boolean action(Event e, Object what)
  {
    if (e.target instanceof ActiveMenuItem)
    {
      return ((ActiveMenuItem)(e.target)).doAction();
    }

    else
    {
      KB_MGR.showTrace("Target = unknown" + "(" + e + ")");
      KB_MGR.showTrace("Invalid command\n");
    }
    return true;
  }


  /** Places argument window in the center of the root window.
   * @param win The window to be placed.
   */
  public static void centerWin(Window win)
  {
    Dimension dim = win.getToolkit().getScreenSize();
    Dimension siz = win.size();
    
    win.move(dim.width/2 - siz.width/2, dim.height/2 - siz.height/2);
  }


  /** Places argument windows on top of each other in the upper right
   * corner of the screen.
   * @param win The window to be placed.
   */
  public static void stackWin(Window win)
  {
    Dimension dim = win.getToolkit().getScreenSize();
    Dimension siz = win.size();
    
    win.move(dim.width - (siz.width + 2), 25 + (stackWindowCount * 22));  // w, v
    stackWindowCount++;
  }


  /**
   * ShowText displays some text in a dismissable, non-modal window.
   * The input contains the title and body of the window.
   * The full input line is:
   *
   * <pre>
   *   show-text<SP>title<NULL>body<NULL>
   * </pre>
   *
   * The body may contain newlines, but the title may not.
   * @see lib.display.TextWindow
   * @param String line The full input line
   * @return void
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Mon Feb 17 18:20:41 1997
   * @version 1.0
   * 
   */
  public static void ShowText(String title, String body)
  {
    // Display the window.
    TextWindow t = new TextWindow(0, title, body);
    stackWin(t);
    // centerWin(t);
  }


  /**
   * viewFrame will display a frame in a pop-up window.
   * Send in a string that contains the frame name (it
   * may be the actual frame name or its public name).
   * You don't need to verify that you are sending in an 
   * actual frame - this method will perform error checking
   * as necessary.
   * @parameter String frameName
   */
  public static void viewFrame(String frameName)
  {
    KB_MGR.send("(:VIEW-FRAME \"" + frameName + "\")");

    LispValue result = KB_MGR.receive();

    viewFrameData(result);
  }


  /**
   * viewRuleCode will display the compiled code for a rule
   * in one or more windows.  Send in a string that contains the rule name (it
   * may be the actual frame name or its public name).
   * You don't need to verify that you are sending in an 
   * actual frame - this method will perform error checking
   * as necessary.
   * @parameter String ruleFrameName
   */
  public static void viewRuleCode(String ruleFrameName)
  {
    KB_MGR.send("(:VIEW-RULE-CODE \"" + ruleFrameName + "\")");

    LispValue result = KB_MGR.receive();

    viewRuleCodeData(result);
  }


  /**
   * Expects a Lisp expression ("title" "body") and
   * calls ShowText to display the body in a pop-up,
   * non-modal window.
   * 
   * @see lib.dynatype.LispValue
   * @param LispValue the list containing the title and body strings.
   * @return void
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 09:58:31 1997
   * @version 1.0
   * 
   */
  static void viewFrameData(LispValue info)
  {
    String title = info.first().toString();
    String body  = info.second().toString();

    // Remove the quotes from the beginning and end of the strings.

    title = title.substring(1, title.length()-1);
    body  = body.substring(1, body.length()-1);

    // Error checking
    //    System.err.print("\nviewFrameData: "); info.internal_prin1(System.err);
    //    System.err.println(title);
    //    System.err.println(body);
    //    System.err.flush();

    ShowText(title, body);
  }

  /**
   * Expects a Lisp expression (<n> "title-1" "body-1" ... "title-n" "body-n").
   * It calls ShowText to display the body in a pop-up,
   * non-modal window.
   * 
   * @see lib.dynatype.LispValue
   * @param LispValue the list containing the title and body strings.
   * @return void
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 09:58:31 1997
   * @version 1.0
   * 
   */
  static void viewRuleCodeData(LispValue info)
  {
    LispValue count = info.first();

    // It was not a rule.
    if (count == LispValue.NIL)
    {
      ShowText(info.second().toString(), info.third().toString());
      return;
    }

    // Else, display a window for each rule version
    long    n = ((LispInteger)count).getValue();
    String  title, body;

    info = info.cdr();

    for (int i=0; i<n; ++i)
    {
      title = info.first().toString();
      body  = info.second().toString();

      // Remove the quotes from the beginning and end of the strings.

      title = title.substring(1, title.length()-1);
      body  = body.substring(1, body.length()-1);

      ShowText(title, body);

      info = info.cdr().cdr();
    }
  }  
}


     

class AmazeFrame extends Frame
{
  Amaze myApplet;

  static TextArea     lispOutputArea;
  static LispInput    lispInputArea;

  AmazePanel   amazePanel;

  public AmazeFrame(String title, Amaze applet, int width, int height)
  {
    // Create the frame with the specified title.
    super(title);
    myApplet = applet;

    // Create the menu
    new AmazeMenuBar(this, applet);

    // Create the body of the frame.
    // HACK - The line that led to this constructor being called is
    //   Amaze.DISPLAY = new AmazeFrame(...);
    // However, the AlgyBrowser constructed in the constructor of AmazePanel
    // needs to know Amaze.DISPLAY at construction time, so I'm setting it
    // here.  (Spencer B.)
    Amaze.DISPLAY = this;
    amazePanel = new AmazePanel();

    lispInputArea  = amazePanel.lispInputArea;
    lispOutputArea = amazePanel.lispOutputArea;

    this.setLayout(new FlowLayout());
    this.add(amazePanel);

    setBackground(new Color(0, 200, 210));  // Cool blue
    // setBackground(new Color(255, 180, 0));     // orange

    this.pack();
    this.show();
  }

  public boolean action(Event e, Object arg)
  {
    if (e.target instanceof MenuItem)
    {
      myApplet.action(e, arg);
    }
    return false;
  }

  /**
   * Returns the LISP Input area managed by the AmazeFrame.
   * @see AmazePanel
   * @see LispInput
   * @see java.awt.TextField
   * @return LispInput
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Feb 12 14:27:35 1997
   * @version 1.0
   * 
   */
  public static LispInput lispIn()
  {
    return lispInputArea;
  }


  /**
   * Returns the LISP Output area managed by the AmazeFrame.
   * @see AmazePanel
   * @see java.awt.TextArea
   * @return TextArea
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Feb 12 14:27:35 1997
   * @version 1.0
   * 
   */
  public static TextArea lispOut()
  {
    return lispOutputArea;
  }


  /**
   * Links the Lisp Input and Output areas to 
   * the socket connection, to make it easier
   * to write to the areas.  This could be
   * more cleanly implemented, if necessary.
   * @see lib.net.Connection
   * @see AmazePanel
   * @param Connection conn
   * @return void
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Feb 12 14:24:50 1997
   * @version 1.0
   * 
   */
  public void initIO(KBManager kbMgr)
  {
    if (kbMgr == null)
    {
      lispOutputArea.appendText("initIO:  Hey!  kbmgr is NULL!\n");
      lispOutputArea.repaint();
    }

    lispInputArea.setKBManager(kbMgr);
    lispInputArea.setFontInfo();
    kbMgr.setTraceWindow(lispOutputArea);
  }

}



