// ===================================================================
// Copyright (c) 1997, All rights reserved, by Micheal Hewett
//
// This software is free for educational and non-profit use.
// Any for-profit use must be governed by a license available
// from the author, who can be contacted via email at 
// "hewett@cs.stanford.edu"
//
// ===================================================================
//
//  AmazeMenu.java  - the ActiveMenuBar for Amaze
//
//  12 Feb 1997
//
// -------------------------------------------------------------------

package amaze;

import java.awt.*;
import java.io.*;
import java.net.*;

import amaze.algy.*;    // For the KBManager      class
import lib.display.*;   // For the AboutBoxDialog class
import lib.menu.*;      // For the ActiveMenu     class

/**
 * AmazeMenuBar creates the menu for the Amaze window.
 * The constructor's argument is the Frame to which 
 * it should be attached.
 * @see AmazeFrame
 * @see java.awt.MenuBar
 * @author  Micheal S. Hewett    hewett@cs.utexas.edu
 * @date    Wed Feb 12 10:41:20 1997
 * @version 1.0
 *
 */
public class AmazeMenuBar extends MenuBar
{
  Amaze  myApplet;
  Frame myFrame;

  /**
   * Send in the frame to which this MenuBar should be attached.
   * @see AmazeFrame
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Feb 12 10:42:37 1997
   * @version 1.0
   * 
   */
  public AmazeMenuBar(Frame theFrame, Amaze theApplet)
  {
    super();

    myApplet = theApplet;
    myFrame  = theFrame;

    setup();

    // MenuBars are not Components ???!
    // Can't set the color.
    // this.setBackground(Color.pink);

    myFrame.setMenuBar(this);
  }


  /**
   * Setup creates the individual menus from the 
   * instances of MenuItem.
   * 
   * @see lib.menu.ActiveMenuItem
   * @see java.awt.MenuItem
   * @return void
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Feb 12 10:43:46 1997
   * @version 1.0
   * 
   */
  public void setup()
  {
    // FILE menu -----------------------------------------------
    Menu file = new Menu("File");
    add(file);

    file.add(new AboutBoxMenuItem("About Amaze...",      myApplet));
    file.addSeparator();
    file.add(new LoadMenuItem("Load",                   myApplet));
    file.add(new QuitMenuItem("Quit",                   myApplet));

    // EXAMPLES menu -----------------------------------------------
    Menu example = new Menu("Example");
    add(example);

    example.add(new ExampleMenuItem("blocks",           myApplet));
    example.add(new ExampleMenuItem("expert",           myApplet));
    example.add(new ExampleMenuItem("flu",              myApplet));
    example.add(new ExampleMenuItem("fsm",              myApplet));
    example.add(new ExampleMenuItem("john",             myApplet));
    example.add(new ExampleMenuItem("peano",            myApplet));

    // TRACE menu -----------------------------------------------
    Menu trace = new Menu("Trace");
    add(trace);

    trace.add(new TraceMenuItem("1  (normal)",               myApplet));
    trace.add(new TraceMenuItem("2  (more detail)",          myApplet));
    trace.add(new TraceMenuItem("3  (detail, machine code)", myApplet));
    trace.add(new TraceMenuItem("4  (detail, machine code and registers)",
				myApplet));
    trace.addSeparator();
    trace.add(new TraceMenuItem("0  (no output)",              myApplet));
    trace.add(new TraceMenuItem("-1 (performance analysis)",   myApplet));
    trace.addSeparator();
    trace.add(new TraceRuleMenuItem("Trace rule...",           myApplet));
    trace.add(new UntraceRuleMenuItem("Untrace rule...",       myApplet));
    trace.add(new ShowTracedRulesMenuItem("List traced rules", myApplet));

    // KB menu -----------------------------------------------
    Menu kb = new Menu("KB");
    add(kb);

    kb.add(new KBMenuItem("Clear and reload the KB(s)",     myApplet));
    kb.add(new KBMenuItem("Clear the KB, but don't reload", myApplet));
    kb.addSeparator();
    kb.add(new KBMenuItem("Display the list of loaded KBs", myApplet));
    kb.add(new KBMenuItem("Reset the loaded KB list",       myApplet));
    kb.addSeparator();
    kb.add(new KBMenuItem("Load the CORE kb",               myApplet));
    kb.add(new KBMenuItem("Load the COMMON kb",             myApplet));
    kb.add(new KBMenuItem("Load the SET THEORY kb",         myApplet));

    // HELP menu -----------------------------------------------
    Menu help = new Menu("Help");
    this.setHelpMenu(help);

    help.add(new GeneralHelpMenuItem("General help...",             myApplet));
    help.add(new FrameDisplayHelpMenuItem("Frame display help...",  myApplet));
    help.add(new BrowserHelpMenuItem("Browser help...",             myApplet));
    help.add(new LispIOHelpMenuItem("LISP I/O help...",             myApplet));
    help.addSeparator();
    help.add(new AlgernonSyntaxHelpMenuItem("Algernon syntax...",            myApplet));
    help.add(new AlgernonChangesHelpMenuItem("Algernon v3 differences...",   myApplet));
    help.addSeparator();
    help.add(new LicenseHelpMenuItem("Algernon license...",         myApplet));
  }
}


/**
 * Extends ActiveMenuItem to store the applet in a 
 * static class variable named APPLET.
 * @see lib.menu.ActiveMenuItem
 * @author  Micheal S. Hewett    hewett@cs.utexas.edu
 * @date    Wed Feb 12 10:54:46 1997
 * @version 1.0
 *
 */
abstract class AmazeActiveMenuItem extends ActiveMenuItem
{
  static Amaze APPLET;

  public AmazeActiveMenuItem(String label, Amaze theApplet)
  {
    super(label);
    APPLET = theApplet;
  }
}




// AboutBoxMenuItem
class AboutBoxMenuItem extends AmazeActiveMenuItem
{
  static AboutBoxDialog ABD = null;

  public AboutBoxMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet);
  }

  private static URL makeURL(String s) {
    URL u = null;
    try { u = new URL(s); }
    catch (MalformedURLException m) {
      System.err.println("Bad URL " + s + ": " + m);
      m.printStackTrace();
    }
    return(u);
  }

  public boolean doAction()
  {
    if (ABD == null)
    {
      ABD = new AboutBoxDialog("About Amaze",
			       makeURL("file:" + Amaze.AMAZE_DIR + "pics/about3.gif"));

      APPLET.centerWin(ABD);
    }

    ABD.show();
    
    return true;
  }
}



// LOAD item
class LoadMenuItem extends AmazeActiveMenuItem
{
  public LoadMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet);
  }

  public boolean doAction()
  {
    FileDialog stdFile = new FileDialog(APPLET.DISPLAY, "Specify file to load", FileDialog.LOAD);

    stdFile.show();
    if (stdFile.getFile() != null)
      APPLET.KB_MGR.eval("(load \"" + stdFile.getDirectory() + stdFile.getFile() + "\")");

    return true;
  }
}  


// QUIT menu
class QuitMenuItem extends AmazeActiveMenuItem
{
  public QuitMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet);
  }

  public boolean doAction()
  {
    APPLET.KB_MGR.send(":QUIT", true);
    return true;
  }
}  

// Example menu
class ExampleMenuItem extends AmazeActiveMenuItem
{
  public ExampleMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet);
  }

  public boolean doAction()
  {
    String itemSelected = getLabel();
    
    APPLET.KB_MGR.send("(acom-reset)", true);
    if ((itemSelected.equals("blocks"))
	|| (itemSelected.equals("expert")))
      APPLET.KB_MGR.send("(load-algy-file \"algy-common-kb\")", true);
  
    APPLET.KB_MGR.send("(load \"" + Amaze.ALGY_DIR + "examples"
		       + File.separator + itemSelected + "\")", true );
    APPLET.KB_MGR.send("(facts-about-" + itemSelected + ")", true );
    APPLET.KB_MGR.send("(queries-about-" + itemSelected + ")", true );

    return true;
  }
}


// TRACE menu
class TraceMenuItem extends AmazeActiveMenuItem
{
  public TraceMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet);
  }

  public boolean doAction()
  {
    String itemSelected    = getLabel();
    String traceLevelStr   = itemSelected.substring(0, itemSelected.indexOf(" "));
      
    APPLET.KB_MGR.send("(algy-trace " + Integer.parseInt(traceLevelStr) + ")", true );

    return true;
  }
}


// TRACE-RULE menu
class TraceRuleMenuItem extends AmazeActiveMenuItem 
{
  String  ruleName = ":UNKNOWN";

  public TraceRuleMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet);
  }

  public boolean doAction()
  {
    // This could get a list of rules and let the user pick one...
    TextInputWindow win = new TextInputWindow(0, TextInputWindow.GET_ATOM,
					      "Enter the name of the rule to trace:", 
                                              APPLET.DISPLAY, true);
    ruleName = win.getUserInput();
    
    APPLET.KB_MGR.send("(algy-trace-rule '" + ruleName.toUpperCase() + ")", true);
    return true;
  }
}

// UNTRACE-RULE menu
class UntraceRuleMenuItem extends AmazeActiveMenuItem 
{
  String  ruleName = ":UNKNOWN";

  public UntraceRuleMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet);
  }

  public boolean doAction()
  {
    // This could get a list of rules and let the user pick one...
    TextInputWindow win = new TextInputWindow(0, TextInputWindow.GET_ATOM,
					      "Enter the name of the rule to untrace:", 
                                              APPLET.DISPLAY, true);
    ruleName = win.getUserInput();
    
    APPLET.KB_MGR.send("(algy-untrace-rule '" + ruleName.toUpperCase() + ")", true);
    return true;
  }
}


// SHOWTRACEDRULES menu
class ShowTracedRulesMenuItem extends AmazeActiveMenuItem 
{
  public ShowTracedRulesMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet);
  }

  public boolean doAction()
  {
    APPLET.KB_MGR.eval("aam::*rules-traced*");

    return true;
  }
}


// KB menu
class KBMenuItem extends AmazeActiveMenuItem
{
  static  int itemCounter = 0;
  private int itemNumber;

  public KBMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet);

    itemNumber = ++itemCounter;
  }

  public boolean doAction()
  {
    switch (itemNumber)
    {
    case 1:   APPLET.KB_MGR.send("(acom-reset)", true);    break;
    case 2:   APPLET.KB_MGR.send("(sfs::kb-reset)", true); break;

    case 3:   APPLET.KB_MGR.eval("*ALGY-LOADED-KBS*"); break;
    case 4:   APPLET.KB_MGR.send("(setq *ALGY-LOADED-KBS* NIL)", true); break;

    case 5:   APPLET.KB_MGR.eval("(load-algy-file \"algy-core-kb\")"); break;
    case 6:   APPLET.KB_MGR.eval("(load-algy-file \"algy-common-kb\")"); break;
    case 7:   APPLET.KB_MGR.eval("(load-algy-file \"algy-set-theory-kb\")"); break;
    }

    return true;
  }
}


/* ------------------   HELP menu    ------------------------------ */


abstract class HelpMenuItem extends AmazeActiveMenuItem
{
  Window   win      = null;
  String   filename = "";
  String   winTitle = "Help";

  public HelpMenuItem(String label, Amaze theApplet, String file, String title)
  {
    super(label, theApplet);

    filename = file;
    winTitle = title;
  }

  public boolean doAction()
  {
    if (win == null)
    {
      try { win = new TextFileWindow(filename, winTitle); }
      catch (FileNotFoundException fe)
	{ win = new TextWindow(99, "Missing help file", "Can't find the file: " + filename); }
    }
    else
      win.show();

    return true;
  }
}

// Help menu

class GeneralHelpMenuItem extends HelpMenuItem 
{
  public GeneralHelpMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet, Amaze.AMAZE_DIR + "help/general.txt", "General Help");
  }
}

class FrameDisplayHelpMenuItem extends HelpMenuItem 
{
  public FrameDisplayHelpMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet, Amaze.AMAZE_DIR + "help/frame.txt", "Help on Frame Viewing");
  }
}

class BrowserHelpMenuItem extends HelpMenuItem 
{
  public BrowserHelpMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet, Amaze.AMAZE_DIR + "help/browser.txt", "Help on Browsing the KB");
  }
}

class LispIOHelpMenuItem extends HelpMenuItem 
{
  public LispIOHelpMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet, Amaze.AMAZE_DIR + "help/lisp.txt", "Help on LISP I/O");
  }
}

class AlgernonSyntaxHelpMenuItem extends HelpMenuItem 
{
  public AlgernonSyntaxHelpMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet, Amaze.ALGY_DIR + "help/forms.txt", "Algernon syntax");
  }
}


class AlgernonChangesHelpMenuItem extends HelpMenuItem 
{
  public AlgernonChangesHelpMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet, Amaze.ALGY_DIR + "help/aam-changes.txt", "Algernon v3 differences");
  }
}


class LicenseHelpMenuItem extends HelpMenuItem 
{
  public LicenseHelpMenuItem(String label, Amaze theApplet)
  {
    super(label, theApplet, Amaze.ALGY_DIR + "help/uta-limited-software-license.txt", "Algernon license");
  }
}
