// ===================================================================
// Copyright (c) 1997, All rights reserved, by Micheal Hewett
//
// This software is free for educational and non-profit use.
// Any for-profit use must be governed by a license available
// from the author, who can be contacted via email at 
// "hewett@cs.stanford.edu"
//
// ===================================================================
//
//  LispInput.java  - a wrapper around TextField
//
//  12 Feb 1997
//
// -------------------------------------------------------------------


package amaze;

import java.awt.*;
import java.io.*;      // For EOFException

import amaze.algy.*;   // For the KBManager class
import lib.net.*;      // For the Connection class
import lib.dynatype.*; // For LispParser class


/**
 * LispInput is a text field that does parenthesis
 * matching and sends its input off to a Lisp
 * Evaluator.
 * 
 * @see java.awt.TextField
 * @author  Micheal S. Hewett    hewett@cs.utexas.edu
 * @date    Wed Mar  5 09:03:03 1997
 * @version 1.0
 *
 */
class LispInput extends TextField implements Runnable
{
/* ------------------  PRIVATE variables   ------------------------------ */

  protected KBManager kbMgr            = null; 
  protected int       matchingPosition = -1;
  protected boolean   flashing         = false;
  protected Thread    myThread         = null;
  protected Graphics  myGraphics       = null;
  protected String    input;

  // Font info
  protected FontMetrics fontInfo       = null;
  protected int         fontWidth      = 0;
  protected int         fontHeight     = 0;
  protected Color       fgColor        = null;
  protected Color       bgColor        = null;

  protected int         hFudge = 0;
  protected int         vFudge = 6;

/* ------------------  CONSTRUCTOR   ------------------------------ */

  public LispInput(int cols)
  {
    super(cols);

    myThread = new Thread(this, "Parenthesis Matching");
    myThread.start();

    this.setFont(new Font("Courier", Font.PLAIN, 16));
  }

/* ------------------  PUBLIC methods   ------------------------------ */

  public void setKBManager(KBManager kbmanager)
  {
    kbMgr = kbmanager;
  }

  // Set the font - can't do this until it is visible.
  public void setFontInfo()
  {
    myGraphics = this.getGraphics();
    
    fontInfo   = this.getFontMetrics(this.getFont());
    fontWidth  = fontInfo.charWidth('A');
    fontHeight = fontInfo.getHeight();
    fgColor    = this.getForeground();
    bgColor    = this.getBackground().brighter();  // Motif is doing something to us.
    
    hFudge     = fontWidth - 2;
  }


  public void send(String message)
  {
    if (kbMgr == null)
    {
      System.err.println("LispInput:  Hey!  kbMgr is NULL!");
      System.err.flush();
    }

    kbMgr.eval(message);   // result will be printed in the Trace window.
  }

  public boolean action(Event e, Object arg)
  {
    boolean    valid = true;
    LispParser parser;
    
    if (e.target == this)
    {
      // Try to parse the input first.  If incomplete, don't send it.
      LispValue value = LispValue.NIL;

      parser = new LispParser(this.getText());
      
      try { value = parser.parse(); }
      catch (EOFException ex)
	{ // display a dialog?
	  kbMgr.showTrace("\n*** Incomplete LISP Input - fix it and try again.\n");
	  valid = false;
	}

      if (valid)
      {
	this.send((String)e.arg);
	this.setText("");
      }
      return true;
    }
    return false;
  }

  /**
   * Implements parenthesis matching
   */ 
  public boolean keyUp(Event e, int key)
  {
    int PAREN_WAIT_TIME = 500;  // milliseconds

    int    quoteCount = 0;
    int    parenCount = 0;

    // Interrupt the parenthesis matching if it's in progress
    synchronized (myThread)
    {
      if (flashing)
      {
	myThread.interrupt();
      }
    }
	

    if (key == ')')      // Parenthesis matching, if matching one is visible.
    {
      input = this.getText();
      
      // Count doublequotes - don't check if we are in a string
      for (int i=0; i<input.length(); ++i)
	if (input.charAt(i) == '"') ++ quoteCount;

      if ((quoteCount % 2) == 0)    // Do a paren match
      {
	for (int i=input.length()-1; i>=0; --i)
	  if (input.charAt(i) == '"')
	    while (input.charAt(--i) != '"');
	  else
	    if (input.charAt(i) == ')')
	      ++parenCount;
	    else if (input.charAt(i) == '(')
	    {
	      --parenCount;
	      if (parenCount == 0)  // Highlight the paren for an instant
	      {
		matchingPosition = i;
		// Determine whether the matchingPosition is visible
		// This only works in JDK 1.1
//		if ((getCaretPosition() - matchingPosition) < this.getColumns())
//		{
		  synchronized(myThread) { myThread.notify(); }
//		}
		return super.keyUp(e, key);
	      }
	    }
      }
    }

    return super.keyUp(e, key);
  }

  public void run()
  {
    int h, v;     // Top left corner of character to be boxed.

    synchronized (myThread) {
      while (true)
      {
	try { myThread.wait(); }  // Wait to be notified to start flashing
	catch (InterruptedException e) {}

	flashing = true;
	// draw a box around the matching character.
	// The font is a fixed-width font.
	h = matchingPosition * fontWidth + hFudge;
	v = vFudge;

	myGraphics.setXORMode(Color.white);
	myGraphics.fillRect(h, v, fontWidth, fontHeight);
	myGraphics.setPaintMode();

	// Sleep for 0.5 sec or until user types another key
	try { myThread.wait(500L); }
	catch (InterruptedException e) {}
	finally {
	  flashing = false;
	  myGraphics.setXORMode(Color.white);
	  myGraphics.fillRect(h, v, fontWidth, fontHeight);
	  myGraphics.setPaintMode();
	}
      }
    }
  }
}
