// ===================================================================
// Copyright (c) 1997, All rights reserved, by Micheal Hewett
//
// This software is free for educational and non-profit use.
// Any for-profit use must be governed by a license available
// from the author, who can be contacted via email at 
// "hewett@cs.stanford.edu"
//
// ===================================================================
//
//  AlgyFactory.java  - Creates instances of the AlgyFrame classes
//
//  20 Feb 1997
//
// -------------------------------------------------------------------

package amaze.algy;

import lib.dynatype.*;


/**
 * The AlgyFactory knows how to create different Algernon types (in Java).
 * Make an instance of this class and use it to create instances of the types. 
 *                                                                     
 * The only possible error is an OutOfMemoryError, which Java will
 * generate on its own, so we don't do any error checking.
 *
 * Example:
 * <pre>
 *   AlgyFactory      af = new AlgyFactory();
 *   LispValueFactory vf = new LispValueFactory();
 *   LispValue   frame = af.makeAlgyFrame("CAR-DOOR");
 *   LispValue   slot  = af.makeAlgySlot("SIZE");
 *   LispValue   facet = af.makeAlgyFacet("VALUE");
 *   LispValue   value = vf.makeInteger(27);
 *
 *   LispValue   expr  = vf.makeCons(slot, vf.makeList(frame, facet, value));
 * </pre>
 * @see LispValue
 * @author  Micheal S. Hewett    hewett@cs.utexas.edu
 * @date    Thu Feb 20 12:06:13 1997
 * @version 1.0
 *
 */
public class AlgyFactory
{
  /**
   * makeAlgyFrame(String name) creates a new Symbol that
   * represents an Algernon frame.  Does <em>not</em>
   * intern the symbol.
   *
   * @see AlgyFrame
   * @see lib.dynatype.LispSymbol
   * @return LispValue
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:08:32 1997
   * @version 1.0
   * 
   */
  public AlgyFrame     makeAlgyFrame (String itsName)
  {  return new AlgyFrame(itsName); }

  public AlgyFrame     makeAlgyFrame (String itsName, String itsPublicName)
  {  return new AlgyFrame(itsName, itsPublicName); }

  public AlgyFrame     makeAlgyFrame (LispString itsName)
  {  return new AlgyFrame(itsName); }

  public AlgyFrame     makeAlgyFrame (LispString itsName, LispString itsPublicName)
  {  return new AlgyFrame(itsName, itsPublicName); }

  /**
   * makeAlgySlot(String name) creates a new Symbol that
   * represents an Algernon slot.  Does <em>not</em>
   * intern the symbol.
   *
   * @see AlgyFrame
   * @see AlgySlot
   * @see lib.dynatype.LispSymbol
   * @return LispValue
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:08:32 1997
   * @version 1.0
   * 
   */
  public AlgySlot     makeAlgySlot (String itsName)
  {  return new AlgySlot(itsName); }

  public AlgySlot     makeAlgySlot (LispString itsName)
  {  return new AlgySlot(itsName); }


  /**
   * makeAlgyFacet(String name) creates a new Symbol that
   * represents an Algernon facet.  Does <em>not</em>
   * intern the symbol.
   *
   * @see AlgyFrame
   * @see AlgyFacet
   * @see lib.dynatype.LispSymbol
   * @return LispValue
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:08:32 1997
   * @version 1.0
   * 
   */
  public AlgyFacet     makeAlgyFacet (String itsName)
  {  return new AlgyFacet(itsName); }

  public AlgyFacet     makeAlgyFacet (LispString itsName)
  {  return new AlgyFacet(itsName); }



  /**
   * makeAlgyRule(String name) creates a new Symbol that
   * represents an Algernon rule.  Does <em>not</em>
   * intern the symbol.
   *
   * @see AlgyFrame
   * @see AlgyRule
   * @see lib.dynatype.LispSymbol
   * @return LispValue
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:08:32 1997
   * @version 1.0
   * 
   */
  public AlgyRule     makeAlgyRule (String itsName)
  {  return new AlgyRule(itsName); }

  public AlgyRule     makeAlgyRule (LispString itsName)
  {  return new AlgyRule(itsName); }

}
