// ===================================================================
// Copyright (c) 1997, All rights reserved, by Micheal Hewett
//
// This software is free for educational and non-profit use.
// Any for-profit use must be governed by a license available
// from the author, who can be contacted via email at 
// "hewett@cs.stanford.edu"
//
// ===================================================================
//
//  AlgyFrame.java  - A Frame class to represent Algernon data.
//
//  20 Feb 1997
//
// -------------------------------------------------------------------

package amaze.algy;

import java.io.*;

import lib.dynatype.*;

/**
 * An AlgyFrame is a symbol that represents a frame in
 * the Algernon world.  The differentiation is needed
 * to distinguish between regular symbols and those
 * that represent frames, slots and rules.
 *
 * @see lib.dynatype.LispSymbol
 * @see AlgySlot
 * @see AlgyRule
 * @author  Micheal S. Hewett    hewett@cs.utexas.edu
 * @date    Thu Feb 20 15:17:39 1997
 * @version 1.0
 *
 */
public class AlgyFrame extends LispSymbol
{
  /**
   * The publicName is the name by which the user should
   * interact with this symbol.
   * 
   * @see AlgyFrame#getPublicName
   * @see AlgyFrame#setPublicName
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Feb 26 10:57:02 1997
   * @version 1.0
   * 
   */
  LispValue publicName = NIL;


  /**
   * Send in a String or LispString and get back an AlgyFrame.
   * This method does <b>not</b> intern the symbol.
   * You can also optionally send in a public name,
   * which should be printed instead of the frame name.
   * @see lib.dynatype.LispSymbol#LispSymbol
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 15:17:39 1997
   * @version 1.0
   * 
   */
  public AlgyFrame(String frameName)
  {
    this(frameName, frameName);
  }

  public AlgyFrame(String frameName, String publicNameStr)
  {
    super(frameName);

    publicName = new LispString(publicNameStr);
  }

  public AlgyFrame(LispString frameNameString)
  {
    this(frameNameString, frameNameString);
  }


  public AlgyFrame(LispString frameNameString,
		   LispString publicNameString)
  {
    super(frameNameString);

    publicName = publicNameString;
  }


  /**
   * Returns the public name of the AlgyFrame as a LispString.
   * @return LispString
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Feb 26 11:00:40 1997
   * @version 1.0
   * 
   */
  public LispValue getPublicName()
  {
    return publicName;
  }


  /**
   * Sets the public name of the AlgyFrame.
   * You don't normally need to do this since the public
   * name can be sent in to the constructor.
   * @parameter LispString theName the public name of the frame.
   * @return LispString
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Feb 26 11:00:40 1997
   * @version 1.0
   * 
   */
  public LispValue getPublicName(LispString theName)
  {
    publicName = theName;
    return publicName;
  }

}
