// ===================================================================
// Copyright (c) 1997, All rights reserved, by Micheal Hewett
//
// This software is free for educational and non-profit use.
// Any for-profit use must be governed by a license available
// from the author, who can be contacted via email at 
// "hewett@cs.stanford.edu"
//
// ===================================================================
//
//  KBManager.java  - An interface to Algernon's KB Manager
//
//  20 Feb 1997
//
// -------------------------------------------------------------------

package amaze.algy;

import java.awt.*;
import java.io.*;

import amaze.*;
import lib.dynatype.*;


/**
 * The <tt>KBManager</tt> class accesses an Algernon KB over a socket
 * connection to Lisp.  It only supplies an interface to the database
 * underlying Algernon not the full functionality of Algernon itself.
 */
public class KBManager {

  /**
   * The heavily-used VALUE facet of Algernon.
   * 
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 19:36:19 1997
   * @version 1.0
   * 
   */
  public static final AlgyFacet VALUE = new AlgyFacet("VALUE");
  
  /**
   * The heavily-used N-VALUE facet of Algernon.
   * 
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 19:36:19 1997
   * @version 1.0
   * 
   */
  public static final AlgyFacet N_VALUE = new AlgyFacet("N-VALUE");

  /**
   * A factory to create Algernon objects.
   * 
   * @see AlgyFactory
   * @see AlgyFrame
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 20:21:53 1997
   * @version 1.0
   * 
   */
  public static final AlgyFactory AF = new AlgyFactory();

  /**
   * kbConnection contains the Socket stream by which the KB Manager
   * should communicate with the external KB.
   * 
   * @see KBManager#setConnection
   * @see KBManager#getConnection
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 17:11:38 1997
   * @version 1.0
   * 
   */
  AmazeConnection kbConnection;

  /**
   * Construct a new KB Manager by giving it
   * the socket connection by which it should
   * communicate.
   * 
   * @param AmazeConnection algyConnection
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 17:10:52 1997
   * @version 1.0
   * 
   */
  public KBManager(AmazeConnection conn) throws KBManagerNoConnection
  {
    kbConnection = conn;

    if (!testConnection())
      throw new KBManagerNoConnection("Unable to communicate with the KB Manager.");
  }


  /**
   * The TextArea where Trace output should be displayed.
   * 
   * @see KBManager#showTrace
   * @see KBManager#setTraceWindow
   * @see KBManager#getTraceWindow
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 09:08:02 1997
   * @version 1.0
   * 
   */
  TextArea traceWindow;
  

/* ------------------  PRIVATE variables   ------------------------------ */

  static final LispValue  FRAME_TAG  = LispValue.intern(":FRAME");
  static final LispValue  SLOT_TAG   = LispValue.intern(":SLOT");
  static final LispValue  FACET_TAG  = LispValue.intern(":FACET");
  static final LispValue  RULE_TAG   = LispValue.intern(":RULE");


/* ------------------  PUBLIC methods   ------------------------------ */
  
  /**
   * Returns a <tt>LispConsOrNil</tt> list of <tt>AlgyFrame</tt>s naming 
   * frames that match <tt>(name ?x publicName)</tt>.
   */
  public LispValue objectsNamed(LispString publicName)
  {
    return objectsNamedNoQuotes(publicName.toString());
  }

  // Does not add quotes around the string.
  LispValue objectsNamedNoQuotes(String publicName)
  {
    kbConnection.send("(:OBJECTS-NAMED " + publicName + ")");
    return  filter(kbConnection.receive());
  }

  /**
   * Same as above, except is called with a Java string.
   */
  public LispValue objectsNamed(String publicName)
  {
    kbConnection.send("(:OBJECTS-NAMED \"" + publicName + "\")");
    return  filter(kbConnection.receive());
  }

  /**
   * Returns the <tt>LispConsOrNil</tt> list representing the list of
   * tuples occupying the frame-slot-facet position specified by the
   * arguments.  If <tt>frame</tt>, <tt>slot</tt>, or <tt>facet</tt> are
   * not valid, then <tt>getValues</tt> throws, respectively, 
   * <tt>KBManagerInvalidFrame</tt>, <tt>KBManagerInvalidSlot</tt>, or
   * <tt>KBManagerInvalidFacet</tt>.
   */
  public LispValue getValues(LispSymbol frame, 
			     LispSymbol slot,
			     LispSymbol facet) 
         throws KBManagerException
  {
    kbConnection.send("(:GET-VALUES " + frame + " " + slot + " " + facet + ")");
    return  filter(kbConnection.receive());
  }


  /**
   * Eval accepts a string or LispValue, which it sends
   * to Lisp.  The input is echoed to the traceWindow,
   * and the result is displayed there too.
   * 
   * @see lib.dynatype.LispValue
   * @param String message
   * @return LispValue The return value from LISP.
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 09:21:41 1997
   * @version 1.0
   * 
   */
  public LispValue eval(String expr)
  {
    LispValue result;

    showTrace(expr + "\n");
    kbConnection.send("(:EVAL " + expr + ")");
    result = kbConnection.receive();
    showTrace(result.toString() + "\n");

    return result;
  }

  /**
   * Same as above, but with LispValue as the argument type.
   */
  public LispValue eval(LispValue expr)
  {
    return eval(expr.toString());
  }

  /**
   * Send sends the string or Lisp Value to LISP.  An optional argument
   * causes it to also display the message sent on the 
   * trace window.
   * 
   * @see lib.dynatype.LispValue
   * @param String message
   * @return LispValue The return value from LISP.
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 09:21:41 1997
   * @version 1.0
   * 
   */
  public LispValue send(String expr, boolean showP)
  {
    if (showP) showTrace(expr + "\n");
    kbConnection.send(expr);

    return LispValue.NIL;
  }

  /**
   * Same as above, but defaults to no trace.
   */
  public LispValue send(String expr)
  {
    return send(expr, false);
  }

  /**
   * Same as above, but with LispValue as the argument type.
   */
  public LispValue send(LispValue expr, boolean showP)
  {
    return send(expr.toString());
  }

  /**
   * Same as above, but defaults to no trace.
   */
  public LispValue send(LispValue expr)
  {
    return send(expr.toString(), false);
  }

  /**
   * Returns a LISP value read from the external stream.
   * 
   * @see lib.dynatype.LispValue
   * @see amaze.AmazeConnection
   * @return LispValue The return value from LISP.
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 09:21:41 1997
   * @version 1.0
   * 
   */
  public LispValue receive()
  {
    return kbConnection.receive();
  }


  /**
   * Assigns a value to the TextArea where trace output
   * should be displayed using KBManager.showTrace(String).
   *
   * @see KBManager#showTrace
   * @see java.awt.TextArea
   * @param TextArea trace
   * @return void
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 09:06:09 1997
   * @version 1.0
   * 
   */
  public void setTraceWindow(TextArea trace)
  {
    traceWindow = trace;
  }

  /**
   * Displays one or more lines of text in the trace window.
   * Can be called directly, or by a command from the 
   * external program.
   *
   * @see java.awt.TextArea
   * @param String traceOutput  A string that may contain embedded newlines.
   * @return void
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 09:09:25 1997
   * @version 1.0
   * 
   */
  public void showTrace(String traceOutput)
  {
    traceWindow.appendText(traceOutput);

    // This scrolls the window so that the last appended text is visible.
    // JDK 1.1:  traceWindow.setCaretPosition(traceWindow.getCaretPosition() + traceOutput.length());

    traceWindow.repaint();    // This is necessary.
  }



/* ------------------  PRIVATE methods   ------------------------------ */

  /**
   * Tests the connection to the KB Manager.
   */
  boolean testConnection()
  {
    // Print :TEST and see if we get a response.
    kbConnection.send("(:TEST)");
    LispValue response = kbConnection.receive();   // read-no-hang

    return true;
  }

  /**
   * The input from LISP may contain :FRAME, :SLOT, :FACET, and :RULE tags.
   * This function eliminates those tags and changes the symbols
   * following them into instances of AlgyFrame, AlgySlot, AlgyFacet,
   * or AlgyRule.
   *
   * @see AlgyFrame
   * @see AlgySlot
   * @see AlgyFacet
   * @see AlgyRule
   * @param LispValue  The expression to be filtered.
   * @return LispValue The filtered expression.
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Fri Feb 21 20:07:52 1997
   * @version 1.0
   * 
   */
  LispValue filter(LispValue input)
  {
    if (input.atom() == LispValue.T)
      return input;
    else
    {
      if (tagP(input.car()))
	return LispValue.VF.makeCons(tagReplacement(input.first(),
						    input.second(),
						    input.third()),
				     (input.first() == FRAME_TAG) ? filter(input.cdr().cdr().cdr())
				                                  : filter(input.cdr().cdr()));
      else
	return LispValue.VF.makeCons(filter(input.car()),
				     filter(input.cdr()));
    }
  }

  boolean tagP(LispValue tag)
  {
    return ((tag == FRAME_TAG)
            || (tag == SLOT_TAG)
            || (tag == FACET_TAG)
            || (tag == RULE_TAG));
  }


  LispValue tagReplacement(LispValue tag, LispValue frameName, LispValue publicName)
  {
    if (tag == FRAME_TAG)
      if (publicName == LispValue.NIL)
	return LispValue.intern(frameName.symbol_name(),
				AF.makeAlgyFrame((LispString)(frameName.symbol_name())));
      else
	return LispValue.intern(frameName.symbol_name(),
				AF.makeAlgyFrame((LispString)(frameName.symbol_name()),
						 (LispString)publicName));
    else if (tag == SLOT_TAG)
      return LispValue.intern(frameName.symbol_name(),
                              AF.makeAlgySlot((LispString)(frameName.symbol_name())));
    else if (tag == FACET_TAG)
      return LispValue.intern(frameName.symbol_name(),
                              AF.makeAlgyFacet((LispString)(frameName.symbol_name())));
    else if (tag == RULE_TAG)
      return LispValue.intern(frameName.symbol_name(),
                              AF.makeAlgyRule((LispString)(frameName.symbol_name())));
    else
      return frameName;
  }
}

