;;; maze.lisp     Starts Algernon's Maze GUI program.
;;;
;;; Mike Hewett   16 Nov 1996
;;;
;;; See: /lusr/lib/franz/acl4.3/home/code/ipc.cl
;;;
;;;
;;; Run in the background using this command:
;;;
;;;    (mp:process-run-function "Algernon GUI" 'open-algernon-gui)


(in-package :cl-user)

;;; Requires the ability to start a program from LISP.
(load (concatenate 'string @algy-gui-path "lib/lisp-to-unix/lisp-to-unix"))

;;; -----------------  VARIABLES  -------------------------

(defparameter *NULL*  (code-char 0))
(defparameter *QUOTE* #\")

;;; Location of Java library and the gui program

(defparameter *amaze-java-classpath*
         (concatenate 'string @algy-gui-path ":" CL-USER::*JAVA-CLASSPATH*))

(defparameter *algy-gui-program* "amaze.Amaze")


;; This lock is used to prevent output interleaving.
(defvar output-lock (mp:make-process-lock))

(defvar *algy-gui-io*           NIL)
(defvar *algy-gui-stream*       NIL)

(defparameter *algy-gui-port* 5175)   ;; Random, > 5000


;;; A multi-processing version of format that uses a process lock.
(defun mp-format (stream string &REST args)
  (mp:with-process-lock (output-lock)
    (apply #'format stream string args))
  )



;;; -----------------  Main functions  -------------------------

(defun amaze (&KEY (host "localhost") (port *algy-gui-port*))

  (format *standard-output* "~%;;*** Starting Amaze")
  (open-algernon-gui :host host :port port)
  )


(defun amaze-bg (&KEY (host "localhost") (port *algy-gui-port*))

  (format *standard-output* "~%;;*** Starting Amaze in the background.")
  (format *standard-output* "~%;;*** Note that querying the user via standard input/output streams")
  (format *standard-output*
	  "~%;;*** will not work correctly due to process synchronization problems.")
  (format *standard-output*  "~%;;*** The Algernon I/O Library will provide better results.")

  (mp:process-run-function `(:name "Amaze for Algernon"
			     :initial-bindings ,excl:*cl-default-special-bindings*)
			   #'open-algernon-gui
			   :host host :port port)
  )



(defun amaze-send (message)
  (format *algy-gui-stream* "~A " message)  ;; Use ~A to eliminate surrounding quotes.
  (force-output *algy-gui-stream*))


(defun open-algernon-gui (&KEY (host "localhost") (port *algy-gui-port*))

  ;; When running in the background, we don't have any
  ;; standard streams available unless we bind *terminal-io*.
  ;;  (setf *terminal-io* excl:*initial-terminal-io*)
  ;; This function isn't running in the background any more.

  ;; Start the remote program
  (setq *algy-gui-io* (open-external-program
		       (format nil "/p/bin/java -classpath ~A -DALGY.HOME=~A ~A ~D"
			       *amaze-java-classpath*
			       @algy-path
			       *algy-gui-program*
			       *algy-gui-port*)
		       :IO))

  (mp-format *trace-output* "~%;;Waiting for GUI program to start...")

  ;; The GUI sends us an (:OPEN portnum) when it has initialized itself.
  (let ((input  NIL))
    (loop
      (setq input (external-read *algy-gui-IO*))
      (when (eq :OPEN (car input))
	(setq port (second input))
	(return))))


  ;; Open a network stream (socket) to it
  (setq *algy-gui-stream* (ipc:open-network-stream :host host :port port))

  (mp-format *trace-output* "~2%Opened stream ~A/~A" host port)

  ;; Loop, reading strings from the GUI.
  (let ((data   NIL))

    (unwind-protect 
	(loop
	  (setq data (read-from-string (read-line *algy-gui-stream* NIL ":EOF")))
	  (when  (or (eq data :QUIT)
		     (eq data :EOF))
	    (amaze-send (format nil ":QUIT~%"))
	    ;;	    (if (eq data :QUIT)
	    ;;	      (mp-format *trace-output* "~2%Received :QUIT from the GUI...")
	    (if (eq data :EOF)
		(mp-format *trace-output*
			   "~2%GUI appears to have died - connection will be terminated.")
	      )
	    ;;	    )
		  
	    (return))

	  ;; (mp-format *trace-output*  "~%Received: ~S" data)

	  (case (car data)
	    (:EVAL             (format *algy-gui-stream* "~S " (eval (second data)))  ;; Add trailing space
			       (force-output *algy-gui-stream*))
	    (:EVAL-NO-RETURN   (eval (second data)))
	    (:VIEW-FRAME       (amaze-view-frame (second data)))
	    (:VIEW-RULE-CODE   (amaze-view-rule-code (second data)))
	    (:OBJECTS-NAMED    (amaze-objects-named (second data)))
	    (:GET-VALUES       (apply #'amaze-get-values (cdr data)))

	    (:TEST             (format *algy-gui-stream* "(:OK) ") (force-output *algy-gui-stream*))

	    (t                 (mp-format *standard-output* "~%~S" (eval data)))
	    )
	  ))
    (close-algernon-gui data)
    )
  )


(defun close-algernon-gui (&OPTIONAL (data NIL))

  (mp-format *trace-output* "~2%;;  LISP: Closing the Amaze stream.")
  (unless (eql data :EOF)
    (external-raw-output *algy-gui-io* "~%:QUIT~%"))

  (close *algy-gui-stream*)
  (close-external-program *algy-gui-io*)


  #+allegro
  (sys::os-wait)   ;; Waits for the external process to finish.

  (mp-format *trace-output* ";; done.~%")
  )


;;; ---------------  Operations  -------------------


(defun amaze-view-frame (frame-name)

  (let ((actual-frame (car (sfs::objects-named frame-name))))
    (if (sfs::kb-frame-p actual-frame)
	(amaze-show-text (format nil "Frame: ~S" frame-name)
			 (with-output-to-string  (xx)
			   (sfs::kb-print actual-frame xx))
			 )
      ;;else
      (amaze-show-text (format nil "Unknown frame: ~S" frame-name)
		       "There is no frame by that name.")
      )))



(defun amaze-view-frame-no-rules (frame-name)

  (amaze-show-text (format nil "The ~S frame" frame-name)
		   (with-output-to-string  (xx)
		     (sfs::kb-print-no-rules frame-name xx))
		   ))


(defun amaze-show-text (title body)
  (amaze-send (format nil "(~S ~S)" title body)))


(defun amaze-objects-named (public-name)

  (amaze-send
   (format nil "~S " (add-tags (sfs:objects-named public-name))))
  )

(defun amaze-get-values (frame slot facet)

  (amaze-send
   (format nil "~S " (add-tags (sfs:kb-get-values frame slot facet))))
  )


(defun amaze-view-rule-code (rule-frame-name)

  (let ((actual-frame (car (sfs::objects-named rule-frame-name))))
    (if (sfs::kb-rule-p actual-frame)
	(if (aam::if-needed-rule-p actual-frame)
	    (let ((rule-count  (length (car (sfs::kb-get-values actual-frame
								'CL-USER::code 'CL-USER::value))))
		  (rule-code   NIL))
	      (setq rule-code
		(loop for i from 0 to (1- rule-count)
		    nconc (list (format nil "Rule: ~A.~D" actual-frame i)
				(with-output-to-string (xx)
				  (aam::rule-pc actual-frame xx i)))
			  ))
	      (amaze-send (format nil "(~D ~{~S ~S ~})" rule-count rule-code))
	      )
	  ;;else - if-added rule
	  (amaze-send (format nil "(1 ~S ~S)"
			      (format nil "Rule: ~A" actual-frame)
			      (with-output-to-string (xx)
				(aam::pc (car (sfs::kb-get-values
					  actual-frame 'CL-USER::code 'CL-USER::value)) xx))))
	  )
      ;;else
      (amaze-send (format nil "(NIL ~S ~S)" 
			  (format nil "Unknown rule: ~S" rule-frame-name)
			  (format nil "~A is not a rule." actual-frame)))
      )))


;;; ------------  UTILITY functions  ---------------------------------------

;; The symbols in the expr need to be tagged if they are
;; frames, slots, facets, or rules.

(defun add-tags (expr)

  (if (atom expr)
      expr

    ;;else
    (if (symbolp (car expr))
	      ;; (:RULE rulename)
	(cond ((sfs:KB-RULE-P (car expr))
	       (cons :RULE
		     (cons (car expr)
			   (add-tags (cdr expr)))))

	      ;; (:FACET slotname)
	      ((sfs:KB-FACET-P (car expr))
	       (cons :FACET
		     (cons (car expr)
			   (add-tags (cdr expr)))))

	      ;; (:SLOT slotname)
	      ((sfs:KB-SLOT-P (car expr))
	       (cons :SLOT
		     (cons (car expr)
			   (add-tags (cdr expr)))))

	      ;; (:FRAME framename "public-name")
	      ((sfs:KB-FRAME-P (car expr))
	       (cons :FRAME
		     (cons (car expr)
			   (cons (caar (sfs::kb-get-values (car expr) 'CL-USER::NAME 'CL-USER::VALUE))
				 (add-tags (cdr expr))))))

	      ;; No tag
	      (T (rplacd expr
			 (add-tags (cdr expr))))
	      )
      ;;else
      (cons (add-tags (car expr))
	    (add-tags (cdr expr)))
      )
    )
  )

