/*****************************************************************************
 **                           Class BrowserTree                             **
 *****************************************************************************/
//    File: BrowserTree.java
// Descrip: See javadoc comments above the BrowserTree class declaration.
//
// History: Thu Feb 20 19:19:33 1997 created by Spencer Bishop
// $Id$
/*****************************************************************************/

package amaze.browser;

import java.util.*;

import amaze.algy.*;
import lib.dynatype.*;

/**
 * A BrowserRow object displays a subtree of a graph defined by some
 * relation in an Algernon KB.  In order to support this display the
 * BrowserRow needs to maintain the structure of the tree itself even for
 * those parts not currently displayed.  Since methods to build, change
 * and inspect such a tree are totally independent of any particular
 * display scheme (well, sort of anyway), the BrowserTree class
 * encapsulates this functionality to keep it separate from any
 * contamination by the UI.  BrowserTree accesses the information it
 * needs from Algernon through a KBManager object that it is configured
 * with through a call to <tt>setKBManager()</tt>.
 * <p>
 * Now, you might be wondering just what information from Algernon a
 * BrowserTree really represents.  Well, accessing and building the
 * entire tree defined by some relation in an Algernon KB isn't practical,
 * and in some cases impossible (when the relation contains cycles).
 * So what BrowserTree actually represents is a selected prefix of what
 * I would call an "augmented branch".  For instance, if the relation you
 * are interested in is "friend" and you start at the frame "bob", then
 * what BrowserTree actually represents would be something like:
 * <pre>
 * -----------------------------------------
 * |        JANE                    / JOE    |
 * |       /                       / /       |
 * |      /        BOB   SUE----BOB--        |
 * |     /  JOE   /     /          \---MARY  |
 * |    /  /     /     /            \        |
 * | BOB --     /     /              \       |
 * |    \---MARY---JOE---MARY         \      |
 * |     \      \                      AL    |
 * |      \      \                           |
 * |       \      \                          |
 * |        AL     SUE                       |
 * |                                         |
 * |  0     1       2     3       4     5    |
 * |             level numbers               |
 * -----------------------------------------
 * </pre>
 * What happens is this: The user of BrowserTree selects "friend" as the
 * default slot and BOB as the root in level 0. BrowserTree automatically
 * fills in and makes available the next "level", ie. the frames to which
 * BOB stands in the friend relation.  (Note that the way I'm using the
 * term "level" isn't exactly correct because we aren't actually
 * representing a full level, just a complete set of siblings on a
 * level.)  Then, MARY is designated as the selected element on level 1
 * so BrowserTree automatically fills in level 2, and so on.  If the user
 * of BrowserTree later designates JOE as the selected element on level
 * 1, then levels 2 on up disappear and children of JOE are filled in as
 * level 2.
 * <p>
 * There is one small complication to this representation that merits
 * mentioning.  Since slots/relations in Algernon aren't restricted to
 * the binary case, we need to deal with the possibility of having
 * multiple values in a single node in the tree.  This is why the
 * interface to BrowserTree has functions like
 * <tt>setSelectedElem(level_no, node_ndx, elem_ndx)</tt> rather than
 * <tt>setSelectedNode(level_no, node_ndx)</tt>.
 * @see AlgyBrowser
 * @see BrowserRow
 * @see KBManager
 */
final class BrowserTree
{
   private Vector levels;      // Vector of TreeLevels
   private AlgySlot dslot;     // Default slot defining tree
   private AlgyFacet dfacet;   // Default facet defining tree
   private KBManager kb;       /* KBManager to talk to to get stuff from
                                * Algernon. */

   public BrowserTree()
   {
      levels = new Vector();
      dslot = null;
      dfacet = null;
      kb = null;
   }

   public void setKBManager(KBManager kb) { this.kb = kb; }

   public void setDefaultSlot(AlgySlot slot) { dslot = slot; }   

   /** If you want a level in a BrowserTree to branch on something
    * besides the default slot, then use this function to set that
    * exception to the rule.  To go back to the default slot pass
    * the <tt>slot</tt> argument as <tt>null</tt>.  Note that this
    * is not a permanent relationship between a level and slot,
    * each time level N is destroyed (and recreated) by changing the
    * selected item at some level less than N, the level N goes back
    * to using the default slot.
    */
   public void setSlot(int level_no, AlgySlot slot)
   {
      ((TreeLevel)levels.elementAt(level_no)).slot = slot;
   }

   public AlgySlot getDefaultSlot() { return dslot; }

   public AlgySlot getSlot(int level_no)
   {
      return ((TreeLevel)levels.elementAt(level_no)).slot;
   }

   public void setDefaultFacet(AlgyFacet facet) { dfacet = facet; }

   /** Analogous explanation to that of </tt>setSlot()</tt> .
    * @see BrowserTree#setSlot
    */
   public void setFacet(int level_no, AlgyFacet facet)
   {
      ((TreeLevel)levels.elementAt(level_no)).facet = facet;
   }

   public AlgyFacet getDefaultFacet() { return dfacet; }

   public AlgyFacet getFacet(int level_no)
   {
      return ((TreeLevel)levels.elementAt(level_no)).facet;
   }

   /** This function is where all the action happens.  When an element
    * is selected on a particular level, the BrowserTree destroys whatever
    * part of the "augmented branch" may have been cut off by the selection
    * change and calls its KBManager to fill in the next level automatically.
    * Even if the arguments don't actually change the current selection
    * on the specified level, the function follows through with the
    * changes anyway.  This is because things may have changed on the
    * Algernon side and what we are doing is a refresh.
    */
   public void setSelectedElem(int level_no, int node_ndx, int elem_ndx)
   {
      // Kill higher levels
      for (int i = levels.size() - 1; i > level_no; --i)
         levels.removeElementAt(i);

      // Create and fill in next level
      TreeLevel lev = (TreeLevel)levels.elementAt(level_no);
      LispValue children;

      lev.snode_ndx = node_ndx;
      lev.selem_ndx = elem_ndx;
      try {
         children = kb.getValues(
            (LispSymbol)(lev.elementAt(node_ndx, elem_ndx)),
            (lev.slot == null) ? dslot : lev.slot,
            (lev.facet == null) ? dfacet : lev.facet);
      }
      catch (KBManagerException e) {
         System.err.println("ERROR in " +
                            this.getClass().getName() +
                            "::setSelectedElem - Caught " + e.toString());
         // No children for some reason so our job is done
         return;
      }
      if (children instanceof LispCons)  // There are some children
      {
         TreeLevel new_lev = new TreeLevel(
            (int)(((LispInteger)(children.length())).getValue()),
            (int)(((LispInteger)(children.car().length())).getValue()));
         LispValue l1 = children;
         for (int i = 0; l1 instanceof LispCons; ++i)
         {
            LispValue l2 = l1.car();
            l1 = l1.cdr();
            for (int j = 0; l2 instanceof LispCons; ++j)
            {
               new_lev.setElement(i, j, l2.car());
               l2 = l2.cdr();
            }
         }
         levels.addElement(new_lev);
      }
      /* else - No children so do nothing */
   }

   public int[] getSelectedElemIndexes(int level_no)
   {
      int ret_val[] = new int[2];
      TreeLevel lev = (TreeLevel)levels.elementAt(level_no);
      ret_val[0] = lev.snode_ndx;
      ret_val[1] = lev.selem_ndx;
      return ret_val;
   }

   public LispValue elementAt(int level_no, int node_ndx, int elem_ndx)
   {
      return ((TreeLevel)levels.elementAt(level_no)).elementAt(node_ndx, elem_ndx);
   }

   public int numRealizedLevels() { return levels.size(); }

   public TreeLevel getLevel(int level_no) 
   { 
      return ((TreeLevel)levels.elementAt(level_no)); 
   }

   public Enumeration levelsEnumeration() 
   {
      return new BrowserTreeLevelsEnum(this);
   }

   public Enumeration levelsEnumeration(int min_level, int max_level)
   {
      return new BrowserTreeLevelsEnum(this, min_level, max_level);
   }

   public void clearTree() { levels.removeAllElements();}

   public void addRoot(LispValue val)
   {
      if (levels.size() == 0)
      {
         levels.addElement(new TreeLevel(1,1));
         ((TreeLevel)levels.elementAt(0)).setElement(0, 0, val);
      }
      else
         ((TreeLevel)levels.elementAt(0)).addElement(val);
   }
}

/**
 * This class is intimitely tied to the class BrowserTree and you should
 * look there to understand the purpose of this class.
 * @see BrowserTree
 */
class TreeLevel
{
   // These field are here for the sake of BrowserTree, TreeLevel doesn't
   // fool with them.
   AlgySlot slot;       // Slot to follow to next level of the tree
   AlgyFacet facet;     // Facet to follow to next level of the tree
   int snode_ndx;       // Selected node index 
   int selem_ndx;       // Selected element index (in the selected node)

   /* Array representing the nodes in the level.  The rows are nodes
    * and the columns are the elements within the nodes */
   private LispValue nodes[][];

   public TreeLevel(int num_nodes, int elems_per_node)
   {
      slot = null;
      facet = null;
      snode_ndx = selem_ndx = -1;
      nodes = new LispValue[num_nodes][elems_per_node];
   }

   public int numNodes() { return nodes.length; }

   public int elemsPerNode() { return nodes[0].length; }

   public void setElement(int node_ndx, int elem_ndx, LispValue val)
   {
      nodes[node_ndx][elem_ndx] = val;
   }

   public void addElement(LispValue val)
   {
      LispValue old_nodes[][] = nodes;
      nodes = new LispValue[nodes.length + 1][nodes[0].length];
      for (int i = 0; i < old_nodes.length; ++i)
         for (int j = 0; j < old_nodes[0].length; ++j)
            nodes[i][j] = old_nodes[i][j];
      nodes[nodes.length - 1][0] = val;
   }

   public LispValue elementAt(int node_ndx, int elem_ndx)
   {
      return nodes[node_ndx][elem_ndx];
   }

   public int[] getSelectedElemIndexes()
   {
      int ret_val[] = new int[2];
      ret_val[0] = snode_ndx;
      ret_val[1] = selem_ndx;
      return ret_val;
   }
}

/** A class that implements the Enumeration interface for enumerating
 * over the levels in a BrowserTree.
 */
class BrowserTreeLevelsEnum implements Enumeration
{
   private BrowserTree t;
   private int curr;
   private int last;
   
   public BrowserTreeLevelsEnum(BrowserTree t)
   {
      this(t, 0, t.numRealizedLevels() - 1);
   }
   
   public BrowserTreeLevelsEnum(BrowserTree t, int first, int last)
   {
      this.t = t;
      curr = first;
      this.last = last;
   }

   public boolean hasMoreElements() { return curr <= last; }
   
   public Object nextElement() { return t.getLevel(curr++); }
}
