/*****************************************************************************
 **                        Gui Server for Algernon                          **
 *****************************************************************************/
//    File: GuiServer.java
// Descrip: Implements a server that uses a GUI to ask simple questions and
//          return the responses to the client program.  The server 
//          communicates with the client through the server's standard input
//          and output.
//
// History: Wed Jan 15 22:17:16 1997 created by Spencer Bishop
// $Id$
/*****************************************************************************/

package ioserver;

import java.awt.*;
import java.io.*;
import java.util.*;

import lib.display.*;

/*****************************************************************************
 **                            Class GuiServer                              **
 *****************************************************************************/
/**
 * The class implementing the the function <TT>main()</TT> for the Java
 * application <EM>GuiServer</EM>.  The GuiServer application is a server
 * that takes as input simple text based commands and displays windows 
 * giving the user information or requesting user input.  Any data the user
 * enters is sent back to the client for further processing.  The server
 * is not very complex since it can handle only one client, and for the
 * time being, it communicates with the client over the server's standard
 * input and output.
 * <P>
 * To learn about the command language the GuiServer application responds
 * to and the function of each of the commands visit 
 * <a href="file:/home/bubbie/work/algy/gui-server/io-server.html">
 * GuiServer specification</a>.
 */
public class GuiServer implements WinOwner
{
  /**
   * The path of the Algernon root directory,
   * e.g. <code>"/u/qr/algy3/"</CODE>
   * 
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Mar  5 18:51:49 1997
   * @version 1.0
   * 
   */
  public static String ALGY_DIR = "/u/qr/algy3/";

  /** Computed location of the Algernon Gui icons
   */
  public static String ICON_DIR = "/u/qr/algy3/gui/icons/";


   private static final char EMARK = '\0';/* Character denoting the end of
                                           * a text field or the end of a
                                           * command in the commands sent
                                           * by the client. */

   static String image_path = "file:/home/bubbie/images/gifs/";

   static WinOwner guiserver;             /* Acts as the WinOwner for each
                                           * created window. */
   static WinPosVector twin_list;         // List of text windows displayed
   static WinPosVector mwin_list;         // List of msg windows displayed 
   static Vector log_list;                // List of log windows displayed
   static Integer input_win_mutex;        /* Guards changes to the variable
                                           * input_win. */
   static InputWindow input_win;          /* Reference to the single input
                                           * window when its active. */
   
   private static PushbackInputStream in; // Standard input with filters

   public void ownedWindowRequestsKill(Window win)
   {
      if (win instanceof InputWindow)
         synchronized (input_win_mutex) {
            if (input_win == win) 
            {
               input_win = null;
               win.dispose();
            }
            /* else the main thread got a reset or kill-input-window
             * command before this point was reached so the main thread
             * has already called dispose() */
         }
      else if (win instanceof MsgWindow)
      {
         boolean was_there;
         synchronized (mwin_list) {
            was_there = mwin_list.removeElement(win);
         }
         if (was_there) win.dispose();
         /* else the main thread got a reset command before this point
          * was reached and dispose() has already been called. */
      }
      else if (win instanceof TextWindow) 
      {
         boolean was_there;
         synchronized (twin_list) {
            was_there = twin_list.removeElement(win);
         }
         if (was_there) win.dispose();
         /* else the main thread got a reset command before this point
          * was reached and killWindow has already been called. */
      }
      else
      {
         System.err.println("ERROR: Unknown window type in GuiServer.ownedWindowIsDying()");
         System.exit(1);
      }
   }
   
   public void ownedWindowOutput(Window win, String text)
   {
      synchronized (input_win_mutex) {
         if (input_win == win)
            System.out.println(
               "(" + ((InputWindow)win).getID() + " " + text + ")");
      }
   }
   
   private static char readChar() throws IOException
   {
      // Busy wait until data is available.  Necessary because a
      // blocking read blocks the entire process not just the thread
      // doing the reading.
      while (in.available() == 0) 
      {
         try { Thread.sleep(1000L); }
         catch (InterruptedException e) {}
      }
      int i = in.read();
      if (i == -1) throw new EOFException();
      return (char)i;
   }

   private static void skipSpace() throws BadCmdException, IOException
   {
      char c = readChar();
      if (c != ' ') 
      {
         in.unread((int)c);
         throw new BadCmdException("skipSpace");
      }
   }
   
   private static String readWord() throws IOException
   {
      StringBuffer buff = new StringBuffer();
      char c = readChar();
      while (c != ' ' && c != EMARK)
      {
         buff.append(c);
         c = readChar();
      }
      in.unread((int)c);
      return buff.toString();
   }

   private static int readInt() throws BadCmdException, IOException
   {
      try {
         return Integer.parseInt(readWord());
      }
      catch (NumberFormatException e) {
         throw new BadCmdException("readInt");
      }
   }

   private static double readDouble() throws BadCmdException, IOException
   {
      try {
         return Double.valueOf(readWord()).doubleValue();
      }
      catch (NumberFormatException e) {
         throw new BadCmdException("readDouble");
      }
   }
   
   private static String readText() throws IOException
   {
      StringBuffer buff = new StringBuffer();
      char c;
      for (c = readChar(); c != EMARK; c = readChar())
         buff.append(c);
      return buff.toString();
   }

   private static void killMostWindows()
   {
      // Kill all the text windows
      synchronized (twin_list) {
         for (Enumeration e = twin_list.elements(); e.hasMoreElements();)
            ((TextWindow)e.nextElement()).dispose();
         twin_list.removeAllElements();
      }
      // Kill all the message windows
      synchronized (mwin_list) {
         for (Enumeration e = mwin_list.elements(); e.hasMoreElements();)
            ((MsgWindow)e.nextElement()).dispose();
         mwin_list.removeAllElements();
      }
      // Kill the input window 
      synchronized (input_win_mutex) {
         if (input_win != null) 
         {
            input_win.dispose();
            input_win = null;
         }
      }
   }
   
   private static void processReset() throws BadCmdException, IOException
   {
      char c = readChar();
      if (c != EMARK) throw new BadCmdException("processReset");

      killMostWindows();
   }

   private static void processTerminate() throws BadCmdException, IOException
   {
      char c = readChar();
      if (c != EMARK) throw new BadCmdException("processTerminate");

      killMostWindows();
      // Kill all the log windows
      synchronized (log_list) {
         for (Enumeration e = log_list.elements(); e.hasMoreElements(); )
            ((LogWindow)e.nextElement()).killWindow();
         log_list.removeAllElements();
      }
   }

   private static void processKillInputWindow() 
      throws BadCmdException, IOException
   {
      char c = readChar();
      if (c != EMARK) throw new BadCmdException("processKillInputWindow");

      synchronized (input_win_mutex) {
         if (input_win != null) 
         {
            input_win.dispose();
            input_win = null;
         }
      }
   }

   private static void processYorN_P() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      String prompt = readText();

      synchronized (input_win_mutex) {
         if (input_win == null)
         {
            input_win = new YorN_PWindow(ID, prompt, guiserver);
// No effect since input_win is a subclass of Dialog
//            WinPosition.centerWin(input_win);
         }
         /* Otherwise discard command because an input window
          * is already open. */
      }
   }

   private static void processReadNumber() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      String prompt = readText();
      
      synchronized (input_win_mutex) {
         if (input_win == null) 
         {
            input_win = 
               new TextInputWindow(ID, TextInputWindow.GET_NUM, prompt,
                                   guiserver);
// No effect since input_win is a subclass of Dialog
//            WinPosition.centerWin(input_win);
         }
         /* Otherwise discard command because an input window
          * is already open. */
      }
   }

   private static void processReadAtom() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      String prompt = readText();
      
      synchronized (input_win_mutex) {
         if (input_win == null)
         {
            input_win = 
               new TextInputWindow(ID, TextInputWindow.GET_ATOM, prompt,
                                   guiserver);
// No effect since input_win is a subclass of Dialog
//            WinPosition.centerWin(input_win);
         }
         /* Otherwise discard command because an input window
          * is already open. */
      }
   }

   private static void processReadString() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      String prompt = readText();
      
      synchronized (input_win_mutex) {
         if (input_win == null)
         {
            input_win = 
               new TextInputWindow(ID, TextInputWindow.GET_STRING, prompt,
                                   guiserver);
// No effect since input_win is a subclass of Dialog
//            WinPosition.centerWin(input_win);
         }
         /* Otherwise discard command because an input window
          * is already open. */
      }
   }

   private static void processReadList() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      String prompt = readText();
      
      synchronized (input_win_mutex) {
         if (input_win == null)
         {
            input_win = 
               new TextInputWindow(ID, TextInputWindow.GET_LIST, prompt,
                                   guiserver);
// No effect since input_win is a subclass of Dialog
//            WinPosition.centerWin(input_win);
         }
         /* Otherwise discard command because an input window
          * is already open. */
      }
   }

   private static void processReadNumberWithBounds() 
      throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      double min = readDouble();
      skipSpace();
      double max = readDouble();
      skipSpace();
      String prompt = readText();
      
      synchronized (input_win_mutex) {
         if (input_win == null)
         {
            input_win = 
               new TextInputWindow(ID, TextInputWindow.GET_NUM, prompt,
                                   guiserver, min, max);
// No effect since input_win is a subclass of Dialog
//            WinPosition.centerWin(input_win);
         }
         /* Otherwise discard command because an input window
          * is already open. */
      }
   }

   private static void processChoose(boolean multiple)
      throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      String prompt = readText();
      skipSpace();
      int num_choices = readInt();
      skipSpace();
      String[] labels = new String[num_choices];
      for (int i = 0; i < num_choices; ++i)
         labels[i] = readText();

      if (num_choices == 0) 
         System.out.println("(" + ID + " )");
      else
         synchronized (input_win_mutex) {
            if (input_win == null) 
            {
               int type = 
                  multiple ? SelectionWindow.MULTIPLE : SelectionWindow.SINGLE;
               input_win = new SelectionWindow(ID, type, prompt, labels,
                                               guiserver);
// No effect since input_win is a subclass of Dialog
//               WinPosition.centerWin(input_win);
            }
            /* Otherwise discard command because an input window
             * is already open. */
      }
   }

   private static void processShowMsg() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      int timeout = readInt();
      skipSpace();
      String msg = readText();
      
      synchronized (mwin_list) {
         mwin_list.addElement(new MsgWindow(ID, timeout, msg, guiserver));
      }
   }

   private static void processShowText() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      String title = readText();
      String text = readText();
      
      synchronized (twin_list) {
         twin_list.addElement(new TextWindow(ID, title, text, guiserver));
      }
   }

   // NOTE - In all of the log processing commands I synchronize on 
   //        log_list even though it isn't necessary since the main
   //        thread is the only one that manipulates log_list. But
   //        it makes the code more uniform w.r.t. the other functions
   //        and it might become necessary some day.
   private static void processOpenLog() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      String title = readText();

      synchronized (log_list) {
         log_list.addElement(new LogWindow(ID, title));
      }
   }

   private static void processClearLog() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      int log_ID = readInt();
      if (readChar() != EMARK)
         throw new BadCmdException("processClearLog()");

      synchronized (log_list) {
         LogWindow l;
         for (Enumeration e = log_list.elements(); e.hasMoreElements(); )
         {
            l = (LogWindow)e.nextElement();
            if (l.logID() == log_ID) { l.clear(); return; }
         }
         System.err.println("ERROR: clear-log - Bad log window ID");
//         throw new BadCmdException("processClearLog() - bad ID: " + log_ID);
      }
   }

   private static void processLog() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      int log_ID = readInt();
      skipSpace();
      String text = readText();

      synchronized (log_list) {
         LogWindow l;
         for (Enumeration e = log_list.elements(); e.hasMoreElements(); )
         {
            l = (LogWindow)e.nextElement();
            if (l.logID() == log_ID) { l.appendText(text); return; }
         }
         System.err.println("ERROR: log - Bad log window ID");
//         throw new BadCmdException("processLog() - bad ID: " + log_ID);
      }
   }

   private static void processCloseLog() throws BadCmdException, IOException
   {
      // Parse rest of command
      skipSpace();
      int ID = readInt();
      skipSpace();
      int log_ID = readInt();
      if (readChar() != EMARK)
         throw new BadCmdException("processCloseLog()");

      synchronized (log_list) {
         LogWindow l;
         for (Enumeration e = log_list.elements(); e.hasMoreElements(); )
         {
            l = (LogWindow)e.nextElement();
            if (l.logID() == log_ID) 
            {
               log_list.removeElement(l);
               l.killWindow();
               return;
            }
         }
         System.err.println("ERROR: close-log - Bad log window ID");
//         throw new BadCmdException("processLog() - bad ID: " + log_ID);
      }
   }
   
   private static void processCommands() throws BadCmdException, IOException
   {
      boolean done = false;
      String cmd;
      
      while (!done)
      {
         cmd = readWord();
         if      (cmd.equalsIgnoreCase("reset"))
            processReset();
         else if (cmd.equalsIgnoreCase("terminate"))
         {
            processTerminate(); done = true;
         }
         else if (cmd.equalsIgnoreCase("kill-input-window"))
            processKillInputWindow();
         else if (cmd.equalsIgnoreCase("y-or-n-p"))
            processYorN_P();
         else if (cmd.equalsIgnoreCase("read-number"))
            processReadNumber();
         else if (cmd.equalsIgnoreCase("read-atom"))
            processReadAtom();
         else if (cmd.equalsIgnoreCase("read-string"))
            processReadString();
         else if (cmd.equalsIgnoreCase("read-list"))
            processReadList();
         else if (cmd.equalsIgnoreCase("read-number-with-bounds"))
            processReadNumberWithBounds();
         else if (cmd.equalsIgnoreCase("choose"))
            processChoose(false);
         else if (cmd.equalsIgnoreCase("choose-multiple"))
            processChoose(true);
         else if (cmd.equalsIgnoreCase("show-msg"))
            processShowMsg();
         else if (cmd.equalsIgnoreCase("show-text"))
            processShowText();
         else if (cmd.equalsIgnoreCase("open-log"))
            processOpenLog();
         else if (cmd.equalsIgnoreCase("clear-log"))
            processClearLog();
         else if (cmd.equalsIgnoreCase("log"))
            processLog();
         else if (cmd.equalsIgnoreCase("close-log"))
            processCloseLog();
         else
            throw new BadCmdException("Bad command: " + cmd);
         System.out.flush();
      }
   }

   /**
    * Currently takes no command-line arguments.
    */
   public static void main(String argv[])
   {
      if (argv.length == 1)
      {
	ALGY_DIR = argv[0];
	ICON_DIR = ALGY_DIR + File.separator + "gui" + File.separator + "icons" + File.separator;
      }

      // Notify client that we are going!
      System.out.println(":OPEN");

      // Init static vars
      guiserver = new GuiServer();

      // The list of text windows will be positioned from the upper
      // left going down and to the right.
      twin_list = new WinPosVector();

      // The list of msg windows will be positioned from the mid-upper
      // left of the screen and go straight across to the right.
      Dimension scrn = Toolkit.getDefaultToolkit().getScreenSize();
      mwin_list = new WinPosVector(
         WinPosVector.UPPER_LEFT, WinPosVector.STANDARD_GAP, 0,
         scrn.width/4, scrn.height/5);

      log_list = new Vector();
      in =  new PushbackInputStream(new FileInputStream(FileDescriptor.in));
      input_win_mutex = new Integer(0);

      // Go!
      try { processCommands(); }
      catch(BadCmdException e) {}
      catch(IOException e) {}
      
      // Notify client that we are dying and exit.
      System.out.println(":QUIT");
      System.exit(0);
   }
}

/*****************************************************************************
 **                         Class BadCmdException                           **
 *****************************************************************************/
/**
 * The exception class <TT>BadCmdException</TT> is only used internally
 * by the <TT>GuiServer</TT> class.
 */
class BadCmdException extends Exception
{
   public BadCmdException() { super(); }
   public BadCmdException(String s) { super(s); }
}
