;;;; -*- Mode:Lisp; Package:ALGY-IO; Syntax:COMMON-LISP; Base:10 -*-

(in-package :ALGY-IO)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1997 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

;;; ********************************************************************
;;;
;;; io-gui.lisp   -  Gui versions of the Algernon I/O routines.
;;;
;;; Mike Hewett, Spencer Bishop      16 January 1997
;;;
;;; These routines are called from "io.lisp" when the GUI is active.
;;; The user-level routines are documented in the file "design.text".
;;;
;;;
;;; ********************************************************************

(defparameter *NULL*         (code-char 0))

(defparameter *NULL-STRING*  (string *NULL*))

(defparameter *last-message-id* 0)

;;; UTILITY ROUTINES


(defun get-new-message-id ()

  (incf *last-message-id*))



(defun gui-send-receive (message-type message-body)

  (let ((message-id  (get-new-message-id)))
    
    (gui-send (format nil "~(~A~) ~D ~A"
			      message-type message-id message-body))
    (gui-receive message-id)
    )
  )

(defun gui-send (message)

  (USER::external-raw-output USER::*GUI-CONNECTION* message)
  )

(defun gui-receive (message-id)

  ;; We have to check the message id because it is possible that
  ;; the user aborted an earlier run and left an input window
  ;; hanging open on the server, which might send the value later.

  ;; I don't think it's possible to reeive a message ID *greater*
  ;; than the one expected...

  (let ((message NIL))
    ;; Receive something from USER::*GUI-SERVER*
    (setq message (USER::external-read USER::*GUI-CONNECTION*))

    (loop
      (when (or (eq message :QUIT)
		(eq (car message) message-id))
	(return)
	)

      ;; Comment this out when done testing
;;      (when (null (car message))
;;	(return)
;;	)
      (format t "~%GUI Manager discarding message: ~S" message)

      ;; Receive something from USER::*GUI-SERVER*
      (setq message (USER::external-read USER::*GUI-CONNECTION*))
      )

    ;; Return the value, if possible
    (if (consp message)
	(second message)
      ;;else
      ;; Probably we should do something here like shut down the connection.
        NIL                  ;; :QUIT
	)
    )
  )



;;; ----------  INPUT  -------------------------

(defun algy-y-or-n-p-gui (prompt)

  (eq :YES
      (gui-send-receive 'y-or-n-p
			(concatenate 'string prompt *NULL-STRING*)))
  )


(defun algy-read-number-gui (prompt)

  (gui-send-receive 'read-number
		    (concatenate 'string prompt *NULL-STRING*))
  )


(defun algy-read-number-with-bounds-gui (min max prompt)

  (gui-send-receive 'read-number-with-bounds
		    (format nil "~D ~D ~A~C" min max prompt *NULL*))
  )


(defun algy-read-atom-gui (prompt)

  (gui-send-receive 'read-atom
		    (concatenate 'string prompt *NULL-STRING*))
  )

(defun algy-read-string-gui (prompt)

  (gui-send-receive 'read-string
		    (concatenate 'string prompt *NULL-STRING*))
  )


(defun algy-read-list-gui (prompt)

  (gui-send-receive 'read-list
		    (concatenate 'string prompt *NULL-STRING*))
  )


;; ---- CHOOSE and CHOOSE-MULTIPLE

  ;; Choices has already been filtered so that it is
  ;; a list of entries of the form (value . "label").

(defun algy-choose-gui-sub (choices)
  (mapcan #'(lambda (x) (list (cdr x) *NULL*)) choices))


(defun algy-choose-gui (choices prompt)

  (let ((selection
	 (gui-send-receive 'choose
			   (format nil "~A~C ~D ~{~A~C~}"
				   prompt *NULL* (length choices) 
				   (algy-choose-gui-sub choices)))))

    (car (nth (1- selection) choices))
    )
  )


(defun algy-choose-multiple-gui (choices prompt)


  (let ((selections
	 (gui-send-receive 'choose-multiple
			   (format nil "~A~C ~D ~{~A~C~}"
				   prompt *NULL* (length choices) 
				   (algy-choose-gui-sub choices)))))

    (loop for choice in choices
	as  i from 1
	when (member i selections :test #'=)
	collect (car choice)
	)
    )
  )


;;; ----------  OUTPUT  -------------------------


(defun algy-show-frame-gui (frame-name)

  (algy-show-text-gui (format nil "The ~A frame" frame-name)
		      (with-output-to-string  (xx)
			(sfs::kb-print-no-rules frame-name xx))
		      )
  )


(defun algy-show-message-gui (timeout prompt)

  (gui-send (format nil "show-msg ~D ~D ~A~C"
		    (get-new-message-id) timeout prompt *NULL*))
  )


(defun algy-show-text-gui (title body)

  (gui-send (format nil "show-text ~D ~A~C~A~C"
		    (get-new-message-id) title *NULL* body *NULL*))
  )



;;; ----- the log window

(defun algy-open-log-gui (title)
  "Opens a new log window to which you can incrementally add text
using the algy-log command.  Returns the log window ID to be
used in commands that manipulate the new log window."

  (gui-send-receive 'OPEN-LOG (format nil "~A~C" title *NULL*))
  )


(defun algy-clear-log-gui (log-window-id)
  "Clears log window designated by log-id."

  (gui-send (format nil "clear-log ~D ~A~C"
		    (get-new-message-id) log-window-id *NULL*))
  )


(defun algy-log-gui (log-window-id output)
  "Appends the formatted text string to the log window
identified by log-id."

  (gui-send (format nil "log ~D ~A ~A~C"
		    (get-new-message-id) log-window-id output *NULL*))
  )


(defun algy-close-log-gui (log-window-id)

  (gui-send (format nil "close-log ~D ~A~C"
		    (get-new-message-id) log-window-id *NULL*))
  )



;;; ----------  CONTROL  -------------------------


(defun algy-reset-gui ()

  (gui-send (format nil "reset~C" *NULL*))
  )


(defun algy-close-gui ()

  (USER::close-external-program USER::*GUI-CONNECTION*
			  (format nil "terminate~C" *NULL*))
  (sys::os-wait)
  )

