;;;; -*- Mode:Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

(in-package :user)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1997 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

;;; ********************************************************************
;;;
;;; io-main.lisp   -  An I/O library for Algernon
;;;
;;; Mike Hewett, Spencer Bishop      16 January 1997
;;;
;;; This is a somewhat platform-independent I/O library for Algernon.
;;; It allows the user to use I/O routines that will be GUI-based
;;; if possible, but will automatically use the LISP text window
;;; if the GUI is not available.
;;;
;;; There is a corresponding set of GUI routines written in Java
;;; that comprise a gui server.  This file contacts the server and
;;; requests some sort of interaction with the user.  The GUI
;;; handles the user interaction and returns a result to the library,
;;; which will return it to the caller.
;;;
;;; The user-level routines are documented in the file "design.text".
;;;
;;;
;;; ********************************************************************


;; Load the external connection manager
(load "/u/qr/algy3/gui/lib/lisp-to-unix/lisp-to-unix")
(load (concatenate 'string @algy-gui-path "lib/lisp-to-unix/lisp-to-unix"))

(defparameter *GUI-SERVER* NIL
  "The GUI Server object, which the gui-based routines write to.")


;;; Locations of Java library and the gui program
(defparameter *IO-JAVA-CLASSPATH*
    (concatenate 'string @algy-gui-path ":" *JAVA-CLASSPATH*))

(defparameter *GUI-PROGRAM* "ioserver.GuiServer")

(defparameter *GUI-CONNECTION*           NIL)
(defparameter *IO-OUTPUT-COMMAND-COUNT*    0)
(defparameter *IO-INPUT-COMMAND-COUNT*     0)



;;; ----------  CONTROL  -------------------------

(defun algy-io (command)
  "COMMAND is one of :OPEN, :CLOSE, :RESET, :STATUS"

  ;;; Perform some operation on the server.

  (case command
    (:OPEN     ;; Start the remote program
     (setq *GUI-CONNECTION*
       (open-external-program
        (format nil "~Aioserver~Ahack.pl ~A ~A"
                @algy-gui-path *PATH-TERMINATOR* *IO-JAVA-CLASSPATH* @algy-path)
;;;	(format nil "/p/lib/java -classpath ~A -DALGY.HOME=~A ~A"
;;;		*IO-JAVA-CLASSPATH*
;;; 	        @algy-path
;;;		*GUI-PROGRAM*)
	:IO))
     (format *trace-output*
	     "~%Waiting for connection to GUI server to open...")

     ;; We catch the EOF error below in case the server didn't
     ;; open for some reason (file not found, server crash, ...)
     ;; If so, we don't want to abort.  Instead we just use
     ;; the text version of the I/O library.

     (let ((response  NIL))
       (loop
	 (setq response
	   (handler-case (external-read *GUI-CONNECTION*)
	     (END-OF-FILE ()     
	       (format *error-output*
		       "~%;; *** Unable to open GUI server.  Will use text I/O instead.~%")
	       (sys::os-wait)
	       (setq *GUI-CONNECTION* NIL)
	       (return))
	     ))

	 (when (eq :OPEN response)
	   (format *trace-output* "done")
	   (return)))
       )

     (setq *IO-INPUT-COMMAND-COUNT*  0)
     (setq *IO-OUTPUT-COMMAND-COUNT* 0)
     )

    (:CLOSE   (when *GUI-CONNECTION*
		(algy-gui-close)
		(setq *GUI-CONNECTION* NIL)
		))

    (:RESET   (algy-gui-reset))

    (:STATUS  (algy-status))
    )
  )


(defun gui-active-p ()

#+ALLEGRO
  (and *GUI-CONNECTION*
       (open-stream-p (external-stream-output-stream *GUI-CONNECTION*)))

#-ALLEGRO 
  NIL
  )


;;; ----------  INPUT  -------------------------

(defun algy-y-or-n-p (format-string &REST args)

  (incf *IO-OUTPUT-COMMAND-COUNT*)
  (incf *IO-INPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-y-or-n-p-gui
       (apply #'format nil format-string args))
      (ALGY-IO::algy-y-or-n-p-text
       (apply #'format nil format-string args))
      )
  )


(defun algy-read-number (format-string &REST args)

  (incf *IO-OUTPUT-COMMAND-COUNT*)
  (incf *IO-INPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-read-number-gui
       (apply #'format nil format-string args))
      (ALGY-IO::algy-read-number-text
       (apply #'format nil format-string args))
      )
  )


(defun algy-read-number-with-bounds (min max format-string &REST args)

  (incf *IO-OUTPUT-COMMAND-COUNT*)
  (incf *IO-INPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-read-number-with-bounds-gui min max
       (apply #'format nil format-string args))
      (ALGY-IO::algy-read-number-with-bounds-text min max
       (apply #'format nil format-string args))
      )
  )


(defun algy-read-atom (format-string &REST args)

  (incf *IO-OUTPUT-COMMAND-COUNT*)
  (incf *IO-INPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-read-atom-gui
       (apply #'format nil format-string args))
      (ALGY-IO::algy-read-atom-text
       (apply #'format nil format-string args))
      )
  )


(defun algy-read-string (format-string &REST args)

  (incf *IO-OUTPUT-COMMAND-COUNT*)
  (incf *IO-INPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-read-string-gui
       (apply #'format nil format-string args))
      (ALGY-IO::algy-read-string-text
       (apply #'format nil format-string args))
      )
  )


(defun algy-read-list (format-string &REST args)

  (incf *IO-OUTPUT-COMMAND-COUNT*)
  (incf *IO-INPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-read-list-gui
       (apply #'format nil format-string args))
      (ALGY-IO::algy-read-list-text
       (apply #'format nil format-string args))
      )
  )


(defun choose-preprocess (choices)

  ;; Each element of 'choices' can either be a single
  ;; value, or a list of (value . label).  We want to
  ;; put it into a canonical (value . label) format
  ;; before calling the lower-level functions.

  (mapcar #'(lambda (x)
	      (if (consp x)
		  x
		;;else
		  (cons x x)))
	  choices)
  )

(defun algy-choose (choices format-string &REST args)

  (incf *IO-OUTPUT-COMMAND-COUNT*)
  (incf *IO-INPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-choose-gui (choose-preprocess choices)
       (apply #'format nil format-string args))
      (ALGY-IO::algy-choose-text (choose-preprocess choices)
       (apply #'format nil format-string args))
      )
  )


(defun algy-choose-multiple (choices format-string &REST args)

  (incf *IO-OUTPUT-COMMAND-COUNT*)
  (incf *IO-INPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-choose-multiple-gui (choose-preprocess choices)
       (apply #'format nil format-string args))
      (ALGY-IO::algy-choose-multiple-text (choose-preprocess choices)
       (apply #'format nil format-string args))
      )
  )


;;; ----------  OUTPUT  -------------------------


(defun algy-show-frame (frame-name)

  (incf *IO-OUTPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-show-frame-gui  frame-name)
      (ALGY-IO::algy-show-frame-text frame-name)
      )
  )


(defun algy-show-message (timeout format-string &REST args)

  (incf *IO-OUTPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-show-message-gui timeout
       (apply #'format nil format-string args))
      (ALGY-IO::algy-show-message-text timeout
       (apply #'format nil format-string args))
      )
  )


(defun algy-show-text (title format-string &REST args)

  (incf *IO-OUTPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-show-text-gui title
       (apply #'format nil format-string args))
      (ALGY-IO::algy-show-text-text title
       (apply #'format nil format-string args))
      )
  )

;;; ----- the log window

(defun algy-open-log (title)
  "Opens a new log window to which you can incrementally add text
using the algy-log command.  Returns the log window ID to be
used in commands that manipulate the new log window."


  (incf *IO-OUTPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-open-log-gui  title)
      (ALGY-IO::algy-open-log-text title)
      )
  )


(defun algy-clear-log (log-window-id)
  "Clears log window designated by log-id."

  (incf *IO-OUTPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-clear-log-gui  log-window-id)
      (ALGY-IO::algy-clear-log-text log-window-id)
      )
  )


(defun algy-log (log-window-id format-string &REST args)
  "Appends the formatted text string to the log window
identified by log-id."

  (incf *IO-OUTPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-log-gui  log-window-id
       (apply #'format nil format-string args))
      (ALGY-IO::algy-log-text log-window-id
       (apply #'format nil format-string args))
      )
  )


(defun algy-close-log (log-window-id)
  "Closes the log window designated by log-window-id."

  (incf *IO-OUTPUT-COMMAND-COUNT*)

  (if (gui-active-p)
      (ALGY-IO::algy-close-log-gui  log-window-id)
      (ALGY-IO::algy-close-log-text log-window-id)
      )
  )




;;; ----------  CONTROL  -------------------------


(defun algy-gui-reset ()

  ;; Don't do anything if the connection is not open.
  (if (gui-active-p)
      (ALGY-IO::algy-reset-gui)
      )
  )


(defun algy-gui-close ()

  ;; Don't do anything if the connection is not open.
  (when (gui-active-p)
    (ALGY-IO::algy-close-gui)
    )
  )


(defun algy-status ()

  (format *standard-output* "~2%I/O Library Status")
  (format *standard-output* "~%------------------")
  (format *standard-output*
	  "~%Using ~:[text output~;GUI server~]."
	  (gui-active-p))
  (format *standard-output* "~%~D output commands processed."
	  *io-output-command-count*)
  (format *standard-output* "~%~D input commands processed."
	  *io-input-command-count*)
  (terpri *standard-output*)
  )


