;;;; -*- Mode:Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

(in-package :ALGY-IO)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1997 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

;;; ********************************************************************
;;;
;;; io-text.lisp   -  Text versions of the Algernon I/O routines.
;;;
;;; Mike Hewett, Spencer Bishop      16 January 1997
;;;
;;; These routines are called from "io.lisp" when the GUI is not active.
;;; The user-level routines are documented in the file "design.text".
;;;
;;;
;;; ********************************************************************


;;; ----------  INPUT  -------------------------

(defun algy-y-or-n-p-text (prompt)

  (y-or-n-p prompt)
  )


(defun algy-read-number-text (prompt)

  (terpri *standard-output*)
  (princ prompt)

  ;; Read until we get a number
  (let ((thing  NIL))
    (loop
      (setq thing (read))

      (when (numberp thing)
	(return))
      
      (format *standard-output* "~%PLEASE ENTER A NUMBER!")
      (terpri *standard-output*)
      (princ prompt)
      )
    
    thing
    )
  )


(defun algy-read-number-with-bounds-text (min max prompt)

  (format *standard-output* "~%Enter a number from ~D to ~D" min max)
  (terpri *standard-output*)
  (princ prompt)

  ;; Read until we get a number
  (let ((thing  NIL))
    (loop
      (setq thing (read))

      (when (and (numberp thing)
		 (<= min thing max))
	(return))
      
      (format *standard-output*
	      "~%PLEASE ENTER A NUMBER from ~D to ~D!" min max)
      (terpri *standard-output*)
      (princ prompt)
      )
    
    thing
    )
  )



(defun algy-read-atom-text (prompt)

  (terpri *standard-output*)
  (princ prompt)

  ;; Read until we get an atom
  (let ((thing  NIL))
    (loop
      (setq thing (read))

      (when (atom thing)
	(return))
      
      (format *standard-output* "~%PLEASE ENTER AN ATOM!")
      (terpri *standard-output*)
      (princ prompt)
      )
    
    thing
    )
  )


(defun algy-read-string-text (prompt)

  (terpri *standard-output*)
  (princ prompt)

  ;; Read until we get an atom
  (let ((thing  NIL))
    (loop
      (setq thing (read-line))

      (when (> (length thing) 0)
	(return))
      
      (format *standard-output* "~%PLEASE ENTER A NON-EMPTY RESPONSE!")
      (terpri *standard-output*)
      (princ prompt)
      )
    
    thing
    )
  )


(defun algy-read-list-text (prompt)

  ;; The user doesn't need to add the outer parentheses.

  (terpri *standard-output*)
  (princ prompt)

  ;; Read until we get a list
  (let ((thing  NIL))
    (loop
      (setq thing (read-line))

      (when (not (null thing))
	(return))
      
      (format *standard-output* "~%PLEASE ENTER A LIST!")
      (terpri *standard-output*)
      (princ prompt)
      )
    
    (read-from-string (concatenate 'string "(" thing ")"))
    )
  )


  ;; Choices has already been filtered so that it is
  ;; a list of entries of the form (value . "label").

(defun algy-choose-text (choices prompt)

  (let ((num-items  (length choices))
	(selection  0))

    (terpri *standard-output*)
    (setq selection 1)
    (dolist (choice choices)
      (format *standard-output* "~% ~2d - ~A" selection (cdr choice))
      (incf selection)
      )

    (terpri *standard-output*)
    (princ prompt)

    ;; Read until we get a valid selection.
    (loop
      (setq selection (read))

      (when (and (integerp selection)
		 (<= 1 selection num-items))
	(return))
      
      (format *standard-output* "~%Please select an item number: [1 - ~D]"
	      num-items)
      (terpri *standard-output*)
      (princ prompt)
      )

    (car (nth (1- selection) choices))
    )
  )


(defun algy-choose-multiple-text (choices prompt)

  (let ((num-items  (length choices))
	(response   NIL)
	(selections NIL)
	(index      1))

    (terpri *standard-output*)
    (dolist (choice choices)
      (format *standard-output* "~% ~2d - ~A" index (cdr choice))
      (incf index)
      )

    (terpri *standard-output*)
    (princ prompt)

    ;; Read until we get a valid selection.
    (loop
      (setq response   (read-line))
      (setq selections
	(read-from-string (concatenate 'string "(" response ")")))

      (when (every #'(lambda (num)
		       (and (integerp num)
			    (<= 1 num num-items)))
		   selections)
	(return))
      
      (format *standard-output* "~%Please select one or more items: [1 - ~D]"
	      num-items)
      (terpri *standard-output*)
      (princ prompt)
      )

    ;; Accumulate the answers
    (loop for choice in choices
	  as  i from 1
	when (member i selections :test #'=)
	collect (car choice)
	)
    )
  )



;;; ----------  OUTPUT  -------------------------


(defun algy-show-frame-text (frame-name)

  (user::silently
   (user::tell `((:show ,frame-name))))
  )


(defun algy-show-message-text (timeout prompt)

  (declare (ignore timeout))

  (format *standard-output* "~%~A~%" prompt)
  )


(defun algy-show-text-text (title prompt)
  
  (format *standard-output* "~%~A~%~A~%" title prompt)
  )



;;; ----- the log window

(defun algy-open-log-text (title)
  "Opens a new log window to which you can incrementally add text
using the algy-log command.  Returns the log window ID to be
used in commands that manipulate the new log window."

  ;; Since there are no windows, we log to *standard-output*
  (format *standard-output* "~%Logging ~A" title)
  )


(defun algy-clear-log-text (log-window-id)
  "Clears log window designated by log-id."

  (declare (ignore log-window-id))
  ;; Nothing we can do here...
  )


(defun algy-log-text (log-window-id output)
  "Appends the formatted text string to the log window
identified by log-id."

  (declare (ignore log-window-id))

  (princ output *standard-output*)
  )


(defun algy-close-log-text (log-window-id)

  (declare (ignore log-window-id))

  ;; Nothing we can do here.
  )
