/*****************************************************************************
 **                           Class InputWindow                             **
 *****************************************************************************/
//    File: InputWindow.java
// Descrip: Superclass to several types of standalone windows that accept
//          user input.  Note the dependency on the WinOwner interface.
//
// History: Fri Feb 14 15:44:59 1997 created by Spencer Bishop
// $Id$
/*****************************************************************************/

package lib.display;

import java.awt.*;

public abstract class InputWindow extends Dialog
{
   static final Font input_font = new Font("TimesRoman", Font.BOLD, 14);

   /* Because InputWindows are Dialog objects they have to have
    * a parent frame when they are created.  bogus_parent
    * fulfills this role. */
   private static Frame bogus_parent;

   protected int ID;
   protected WinOwner owner;
   protected String user_input;

   static
   {
      bogus_parent = new Frame();
      bogus_parent.setLayout(new FlowLayout());
      bogus_parent.pack();
      bogus_parent.show();
      WinPosition.centerWin(bogus_parent);
      bogus_parent.hide();
   }

   public InputWindow(int ID, String title, WinOwner owner,
                      Frame parent, boolean modal) 
   { 
      super((parent == null) ? bogus_parent : parent, title, modal);
      this.ID = ID;
      this.owner = owner;
      user_input = null;
   }

   /** Returns InputWindow's ID specified during construction. */
   public int getID() { return ID; }

   /** Call this function on subclass instances to retrieve input from
    * modal input windows.  This function shouldn't be called on non-modal
    * windows because it makes no sense.  In that situation, user input
    * is returned through the WinOwner specified during object construction.
    */
   public String getUserInput()
   {
      // Report an error if called on a non-modal window
      if (!isModal())
      {
         System.err.println("In " + getClass().getName() +
                            "::getUserInput() - called method on non-modal window.");
         return null;
      }
      WinPosition.showWin(this);
      return user_input;
   }

   /** Checks if the argument string contains a number with possibly
    * leading and trailing whitespace.  I had to write this because
    * the built in routines in java.lang.Double for parsing numbers
    * only search for a prefix of the argument string that is a valid
    * number.
    */
   public static boolean isaNumber(String s)
   {
      int state = 0, i = 0;
      boolean okay = true;
      char c;
      
      while (okay && i < s.length())
      {
         c = s.charAt(i); ++i;
         switch (state) 
         {
         case 0:
            if      (c == ' ' || c == '\t') state = 0;
            else if (c == '+' || c == '-')  state = 1;
            else if (c >= '0' && c <= '9')  state = 2;
            else                            okay = false;
            break;
         case 1:
            if (c >= '0' && c <= '9') state = 2;
            else                      okay = false;
            break;
         case 2:
            if      (c >= '0' && c <= '9')  state = 2;
            else if (c == ' ' || c == '\t') state = 3;
            else if (c == '.')              state = 4;
            else if (c == 'e' || c == 'E')  state = 6;
            else                            okay = false;
            break;
         case 3:
            if (c == ' ' || c == '\t') state = 3;
            else                       okay = false;
            break;
         case 4:
            if (c >= '0' && c <= '9') state = 5;
            else                      okay = false;
            break;
         case 5:
            if      (c >= '0' && c <= '9')  state = 5;
            else if (c == ' ' || c == '\t') state = 3;
            else if (c == 'e' || c == 'E')  state = 6;
            else                            okay = false;
            break;
         case 6:
            if      (c == '+' || c == '-') state = 7;
            else if (c >= '0' && c <= '9') state = 8;
            else                           okay = false;
            break;
         case 7:
            if (c >= '0' && c <= '9') state = 8;
            else                      okay = false;
            break;
         case 8:
            if      (c >= '0' && c <= '9')  state = 8;
            else if (c == ' ' || c == '\t') state = 3;
            else                            okay = false;
            break;
         }
      }
      return okay && 
         (state == 2 || state == 3 || state == 5 || state == 8);
   }
}
