/*****************************************************************************
 **                      Multi-line Text Label Class                        **
 *****************************************************************************/
//    File: MultiLabel.java
// Descrip: Implements a GUI component that can display a text label that
//          has multiple lines.  There are also some minimal features for
//          customizing how the text appears.
//
// History: Sat Jan 18 23:51:54 1997 created by Spencer Bishop
// $Id$
/*****************************************************************************/

package lib.display;

import java.awt.*;
import java.util.*;

public class MultiLabel extends Canvas
{
   private int alignment;
   private Point[] points;
   private Vector lines;
   private int xgap, ygap;
   private boolean layed_out;

   public final static int CENTER = 0;
   public final static int LEFT = 1;
   public final static int RIGHT = 2;

   static Vector splitPrompt(String prompt)
   {
      int f, l;
      Vector v = new Vector();

      f = 0;
      while (f < prompt.length()) 
      {
         for (l = f; l < prompt.length() && prompt.charAt(l) != '\n'; ++l) ;
         v.addElement(prompt.substring(f, l));
         f = l + 1;
      }
      return v;
   }

   public MultiLabel() { this("", 10, 10, LEFT); }
   
   public MultiLabel(String label) { this(label, 10, 10, LEFT); }
      
   public MultiLabel(String label, int alignment)
   {
      this(label, 10, 10, alignment);
   }
   
   public MultiLabel(String label, int xgap, int ygap, int alignment)
   {
      lines = splitPrompt(label);
      this.xgap = xgap;
      this.ygap = ygap;
      this.alignment = alignment;
      layed_out = false;
   }
   
   public void setAlignment(int alignment) { }

   public void setText(String label)
   {
      lines = splitPrompt(label);
      layed_out = false;
      if (getParent() != null) getParent().layout();
   }
   
   private void computePositions()
   {
//      System.out.println("I'm in computePositions()");
      
      // Compute the size of the MultiLabel and the positions
      // of each line of text based on the current font.
      int width, height;
      Font f = getFont();
      FontMetrics fm = getFontMetrics(f);
      points = new Point[lines.size()];

      // First loop through the lines determining the maximum width
      int max_width = 0;
      for (int i = 0; i < lines.size(); ++i)
      {
         points[i] = new Point(0, 0);
         points[i].x = fm.stringWidth((String)lines.elementAt(i));
         if (points[i].x > max_width) max_width = points[i].x;
      }
      width = max_width + 2*xgap;
      
      // Now that we know the maximum width, we can determine the
      // starting position of each line of text.
      int curr_y = fm.getAscent() + ygap;
      int ystep = fm.getHeight();
      for (int i = 0; i < lines.size(); ++i)
      {
         points[i].y = curr_y; curr_y += ystep;
         switch (alignment)
         {
         case LEFT: points[i].x = xgap; break;
         case CENTER: 
            points[i].x = (max_width - points[i].x)/2 + xgap;
            break;
         case RIGHT:
            points[i].x = max_width - points[i].x;
            break;
         }
      }
      height = lines.size() * ystep + 2*ygap;

      layed_out = true;
      resize(width, height);
   }

   public Dimension preferredSize()
   {
//      System.out.println("I'm in preferredSize()");
      if (!layed_out) computePositions();
      return super.preferredSize();
   }

   public Dimension minimumSize()
   {
//      System.out.println("I'm in minimumSize()");
      if (!layed_out) computePositions();
      return super.minimumSize();
   }

   public void paint(Graphics g)
   {
//      System.out.println("I'm in paint()");
      if (!layed_out) 
         computePositions();
      for (int i = 0; i < lines.size(); ++i)
         g.drawString((String)lines.elementAt(i), points[i].x, points[i].y);
   }
}
