/*****************************************************************************
 **                         Class SelectionWindow                           **
 *****************************************************************************/
//    File: SelectionWindow.java
// Descrip: Displays a window that allows the user to make choices from a list.
//          The window can operate in two modes, in one mode the user can make
//          one or more selections in the other mode the user must make only
//          one selection.
//
// History: Fri Feb 14 15:51:50 1997 created by Spencer Bishop
// $Id$
/*****************************************************************************/

package lib.display;

import java.awt.*;
import java.io.File;  // For File.separator

public class SelectionWindow extends InputWindow
{
   /**
    * ICON_DIR contains the path to the icon directory,
    * e.g. <CODE>"/u/qr/algy3/gui/icons/"</CODE>.n
    * 
    * @see #TextInputWindow
    * @author  Micheal S. Hewett    hewett@cs.utexas.edu
    * @date    Wed Mar  5 19:06:19 1997
    * @version 1.0
    * 
    */
   public static String ICON_DIR = null;

   private ActiveButton ok_button;
   private CheckboxGroup cbg;
   private ActiveCheckbox[] cbxs;
   private int type;
   private Label e_msg;

   public final static int SINGLE = 0;
   public final static int MULTIPLE = 1;

   public SelectionWindow(int ID, int type, String prompt, String[] labels)
   {
      this(ID, type, prompt, labels, null, null, false);
   }

   public SelectionWindow(int ID, int type, String prompt, String[] labels,
                          WinOwner owner)
   {
      this(ID, type, prompt, labels, owner, null, false);
   }

   public SelectionWindow(int ID, int type, String prompt, String[] labels,
                          Frame parent, boolean modal)
   {
      this(ID, type, prompt, labels, null, parent, modal);
   }

   public SelectionWindow(int ID, int type, String prompt, String[] labels,
                          WinOwner owner, Frame parent, boolean modal)
   {
      super(ID, "Input Needed", owner, parent, modal);
      
      // Initialize the path to the icon directory, if necessary
      if (ICON_DIR == null)
      {
	String algyHomePath = System.getProperty("ALGY.HOME");
	if (algyHomePath == null)
	{
	  System.err.print("-DALGY.HOME not specified.  Using default path: ");
	  algyHomePath = "/u/qr/algy3/";
	  System.err.println(algyHomePath);
	}
	ICON_DIR = algyHomePath + "gui" + File.separator + "icons" + File.separator;
      }

      this.type = type;
      
      // Set up the window and display it
      Panel north = new Panel();
      north.setFont(InputWindow.input_font);
      north.setLayout(new BorderLayout());
      north.add("West", new ImageLabel(
         "file:" + ICON_DIR + File.separator + "check.gif"));
      north.add("Center", new MultiLabel(prompt));
      e_msg = new Label("", Label.CENTER);
      e_msg.setForeground(Color.red);
      north.add("South", e_msg);
      add("North", north);

      if (type == SINGLE) cbg = new CheckboxGroup();
      else                cbg = null;
      Panel center = new Panel();
      cbxs = new ActiveCheckbox[labels.length];
      if (labels.length > 5) 
      {
         int rows = labels.length / 2 + labels.length % 2;
         center.setLayout(new GridLayout(rows, 2));
         for (int i = 0; i < rows; ++i)
         {
            cbxs[i] = new ActiveCheckbox(labels[i], cbg, false);
            center.add(cbxs[i]);
            if (i + rows < labels.length) 
            {
               cbxs[i + rows] = 
                  new ActiveCheckbox(labels[i + rows], cbg, false);
               center.add(cbxs[i + rows]);
            }
         }
      }
      else 
      {
         center.setLayout(new GridLayout(labels.length, 1));
         for (int i = 0; i < labels.length; ++i)
         {
            cbxs[i] = new ActiveCheckbox(labels[i], cbg, false);
            center.add(cbxs[i]);
         }
      }
      if (type == SINGLE) cbxs[0].setState(true);
      // else - No default for multi-choice selection windows
      add("Center", center);
      
      ok_button = new ActiveButton("Ok");
      add("South", ok_button);

      if (!isModal()) {
         WinPosition.showWin(this);
         cbxs[0].requestFocus();
      }
   }

   public boolean handleEvent(Event e)
   {
      if (e.id == Event.ACTION_EVENT && e.target == ok_button)
      {
         // First check that the user actually selected something
         boolean ok = false;
         for (int i = 0; i < cbxs.length && !ok; ++i)
            ok = cbxs[i].getState();
         if (!ok)
         {
            if (type == SINGLE) e_msg.setText("YOU MUST MAKE A CHOICE");
            else                e_msg.setText("MAKE AT LEAST ONE CHOICE");
            return true;
         }

         // User input okay, send the answer and kill window
         String values = "";
         for (int i = 0; i < cbxs.length; ++i) 
            if (cbxs[i].getState()) values += " " + (i + 1);
         if (type == MULTIPLE)
            values = "(" + values + ")";
         if (owner != null)
         { 
            owner.ownedWindowOutput(this, values);
            owner.ownedWindowRequestsKill(this);
         }
         else
            dispose();
         user_input = values;
         return true;
      }
      else
         return super.handleEvent(e);
   }
}

