/*****************************************************************************
 **                         Class TextInputWindow                           **
 *****************************************************************************/
//    File: TextInputWindow.java
// Descrip: Displays a window allow the user to enter one of several classes
//          of text input.  Current possibilities are numbers, atoms, a one
//          line string and a Lisp list.
//
// History: Fri Feb 14 15:49:21 1997 created by Spencer Bishop
// $Id$
/*****************************************************************************/

package lib.display;

import java.awt.*;
import java.io.File;  // For File.separator


public class TextInputWindow extends InputWindow
{
   /**
    * ICON_DIR contains the path to the icon directory,
    * e.g. <CODE>"/u/qr/algy3/gui/icons/"</CODE>.n
    * 
    * @see #TextInputWindow
    * @author  Micheal S. Hewett    hewett@cs.utexas.edu
    * @date    Wed Mar  5 19:06:19 1997
    * @version 1.0
    * 
    */
   public static String ICON_DIR = null;

   public final static int GET_NUM = 0;
   public final static int GET_ATOM = 1;
   public final static int GET_STRING = 2;
   public final static int GET_LIST = 3;

   private TextComponent text;
   private ActiveButton ok_button;
   private int get_type;
   private double min, max;
   private Label e_msg;
   private Color e_msg_fg;

   public TextInputWindow(int ID, int get_type, String prompt)
   {
      this(ID, get_type, prompt, null, null, false,
           -Double.MAX_VALUE, Double.MAX_VALUE);
   }

   public TextInputWindow(int ID, int get_type, String prompt, WinOwner owner)
   {
      this(ID, get_type, prompt, owner, null, false,
           -Double.MAX_VALUE, Double.MAX_VALUE);
   }

   public TextInputWindow(int ID, int get_type, String prompt, WinOwner owner,
                          double min, double max)
   {
      this(ID, get_type, prompt, owner, null, false, min, max);
   }

   public TextInputWindow(int ID, int get_type, String prompt, 
                          Frame parent, boolean modal)
   {
      this(ID, get_type, prompt, null, parent, modal,
           -Double.MAX_VALUE, Double.MAX_VALUE);
   }

   public TextInputWindow(int ID, int get_type, String prompt,
                          Frame parent, boolean modal,
                          double min, double max)
   {
      this(ID, get_type, prompt, null, parent, modal, min, max);
   }
   
   public TextInputWindow(int ID, int get_type, String prompt,
                          WinOwner owner, Frame parent, boolean modal,
                          double min, double max)
   {
      super(ID, "Input Needed", owner, parent, modal);

      // Initialize the path to the icon directory, if necessary
      if (ICON_DIR == null)
      {
	String algyHomePath = System.getProperty("ALGY.HOME");
	if (algyHomePath == null)
	{
	  System.err.print("-DALGY.HOME not specified.  Using default path: ");
	  algyHomePath = "/u/qr/algy3/";
	  System.err.println(algyHomePath);
	}
	ICON_DIR = algyHomePath + "gui" + File.separator + "icons" + File.separator;
      }


      this.get_type = get_type;
      this.min = min;
      this.max = max;

      // Set up the window and display it
      Panel panel = new Panel();
      panel.setFont(InputWindow.input_font);
      panel.setLayout(new BorderLayout());
      panel.add("West", new ImageLabel(
         "file:" + ICON_DIR + File.separator + "icon.ques.gif"));
      panel.add("Center", new MultiLabel(prompt));
      e_msg = new Label("", Label.CENTER);
      e_msg_fg = Color.red;
      panel.add("South", e_msg);

      if (get_type != GET_LIST)
      {
         add("Center", panel);
         text = new TextField(40);
         add("South", text);
      }
      else
      {
         add("North", panel);
         text = new TextArea(20, 40);
         add("Center", text);
         ok_button = new ActiveButton("Ok");
         add("South", ok_button);
      }

      if (!isModal()) 
      {
         WinPosition.showWin(this);
         text.requestFocus();
      }
   }

   public boolean handleEvent(Event e)
   {
      if (e.id == Event.ACTION_EVENT && e.target == text && 
          text instanceof TextField)
      {
         String ans = (String)e.arg;
         
         // Make sure answer is of the proper type, if not then
         // we need to notify the user and wait for input again.
         if (get_type == GET_NUM) 
         {
            Double d;
            if (!InputWindow.isaNumber(ans))
            {
               e_msg.setForeground(e_msg_fg);
               e_msg.setText("PLEASE ENTER A NUMBER");
               if (e_msg_fg == Color.red) e_msg_fg = Color.blue;
               else                       e_msg_fg = Color.red;
               return true;
            }
            try { d = Double.valueOf(ans); }
            catch (NumberFormatException ex) {
               e_msg.setForeground(e_msg_fg);
               e_msg.setText("PLEASE ENTER A NUMBER");
               if (e_msg_fg == Color.red) e_msg_fg = Color.blue;
               else                       e_msg_fg = Color.red;
               return true;
            }
            if (d.doubleValue() < min || d.doubleValue() > max) 
            {
               e_msg.setForeground(e_msg_fg);
               e_msg.setText("ACCEPTABLE RANGE: " + min + " - " + max);
               if (e_msg_fg == Color.red) e_msg_fg = Color.blue;
               else                       e_msg_fg = Color.red;
               return true;
            }
         }
         else if (get_type == GET_ATOM)
         {
         }

         // User input okay, send the answer and kill window
         if (get_type == GET_STRING)
            ans = "\"" + ans + "\"";
         if (owner != null)
         {
            owner.ownedWindowOutput(this, ans);
            owner.ownedWindowRequestsKill(this);
         }
         else 
            dispose();
         user_input = ans;
         return true;
      }
      else if (e.id == Event.ACTION_EVENT && e.target == ok_button)
      {
         String ans = text.getText();

         // Make sure answer is syntactically correct (as best as
         // we can!)
         if (get_type == GET_LIST)
         {
            int parens = 0, state = 1, ndx = 0;
            boolean done = false;
            char c;

            while (!done && ndx < ans.length())
            {
               c = ans.charAt(ndx); ++ndx;
               switch (state)
               {
               case 1:
                  if      (Character.isSpace(c)) state = 1;
                  else if (c == '(') { ++parens; state = 2; }
                  else                           done = true;
                  break;
               case 2:
                  if      (c == '(') { ++parens; state = 2; }
                  else if (c == ')')
                  {
                     --parens;
                     if (parens == 0) state = 3;
                     else             state = 2;
                  }
                  else if (c == '\'') state = 5;
                  else if (c == '"')  state = 6;
                  else                state = 2;
                  break;
               case 3:
                  if (Character.isSpace(c)) state = 3;
                  else { done = true; state = 4; }
                  break;
               case 4:
                  break;
               case 5:
                  if      (Character.isSpace(c)) state = 5;
                  else if (c != ')')             state = 2;
                  else                           done = true;
                  break;
               case 6:
                  if (c == '"') state = 2;
                  else          state = 6;
                  break;
               }
            }
            if (state != 3) 
            {
               e_msg.setForeground(e_msg_fg);
               e_msg.setText("SYNTAX ERROR, PLEASE CORRECT");
               if (e_msg_fg == Color.red) e_msg_fg = Color.blue;
               else                       e_msg_fg = Color.red;
               return true;
            }
         }

         // User input okay, send the answer and kill window
         if (owner != null)
         {
            owner.ownedWindowOutput(this, ans);
            owner.ownedWindowRequestsKill(this);
         }
         else
            dispose();
         user_input = ans;
         return true;
      }
      else
         return super.handleEvent(e);
   }
}
