/*****************************************************************************
 **                        Class WinPosEnumeration                          **
 *****************************************************************************/
//    File: WinPosEnumeration.java
// Descrip: An enumerator for WinPosVector objects.
//
// History: Wed Jan 15 22:17:16 1997 created by Spencer Bishop
// $Id$
/*****************************************************************************/

package lib.display;

import java.util.*;

/**
 * A wrapper around the <TT>Enumeration</TT> of the <TT>Vector</TT> field
 * of a <TT>WinPosVector</TT>.  Allows iteration over the windows being
 * tracked by the <TT>WinPosVector</TT> rather than the <TT>WinPosition</TT>
 * objects actually being stored.
 * <p>
 * 
 * <b>SCREWBALL REASONING!</b> It turns out that I couldn't use a built in
 * Java Enumeration of the underlying vector to implement this class because
 * enumerations of Vectors are sort of screwed up.  They are screwed up in the
 * sense that you can't do something like this:
 * <pre>
 * for (Enumeration e = v.elements(); e.hasMoreElements();) {
 *    Object tmp = (Whatever)e.nextElement();
 *    if (tmp.doSomething())
 *       v.removeElement(tmp);
 * }
 * </pre>
 * This ends up accessing and removing only every other element in the Vector
 * which is certainly not what it seems like it should do!
 * <p>
 * 
 * Given the way I ended up using the enumeration, I could have just stuck
 * with the underlying enumeration as my method of implementation.  But, since
 * it works, the screwball version will do.
 * @see WinPosVector
 */
public class WinPosEnumeration implements Enumeration
{
   private Vector v;
   private int next;
   private Object last;
   
   public WinPosEnumeration(Vector v) 
   {
      this.v = v;
      next = 0;
      last = null;
   }
   
   public boolean hasMoreElements() 
   {
      return (next < v.size()) &&
         (v.elementAt(next) != last || next + 1 < v.size());
   }
   
   public Object nextElement() 
   { 
      if (v.elementAt(next) == last) ++next;
      last = v.elementAt(next);
      return ((WinPosition)last).win;
   }
}
