/*****************************************************************************
 **                       Window Positioning Support                        **
 *****************************************************************************/
//    File: WinPosVector.java
// Descrip: Implements the class WinPosVector and two supporting classes
//          (WinPosition and WinPosEnumeration) which enable automatic 
//          placement of a list of windows in a stacked fashion on the
//          root window.
//
// History: Wed Jan 15 22:17:16 1997 created by Spencer Bishop
// $Id$
/*****************************************************************************/

package lib.display;

import java.awt.*;
import java.util.*;

/*****************************************************************************
 **                           Class WinPosition                             **
 *****************************************************************************/
/**
 * Used by the class <TT>WinPosVector</TT> to manage the placement
 * of a single window.
 * @see WinPosVector
 */
class WinPosition
{
   private final static int XTRA_WIDTH = 5; 
   private final static int XTRA_HEIGHT = 20;

   /** Minimum width of a window displayed by <tt>showWin()</tt>. */
   public final static int MIN_WIN_WIDTH = 200;
   /** Minimum height of a window displayed by <tt>showWin()</tt>. */
   public final static int MIN_WIN_HEIGHT = 100;

   Window win;   // Window whose position we are talking about
   int corner;   /* corner is one of LOWER_LEFT, etc telling which */
   int x, y;     /* corner of the window should be put at (x, y) in
                  * in the root window. */
   int old_x, old_y;
   boolean needs_update;

   /** Create a WinPosition object to track a displayed window.
    * @param <b>win</b> The window being managed.
    * @param <b>corner</b> 
    * The corner of the window by which positioning is specified.  The
    * value is one of <TT>WinPosVector.UPPER_LEFT</TT>, etc.
    * @param <b>x</b> The x coordinate of the window's position.
    * @param <b>y</b> The y coordinate of the window's position.
    */
   public WinPosition(Window win, int corner, int x, int y)
   {
      this.win = win; 
      this.corner = corner; 
      needs_update = true;
      this.x = x;
      this.y = y;
   }
   
   /** Places argument window in the center of the root window.
    * @param win The window to be placed.
    */
   public static void centerWin(Window win)
   {
      Dimension dim = win.getToolkit().getScreenSize();
      Dimension siz = win.size();

      win.move(dim.width/2 - siz.width/2, dim.height/2 - siz.height/2);
   }

   /** Packs and shows argument window with one addition.  The method
    * makes sure the x and y dimensions of the window are at least
    * <tt>MIN_WIN_WIDTH</tt> and <tt>MIN_WIN_HEIGHT</tt>, respectively,
    * before showing.
    */
   public static void showWin(Window win)
   {
      win.pack();
      Dimension d = win.size();
      win.resize(d.width < MIN_WIN_WIDTH ? MIN_WIN_WIDTH : d.width,
                 d.height < MIN_WIN_HEIGHT ? MIN_WIN_HEIGHT : d.height);
      
      win.show();
   }

   /** Sets the corner field to the argument value. */
   public void setCorner(int corner)
   {
      this.needs_update = true;
      this.corner = corner;
   }
   
   /** Sets the desired x and y position to the argument coordinates. */
   public void setPosition(int x, int y)
   {
      old_x = this.x; this.x = x;
      old_y = this.y; this.y = y;
   }
   
   /** Positions the window being managed according to the desired 
    *  coordinates in the root window. 
    */
   public void positionWin()
   {
      Dimension siz = win.size();

      if (old_x == x && old_y == y && !needs_update) return;
      needs_update = false;
      old_x = x;
      old_y = y;

      switch (corner) 
      {
      case WinPosVector.LOWER_LEFT: y -= siz.height; break;
      case WinPosVector.LOWER_RIGHT: x -= siz.width; y -= siz.height; break;
      case WinPosVector.UPPER_LEFT: break;
      case WinPosVector.UPPER_RIGHT: x -= siz.width; break;
      }
      win.move(x + XTRA_WIDTH, y + XTRA_HEIGHT);
      // !!!BUG!!! Should be a sanity check on the window positioning
      // to make sure the upper left corner of the windows are on the
      // screen so the user has a handle for moving the windows if
      // necessary.
   }
}

/*****************************************************************************
 **                           Class WinPosVector                            **
 *****************************************************************************/
/**
 * This class is used to control the placement of a list of windows in
 * the root window.  The default management technique is to position the
 * windows starting in the upper left corner of the root window and then
 * display subsequent windows going down and to the right.  However, a
 * variety of layout behaviours can be gotten by supplying your on choice
 * for parameretizations at the time of object construction.  For
 * explanation of how the parameretizations work see the documentation
 * for the constructors.
 */
public class WinPosVector
{
   /** Denotes the lower left corner of a window. */
   public final static int LOWER_LEFT = 0;
   /** Denotes the lower right corner of a window. */
   public final static int LOWER_RIGHT = 1;
   /** Denotes the upper left corner of a window. */
   public final static int UPPER_LEFT = 2;
   /** Denotes the upper right corner of a window. */
   public final static int UPPER_RIGHT = 3;

   /** The default increment in pixels to use in the x and y directions
    * when placing successive windows in the <TT>WinPosVector</TT>.
    */
   public final static int STANDARD_GAP = 40;

   private Vector pos_list;  // Vector of WinPositions
   private int corner;       // To understand how corner, x, y, gap_x
   private int x, y;         // and gap_y work read the documentation
   private int gap_x, gap_y; // above the constructors.

   /** Creates a WinPosVector that lays windows out starting at the
    * upper left corner of the root window and places subsequent windows
    * down and to the right.
    */
   public WinPosVector()
   {
      this(UPPER_LEFT, STANDARD_GAP, STANDARD_GAP, 1, 1);
   }

   /** Creates a WinPosVector with a layout policy determined by the
    * parameters to the constructor. For example, the call
    * <pre>
    * WinPosVector(WinPosVector.UPPER_RIGHT, 40, 10, 900, 200)
    * </pre>
    * would indicate a layout policy that placed the first window's
    * upper right corner at the point (900, 200), with subsequent
    * windows being placed 40 pixels to the left and 10 pixels down.
    * The call
    * <pre>
    * WinPosVector(WinPosVector.LOWER_LEFT, 70, 0, 1, 500)
    * </pre>
    * would indicate a layout policy that placed the first window's
    * lower left corner at the point (1, 500), with subsequent
    * windows being placed 40 pixels to the right and 0 pixels up.
    * 
    * @param <b>corner</b> One of <TT>WinPosVector.UPPER_LEFT</TT>, etc. 
    * Indicates by which corner the windows will be placed. Furthermore, 
    * <TT>corner</TT> indicates how the values of <tt>gap-x</tt> and
    * <tt>gap-y</tt> will be construed. The values of <tt>gap-x</tt>
    * and <tt>gap-y</tt> are taken to mean increments towards the center
    * of the root window if you start at the corner of the
    * root window given by <tt>corner</tt>.
    * @param <b>gap_x</b> Indicates the step in pixels in the x direction
    * for placing subsequent windows in the list.
    * @param <b>gap_y</b> Analagous to <tt>gap_x</tt> for the y direction
    * @param <b>x</b> Starting x position of the first window's corner on the
    * root window.
    * @param <b>y</b> Starting y position of the first window's corner on the
    * root window.
    */                  
   public WinPosVector(int corner, int gap_x, int gap_y, int x, int y)
   {
      pos_list = new Vector();
      this.corner = corner;
      this.gap_x = gap_x;
      this.gap_y = gap_y;
      this.x = x; this.y = y;
   }

   private void positionWindows()
   {
      int curr_x = x;
      int curr_y = y;
      int stepx = 0, stepy = 0;

      switch (corner) 
      {
      case LOWER_LEFT: stepx = gap_x; stepy = -gap_y; break;
      case LOWER_RIGHT: stepx = -gap_x; stepy = -gap_y; break;
      case UPPER_LEFT: stepx = gap_x; stepy = gap_y; break;
      case UPPER_RIGHT: stepx = -gap_x; stepy = gap_y; break;
      }

      WinPosition tmp;
      for (int i = 0; i < pos_list.size(); ++i)
      {
         tmp = (WinPosition)pos_list.elementAt(i);
         tmp.setPosition(curr_x, curr_y);
         curr_x += stepx; curr_y += stepy;
         tmp.positionWin();
      }
   }

   /** Self explanatory. */
   public Enumeration elements()
   {
      return new WinPosEnumeration(pos_list);
   }
   
   /** Self explanatory. */
   public void addElement(Frame win)
   {
      pos_list.addElement(new WinPosition(win, corner, 0, 0));
      positionWindows();
   }
   
   /** Self explanatory. */
   public boolean removeElement(Window win)
   {
      int i;
      for (i = 0; i < pos_list.size(); ++i)
         if (((WinPosition)pos_list.elementAt(i)).win == win) break;
      if (i < pos_list.size())
      {
         pos_list.removeElementAt(i);
         return true;
      }
      else
         return false;
   }

   /** Self explanatory. */
   public void removeAllElements()
   {
      pos_list.removeAllElements();
   }
   
   /** Self explanatory. */
   public int size() { return pos_list.size(); }
}
