package lib.dynatype;

import java.io.*;


// See LispValue.java for documentation



//--------------------------------  LispSymbol  ------------------------------

public class LispSymbol extends LispAtom
{

  // Every LISP symbol has these four characteristics
  protected  LispValue  function;       // Function value
  protected  LispString name;           // Print name
  protected  LispValue  value;          // Assigned value
  protected  LispValue  plist;          // Property list
		
  protected  boolean    mixedCase;      // For mixed-case symbols

  // CONSTRUCTORS 
  public LispSymbol(String symbolName)
  {
    this(new LispString(symbolName));
  }

  public LispSymbol(LispString symbolNameString)
  {
    name       = symbolNameString;
    value      = null;              // Default to UNBOUND
    function   = null;              // Default to UNBOUND
    plist      = NIL;               // Default to NIL

    mixedCase  = (LispParser.firstCharNotInSet(0, symbolNameString.getValue(),
 					       "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ:-?!.%$")
 		  < ((LispInteger)symbolNameString.length()).getValue());

  }


  // ------  BASIC (non-LISP) methods  --------

  public boolean basic_symbolp()
  { return true; }

  // 'equals' is required for HashTable.
  public boolean equals(LispSymbol otherSymbol)
  {
    return (this == otherSymbol);   // Same symbols have same address
  }

  public void internal_prin1(PrintStream os)
  {
    if (mixedCase) os.print("|");
    os.print(name.getValue());
    if (mixedCase) os.print("|");
  }
  
  public void internal_princ(PrintStream os) { os.print(name.getValue()); }

  public void internal_print(PrintStream os)
  {
    if (mixedCase) os.print("|");
    os.print(name.getValue());
    if (mixedCase) os.print("|");
  }
  
  public String toString()
  {
    if (mixedCase)
      return "|" + name.getValue() + "|";
    else
      return name.getValue();
  }


  // ------  LISP methods  --------

  public LispValue  apply(LispValue args)
  {
    if (function == null)
      return super.apply(args);
    else
    {
      System.err.println("\nSorry, APPLY is not yet implemented.");
      
      return LispValue.NIL;
    }
  }

  public LispValue  boundp()
  {
    if (value == null)
      return LispValue.NIL;
    else
      return LispValue.T;
  }

  public LispValue  fboundp()
  {
    if (function == null)
      return LispValue.NIL;
    else
      return LispValue.T;
  }

  /*  NO FUNCALL in this version - uses SECDMachine

  public LispValue  funcall(LispValue args)
  {
    if (function == null)
      throw new LispValueNotAFunctionException("The first argument to FUNCALL");
    else
    {
      // push the args back on the stack
      for (LispValue v = args; v != NIL; v = v.cdr())
	Jill.MACHINE.S.push(v.car());

      // get the function, and push it on the code stack.
      // Note that we don't do error checking on the number
      // of arguments.  This is bad...
      Jill.MACHINE.C.pop();                     // Pop the funcall function off.
      Jill.MACHINE.C.push(function.second());   // Push the new one on.
    }

    return NIL;
  }
  */

  public LispValue     pop          ()
  {
     LispValue returnValue;
     
     if (value.listp() == T)
     {
	returnValue       = value.car();
	setf_symbol_value(value.cdr());
	return returnValue;
     }
    else 
       throw new LispValueNotAListException("The value of "
                    + ((LispString)name).getValue());
  }

  public LispValue     push         (LispValue newValue)
  {
     if (value.listp() == T)
     {
	setf_symbol_value(new LispCons(newValue, value));
	return newValue;
     }
    else 
       throw new LispValueNotAListException("The value of "
                    + ((LispString)name).getValue());
  }

  public LispValue setf_symbol_function(LispValue newFunction)
  {
    function = newFunction;
    return function;
  }

  public LispValue setf_symbol_plist(LispValue newPlist)
  {
    plist = newPlist;
    return plist;
  }

  public LispValue setf_symbol_value(LispValue newValue)
  {
    value = newValue;
    return value;
  }

  public LispValue     symbolp() { return T; }

  public LispValue symbol_function() throws LispException
  {
    if (function == null)
    {
      throw new LispUndefinedFunctionException(name.getValue());
    }
    return function;
  }

  public LispValue symbol_name()
  {
    return name;
  }

  public LispValue symbol_plist()
  {
    return plist;
  }

  public LispValue symbol_value() throws LispException
  {
    // System.out.print("\nReturning value of '"); prin1(); System.out.print("': "); value.prin1();

    if (value == null)
    {
      throw new LispUnboundVariableException(name.getValue());
    }
    return value;
  }

  public LispValue     type_of     ()  { return LispValue.SYMBOL_TYPE;   }

};

