/* Copyright (c) 1995, 1996, 1997    Micheal Hewett   hewett@cs.utexas.edu
 *
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *  This source code is copyrighted as shown above.  If this
 *  code was obtained as part of a freeware or shareware release,
 *  assume that the provisions of the Gnu "copyleft" agreement
 *  apply.
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 *
 *        File:  LispValue.java
 *
 *      Author:  Micheal Hewett
 *     Created:  12 Jan 1997, from original of 21 June 1995
 *
 *    Compiler:  javac 1.0.2
 *
 * Description:  Declarations for the LISP-type data types
 *
 ****************************************************************************
 *  Content Summary
 *  ---------------
 *
 *  Values can be any type:
 *
 *     integer, float, symbol, string, list
 *
 *  This file contains the function definitions to perform all the 
 *  operations involving these values.
 *
 *  Classes:
 *
 *     LispException
 *       LispConstantRedefinedException
 *       LispIndexOutOfRangeException
 *       LispUnboundVariableException
 *       LispUndefinedFunctionException
 *
 *       LispValueNotAnAtomException
 *       LispValueNotAFunctionException
 *       LispValueNotAnIntegerException
 *       LispValueNotAListException
 *       LispValueNotANumberException
 *       LispValueNotASequenceException
 *       LispValueNotASymbolException
 *
 *     LispValue
 *	 LispConsOrNil                 (abstract)
 *         LispCons
 *         LispNil
 *       LispAtom                      (abstract)
 *         LispCharacter
 *         LispNumber                  (abstract)
 *           LispInteger
 *           LispReal
 *         LispString
 *         LispSymbol
 *           LispConstant
 *             LispKeyword
 *       LispStructure      (NYI)
 *       LispForeignObject             (abstract)
 *         LispJavaObject
 *           JavaConnection     (NYI)
 *           JavaInetAddress    (NYI)
 *           JavaServerSocket   (NYI)
 *           JavaSocket         (NYI)
 *           JavaURL            (NYI)
 *           JavaURLEncoder     (NYI)
 *
 *
 *     LispValueFactory
 *
 *  All operations are defined on LispValue (e.g. '+') so that you can
 *  legally (in Java) add any two values but at runtime you will get
 *  an exception unless you actually have valid operands.
 *
 *  Use the ValueFactory object to create the values.
 *
 *
 ****************************************************************************
 */

package lib.dynatype;

import java.io.*;

import lib.util.SymbolTable;


//--------------------------  LispException  ---------------------------------

abstract class LispException extends RuntimeException
{
  LispException()         { super();  }
  LispException(String s) { super(s); }
}

class LispConstantRedefinedException extends LispException
{
  LispConstantRedefinedException()            { super();  }
  LispConstantRedefinedException(String s)
  { super("The constant " + s + " can not be redefined.");  }
}

class LispIndexOutOfRangeException extends LispException
{
  LispIndexOutOfRangeException()            { super();  }
  LispIndexOutOfRangeException(String s)
  { super("The index " + s + " is out of range.");  }
}

class LispUnboundVariableException extends LispException
{
  LispUnboundVariableException()            { super();  }
  LispUnboundVariableException(String s)    { super("The variable " + s + " is unbound.");  }
}

class LispUndefinedFunctionException extends LispException
{
  LispUndefinedFunctionException()          { super();  }
  LispUndefinedFunctionException(String s)  { super(s + " is not a defined function.");  }
}

class LispValueNotAnAtomException extends LispException
{
  LispValueNotAnAtomException()             { super();  }
  LispValueNotAnAtomException(String s)     { super(s + " is not an ATOM."); }
}

class LispValueNotAFunctionException extends LispException
{
  LispValueNotAFunctionException()             { super();  }
  LispValueNotAFunctionException(String s)     { super(s + " is not a FUNCTION."); }
}

class LispValueNotAnIntegerException extends LispException
{
  LispValueNotAnIntegerException()          { super();  }
  LispValueNotAnIntegerException(String s)  { super(s + " is not an INTEGER."); }
}

class LispValueNotAListException extends LispException
{
  LispValueNotAListException()              { super();  }
  LispValueNotAListException(String s)      { super(s + " is not a LIST."); }
}

class LispValueNotANumberException extends LispException
{
  LispValueNotANumberException()            { super();  }
  LispValueNotANumberException(String s)    { super(s + " is not a NUMBER."); }
}

class LispValueNotASequenceException extends LispException
{
  LispValueNotASequenceException()          { super();  }
  LispValueNotASequenceException(String s)  { super(s + " is not a SEQUENCE.");}
}

class LispValueNotAStringException extends LispException
{
  LispValueNotAStringException()            { super();  }
  LispValueNotAStringException(String s)    { super(s + " is not a STRING."); }
}

class LispValueNotASymbolException extends LispException
{
  LispValueNotASymbolException()            { super();  }
  LispValueNotASymbolException(String s)    { super(s + " is not a SYMBOL."); }
}


//----------------------------  LispValue  ---------------------------------


/**
 * LispValue is the root of all of the dynamically-typed LISP-like
 * structures.  It contains definitions of all methods that operate
 * on Lisp values.  This class is not instantiated directly.  Instead,
 * use a LispValueFactory instance to create the instances.
 * @see LispValueFactory
 * @author  Micheal S. Hewett    hewett@cs.utexas.edu
 * @date    Wed Feb 19 17:01:40 1997
 * @version 1.0
 *
 */
public abstract class LispValue      // Base class for all the LISP data types
{
  public static final LispValueFactory VF     = new LispValueFactory();
  public static final SymbolTable      SYMTAB = new SymbolTable();

  // The '.' to represent a cons cell.
  public  static LispValue   DOT;

  // The list/symbol NIL.
  public  static LispValue   NIL;

  // The apostrophe.
  public  static LispValue   QUOTE;

  // The symbol T
  public  static LispValue   T;
  
  // Types
  public  static LispValue   CHARACTER_TYPE;
  public  static LispValue   CONS_TYPE;
  public  static LispValue   INTEGER_TYPE;
  public  static LispValue   NULL_TYPE;
  public  static LispValue   DOUBLE_FLOAT_TYPE;
  public  static LispValue   STRING_TYPE;
  public  static LispValue   SYMBOL_TYPE;


  // Static initializor method for the above variables
  static 
  {
    DOT = new LispSymbol(".");
    intern(VF.makeString("DOT"), DOT);
		
    NIL = new LispNil("NIL");
    intern(VF.makeString("NIL"), NIL);
		
    QUOTE = new LispSymbol("QUOTE");
    intern(VF.makeString("QUOTE"), QUOTE);
		
    T = new LispConstant("T");
    intern(VF.makeString("T"), T);
    T.setf_symbol_value(T);

    // Lisp data types
    CHARACTER_TYPE = new LispSymbol("CHARACTER");
    intern(VF.makeString("CHARACTER"), CHARACTER_TYPE);
    
    CONS_TYPE = new LispSymbol("CONS");
    intern(VF.makeString("CONS"), CONS_TYPE);

    INTEGER_TYPE = new LispSymbol("INTEGER");
    intern(VF.makeString("INTEGER"), INTEGER_TYPE);

    NULL_TYPE = new LispSymbol("NULL");
    intern(VF.makeString("NULL"), NULL_TYPE);

    DOUBLE_FLOAT_TYPE = new LispSymbol("DOUBLE-FLOAT");
    intern(VF.makeString("DOUBLE-FLOAT"), DOUBLE_FLOAT_TYPE);

    STRING_TYPE = new LispSymbol("STRING");
    intern(VF.makeString("STRING"), STRING_TYPE);

    SYMBOL_TYPE = new LispSymbol("SYMBOL");
    intern(VF.makeString("SYMBOL"), SYMBOL_TYPE);

  }
	
  public void internal_princ(PrintStream os)
  { os.print("#<unprintable object>"); }
  public void internal_princ_as_cdr(PrintStream os)
  { os.print(" . "); internal_princ(os);  }
  
  public void internal_prin1(PrintStream os)
  { os.print("#<unprintable object>"); }
  public void internal_prin1_as_cdr(PrintStream os)
  { os.print(" . "); internal_prin1(os);  }

  public void internal_print(PrintStream os)
  { System.err.print("#<unprintable object>"); }
  public void internal_print_as_cdr(PrintStream os)
  { os.print(" . "); internal_print(os);  }
  
  public boolean basic_atom()      { return false; }
  public boolean basic_consp()     { return false; }
  public boolean basic_constantp() { return false; }
  public boolean basic_foreignp()  { return false; }
  public boolean basic_integerp()  { return false; }
  public boolean basic_keywordp()  { return false; }
  public boolean basic_listp()     { return false; }
  public boolean basic_null()      { return false; }
  public boolean basic_numberp()   { return false; }
  public boolean basic_symbolp()   { return false; }

  /**
   * <code>toString()</code> returns a printed representation
   * of the form (as printed by <code>(prin1)</code>) in 
   * a Java string.
   * @return String The value in a string.
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Wed Feb 19 17:18:50 1997
   * @version 1.0
   * 
   */
  public String toString() { return "<unprintable object>"; }

  public String toStringAsCdr()
  {
    StringBuffer buf = new StringBuffer();
    
    buf.append(" . ");
    buf.append(toString());

    return buf.toString();
  }

/* ------------------  LISP functions    ------------------------------ */

  public LispValue     append       (LispValue otherList)
  { throw new LispValueNotAListException("The first argument to APPEND");  }

  public LispValue     apply        (LispValue args)
  { throw new LispValueNotAFunctionException("The first argument to APPLY");  }

  public LispValue     assoc        (LispValue index)
  { throw new LispValueNotAListException("The second argument to ASSOC"); }

  public LispValue     atom         ()  { return NIL;  }

  public LispValue     boundp       ()
  { throw new LispValueNotASymbolException("The argument to BOUNDP");  }

  public LispValue     butlast      ()
  { throw new LispValueNotAListException("The argument to BUTLAST");  }

  public LispValue     car          ()
  { throw new LispValueNotAListException("The argument to CAR");  }

  public LispValue     cdr          ()
  { throw new LispValueNotAListException("The argument to CDR");  }

  public LispValue     characterp   ()  { return NIL; }

  public LispValue     consp        ()  { return NIL;  }

  public LispValue     copy_list    ()
  { throw new LispValueNotAListException("The argument to COPY-LIST");  }

  public LispValue     eighth       ()
  { throw new LispValueNotASequenceException("The first argument to EIGHTH"); }

  public LispValue     elt          (LispValue index)
  { throw new LispValueNotASequenceException("The first argument to ELT"); }

  public LispValue     eq           (LispValue val)
  {
    if (this == val)
      return T;
    else
      return NIL;
  }

  public LispValue     eql          (LispValue val)
  {
    if (this == val)
      return T;
    else
      return NIL;
  }

  public LispValue     equal        (LispValue val)
  {
    if (this == val)
      return T;
    else
      return NIL;
  }

  public LispValue     fboundp      ()
  { throw new LispValueNotASymbolException("The argument to FBOUNDP");  }

  public LispValue floatp()  { return NIL; }

  public LispValue     fifth        ()
  { throw new LispValueNotASequenceException("The first argument to FIFTH"); }

  public LispValue     first        ()
  { throw new LispValueNotASequenceException("The first argument to FIRST"); }

  public LispValue     fourth       ()
  { throw new LispValueNotASequenceException("The first argument to FOURTH"); }

  public LispValue     funcall      (LispValue args)
  { throw new LispValueNotAFunctionException("The first argument to FUNCALL"); }

  public LispValue     integerp     ()  { return NIL; }

  public static LispValue intern(LispValue symbolString)
  {
    LispValue newSymbol;
		
    // First, check to see whether one exists already.
    newSymbol = SYMTAB.get(symbolString);
		
    if (newSymbol == null)
      System.err.print("\nSYMTAB.get() returned NULL in intern() for " + symbolString.toString());

    if (newSymbol != NIL)    // Already there, don't add it again.
      return newSymbol;		
    else
      if (symbolString.elt(LispInteger.ZERO).eql(LispCharacter.COLON) == LispValue.T)
      {
	newSymbol = VF.makeKeyword(symbolString);
	return intern(symbolString, newSymbol);
      }
      else
      {
	newSymbol = VF.makeSymbol(symbolString);
	return intern(symbolString, newSymbol);
      }
  }
	
  // We need this for the startup when we create LispValue.NIL and LispValue.T.
  // Actually, the Symbol parameter is always a LispSymbol, but because of NIL's 
  // strange properties, we must make the type be LispValue.
  public static LispValue intern(LispValue symbolString, LispValue symbol)
  {
    return SYMTAB.put(symbolString, symbol);
  }

  // I'm not sure why we need this, except for testing...
  public static LispValue intern(String str) 
  {
    return intern(VF.makeString(str.toUpperCase()));
  }

  public LispValue     keywordp     ()  { return NIL; }

  public LispValue     last         ()
  { throw new LispValueNotAListException("The argument to LAST");  }
       
  public LispValue     length       ()
  { throw new LispValueNotASequenceException("The argument to LENGTH");  }

  public LispValue     listp        ()  { return NIL;  }
       
  public LispValue     member       (LispValue elt)
  { throw new LispValueNotAListException("The second argument to MEMBER");  }

  public LispValue     ninth        ()
  { throw new LispValueNotASequenceException("The first argument to NINTH"); }
  
  public LispValue     nreverse     ()
  { throw new LispValueNotAListException("The argument to NREVERSE");  }
  
  public LispValue     lisp_null    () { return NIL; }
  
  public LispValue     numberp      () { return NIL; }

  public LispValue     pop          ()
  { throw new LispValueNotASymbolException("The argument to POP");  }

  public LispValue     prin1        ()  { internal_prin1(System.out); return this; }

  public LispValue     princ        ()  { internal_princ(System.out); return this; }

  public LispValue     print        ()
  { System.out.println(); internal_print(System.out); System.out.print(" "); return this; }

  public LispValue     push         (LispValue value)
  { throw new LispValueNotASymbolException("The second argument to PUSH");  }

  public LispValue     rassoc       (LispValue index)
  { throw new LispValueNotAListException("The second argument to RASSOC");  }
  
  public LispValue     remove       (LispValue elt)
  { throw new LispValueNotASequenceException("The second argument to REMOVE");  }
  
  public LispValue     reverse      ()
  { throw new LispValueNotASequenceException("The argument to REVERSE"); }
  
  public LispValue     rplaca       (LispValue newCar)
  { throw new LispValueNotAListException("The first argument to RPLACA");  }
  
  public LispValue     rplacd       (LispValue newCdr)
  { throw new LispValueNotAListException("The first argument to RPLACD");  }

  public LispValue     second       ()
  { throw new LispValueNotASequenceException("The first argument to SECOND"); }

  public LispValue     setf_symbol_function(LispValue newFunction)
  { throw new LispValueNotASymbolException("The argument to SETF-SYMBOL-FUNCTION"); }
  
  public LispValue setf_symbol_plist(LispValue newPlist)
  { throw new LispValueNotASymbolException("The argument to SETF-SYMBOL-PLIST"); }
  
  public LispValue setf_symbol_value(LispValue newValue)
  { throw new LispValueNotASymbolException("The argument to SETF-SYMBOL-VALUE"); }

  public LispValue     seventh      ()
  { throw new LispValueNotASequenceException("The first argument to SEVENTH"); }

  public LispValue     sixth        ()
  { throw new LispValueNotASequenceException("The first argument to SIXTH"); }
  
  public LispValue     stringp      ()     { return NIL; }

  public LispValue     subst(LispValue oldValue, LispValue newValue)
  { throw new LispValueNotAListException("The third argument to SUBST");  }

  public LispValue     symbolp      ()     { return NIL; }
   
  public LispValue     symbol_function() throws LispException
  { throw new LispValueNotASymbolException("The argument to SYMBOL_FUNCTION");  }
  
  public LispValue     symbol_name()
  { throw new LispValueNotASymbolException("The argument to SYMBOL_NAME");  }
  
  public LispValue     symbol_plist()
  { throw new LispValueNotASymbolException("The argument to SYMBOL_PLIST");  }
  
  public LispValue     symbol_value() throws LispException
  { throw new LispValueNotASymbolException("The argument to SYMBOL_VALUE");  }

  public LispValue     tenth        ()
  { throw new LispValueNotASequenceException("The first argument to TENTH"); }

  public LispValue     third       ()
  { throw new LispValueNotASequenceException("The first argument to THIRD"); }

  // Everything not anything else is a T, although this return value is illegal in CLTL2.
  public LispValue     type_of     ()  { return T;   }

  public LispValue     zerop       ()
  { throw new LispValueNotANumberException("The argument to ZEROP"); }


  // Arithmetic functions

  public LispValue     add         (LispValue args)
  { throw new LispValueNotANumberException("An argument to + (add)"); }

  public LispValue     divide      (LispValue args)
  { throw new LispValueNotANumberException("An argument to / (divide)"); }

  public LispValue     multiply    (LispValue args)
  { throw new LispValueNotANumberException("An argument to * (multiply)"); }

  public LispValue     subtract    (LispValue args)
  { throw new LispValueNotANumberException("An argument to - (subtract)"); }
    
}
