// ===================================================================
// Copyright (c) 1997, All rights reserved, by Micheal Hewett
//
// This software is free for educational and non-profit use.
// Any for-profit use must be governed by a license available
// from the author, who can be contacted via email at 
// "hewett@cs.stanford.edu"
//
// ===================================================================
//
//  LispValueFactory.java  - Creates instances of the LispValue classes
//
//  20 Feb 1997
//
// -------------------------------------------------------------------

package lib.dynatype;



/**
 * The LispValueFactory knows how to create values of different types. 
 * Make an instance of this class and use it to create instances of the types. 
 *                                                                     
 * The only possible error is an OutOfMemoryError, which Java will
 * generate on its own, so we don't do any error checking.
 *
 * Example:
 * <pre>
 *   LispValueFactory vf = new LispValueFactory();
 *   LispValue         a = vf.makeInteger(5);
 *   LispValue         b = vf.makeString("Hello");
 *   LispValue         x = vf.makeCons(a, b);
 * </pre>
 * @see LispValue
 * @author  Micheal S. Hewett    hewett@cs.utexas.edu
 * @date    Thu Feb 20 12:06:13 1997
 * @version 1.0
 *
 */
public class LispValueFactory
{
  /**
   * makeCons(a,b) creates a new Cons cell, initialized with
   * the values a and b as the CAR and CDR respectively.
   * 
   * @see LispCons
   * @param LispValue theCar
   * @param LispValue theCdr
   * @return LispValue
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:08:32 1997
   * @version 1.0
   * 
   */
  public LispCons        makeCons    (LispValue theCar, LispValue theCdr)
  {  return new LispCons(theCar, theCdr); }


  /**
   * Creates a list of the 2 or 3 items parameters,
   * and returns it.
   *
   * @see LispCons
   * @see LispValue
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:10:00 1997
   * @version 1.0
   * 
   */
  public LispCons        makeList    (LispValue first, LispValue second)
  {  return new LispCons(first, new LispCons(second, LispValue.NIL)); }

  public LispCons makeList(LispValue first, LispValue second, LispValue third)
  {  return new LispCons(first, 
                   new LispCons(second, 
                          new LispCons(third, LispValue.NIL))); }


  /**
   * Creates a LispInteger type initialized with the value
   * provided and returns it.
   * @see LispInteger
   * @see LispValue
   * @return LispInteger
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:16:21 1997
   * @version 1.0
   * 
   */
  public LispInteger     makeInteger (Long value)
  { return new LispInteger(value.longValue()); }

  public LispInteger     makeInteger (long value)
  { return new LispInteger(value); }

  public LispInteger     makeInteger ()
  { return new LispInteger(0); }


  /**
   * Creates an instance of LispReal initialized with
   * the given value.
   * @see LispInteger
   * @see LispValue
   * @return LispReal
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:19:15 1997
   * @version 1.0
   * 
   */
  public LispReal        makeReal    (Double value)
  { return new LispReal(value.doubleValue()); }

  public LispReal        makeReal    (double value)
  { return new LispReal(value); }

  public LispReal        makeReal    ()
  { return new LispReal(0.0); }


  /**
   * Creates a LispString from a Java string.
   * 
   * @see LispString
   * @see LispValue
   * @return LispString
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:20:13 1997
   * @version 1.0
   * 
   */
  public LispString      makeString  (String str)
  { return new LispString(str); }


  /**
   * Creates a LispSymbol from a string or LispString.
   * This method does <b>not</b> intern the symbol.
   * 
   * @see LispSymbol
   * @see LispValue
   * @return LispSymbol
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:20:57 1997
   * @version 1.0
   * 
   */
  public LispSymbol      makeSymbol  (String      symbolName)
  { return new LispSymbol(symbolName); }

  public LispSymbol      makeSymbol  (LispValue  symbolName)
  { return new LispSymbol((LispString)symbolName); }
  

  /**
   * Creates a LispConstant (a type of Symbol whose value
   * can not be changed).  This method does <b>not</b> 
   * intern the symbol.
   * 
   * @see LispConstant
   * @see LispSymbol
   * @see LispValue
   * @return LispSymbol
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:20:57 1997
   * @version 1.0
   * 
   */
  public LispSymbol      makeConstant  (String      symbolName)
  { return new LispConstant(symbolName); }

  public LispSymbol      makeConstant  (LispValue   symbolName)
  { return new LispConstant((LispString)symbolName); }


  /**
   * Creates a LispKeyword (a type of Symbol that evaluates
   * to itself).  This method does <b>not</b> intern the symbol.
   * 
   * @see LispKeyword
   * @see LispConstant
   * @see LispSymbol
   * @see LispValue
   * @return LispSymbol
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:20:57 1997
   * @version 1.0
   * 
   */
  public LispSymbol      makeKeyword  (String      symbolName)
  { return new LispKeyword(symbolName); }

  public LispSymbol      makeKeyword  (LispValue  symbolName)
  { return new LispKeyword((LispString)symbolName); }
  

  /**
   * Creates a LispNil (the funny symbol/cons that is the LISP NIL).
   * This method does <b>not</b> intern the symbol.
   * 
   * @see LispNil
   * @see LispCons
   * @see LispSymbol
   * @see LispValue
   * @return LispSymbol
   * @author  Micheal S. Hewett    hewett@cs.utexas.edu
   * @date    Thu Feb 20 12:20:57 1997
   * @version 1.0
   * 
   */
  public LispNil         makeNIL     (String      symbolName)
  { return new LispNil(symbolName); }

  public LispNil         makeNIL     (LispValue   symbolName)
  { return new LispNil((LispString)symbolName); }

};
