;;;; -*- Mode:Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

#-(or LUCID GCL) (in-package :CL-USER)
#+(or LUCID GCL) (in-package :USER)

;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Copyright (c) 1995 by Micheal Scott Hewett
;;;
;;; This code may be used by anyone for any project, but may not
;;; be sold in source or object form without permission.
;;; If in doubt, follow the GNU "copyleft" guidelines.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; (Contact: hewett@cs.utexas.edu or hewett@cs.stanford.edu)
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 


;;;;----------------  LISP-UNIX interface code -------------------------------
;;;
;;; lisp-to-unix.lisp     Handles communication between a LISP program and
;;;                       another program, usually a Motif program written
;;;                       in C.  Passes all communication back and forth.
;;;                       Written for LUCID Common LISP.
;;;

;;; 04 March 1995   Mike Hewett  
;;;                 From code written in Fall, 1994 for use with Algernon
;;;                 and code written in Fall, 1992 for use with BB1.
;;;
;;; Modifications - - - - - - - - - - - - - - - - - - - - - - - - 
;;;   
;;;    30 May   1995  (mh)  added external-flush.
;;;
;;;    10 March 1995  (mh)  Changes to run under Allegro Franz CL.
;;;    22 Nov   1996  (mh)  Changed package to :CL-USER
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;;;
;;; This file contains low-level functions to pass information into and
;;; out of LISP.  User-level functions such as "display-object" should
;;; call these functions.
;;;
;;;
;;; Comm Functions:
;;; ---------------
;;;  open-external-program       (path direction)
;;;  shutdown-external-program   (external-stream-structure) ;;same as close
;;;  close-external-program      (external-stream-structure &OPTIONAL command)
;;;  external-flush              (external-stream-structure)
;;;  external-raw-output         (external-stream-structure args...)
;;;  external-read               (external-stream-structure)
;;;  external-string-output      (external-stream-structure format args...)
;;;
;;;  external-input-stream-p     (external-stream-structure)
;;;  external-output-stream-p    (external-stream-structure)
;;;
;;;  with-no-external-flush      MACRO: surrounds any of the above fns.
;;;
;;; Structures
;;; -------------------
;;; external-stream
;;;
;;; Sample usage:
;;; ------------
;;;    ...
;;;    (setq ext-stream-1 (open-external-program "/lusr/bin/lpr" :IO))
;;;    (external-string-output ext-stream-1 "~A is ~A" 'Grinder-1 'Running)
;;;    (setq response (external-read ext-stream-1))
;;;    ...
;;;    (close-external-program ext-stream-1)
;;;
;;;  - - - - - - - - - - - - - - - - - - - -
;;;  Also see the function "test-l2u" below
;;;  - - - - - - - - - - - - - - - - - - - -
;;;
;;;;-----------------------------------------------------------------  


;;;;------------  IMPORTANT  ----------------------------------------  

;;; Some tokens that the external program MUST recognize.

(defvar *command-text-begin*              #\s)
(defvar *command-text-end*                (code-char 0))  ;; EOS in UNIX
(defvar *command-quit*                    #\q)

;;;;-----------------------------------------------------------------  

(defparameter *external-flush-enabled* T
  "Use Macro WITH-NO-EXTERNAL-FLUSH to disable flush after each output event.")

(defmacro with-no-external-flush (&REST forms)
  
  `(let ((*external-flush-enabled* NIL))
     ,@forms)
  )


;;;;-----------------------------------------------------------------  

(defstruct external-stream

  (input-stream    NIL)        ;;; Where to read
  (output-stream   NIL)        ;;; Where to write
  (direction       NIL)        ;;; :INPUT, :OUTPUT, or :IO
  (process         NIL)        ;;; UNIX process number
  (user-state      NIL)        ;;; for user info
  )  

(defmacro external-input-stream-p (macro-ext-stream)
  `(member (external-stream-direction ,macro-ext-stream) '(:IO :INPUT))
  )

(defmacro external-output-stream-p (macro-ext-stream)
  `(member (external-stream-direction ,macro-ext-stream) '(:IO :OUTPUT))
  )


;;;---------  Algernon-level Functions  --------------------

;;;  OPEN-EXTERNAL-PROGRAM

(defun open-external-program (path direction &KEY (verbose T))
  "Starts the external program from 'path' and allows input/output according
to 'direction', which can be :INPUT, :OUTPUT, or :IO
Returns the 'external-stream' structure to be used when communicating 
with the external-stream."

  ;;; Check arguments

  (unless (member direction '(:INPUT :OUTPUT :IO))
    (break "OPEN-EXTERNAL-PROGRAM: 'direction' must be :INPUT :OUTPUT or :IO")
    )

  (unless (stringp path)
    (break "OPEN-EXTERNAL-PROGRAM: 'path' must be a string.")
    )


  ;;; Some preliminary setup needed for Allegro CL and shell interaction.
  ;;; See Section 8-2 of the Allegro manual, volume 1.
  ;;; Since "run-shell-command" invokes a shell, it processes
  ;;; the ".kshrc" or ".cshrc" init file.  You can use the
  ;;; LISP shell variable to prevent certain code from executing.

#+ALLEGRO
  (progn
    (load "" :unreferenced-lib-names `(,(ff:convert-to-lang "putenv")))
    (ff:defforeign 'putenv :arguments '(integer))
    (putenv (ff:string-to-char* "LISP=T"))
    )

  ;;; Open external program connection

  (let (to-external from-external process-number ignore2 ignore3 ignore5
	external-io
        (new-stream  (make-external-stream))
	)
#+LUCID
    (declare (ignore ignore2 ignore3 ignore5 external-io))
#+ALLEGRO
    (declare (ignore ignore2 ignore3 ignore5 to-external from-external))

#+LUCID
    (multiple-value-setq
     (to-external from-external ignore3 process-number ignore5)
     (run-program path
		  :input  :stream
		  :output :stream
		  :error-output nil  :wait nil))

#+ALLEGRO
    (multiple-value-setq
     (external-io ignore2 process-number)
     (excl::run-shell-command    (format nil "exec ~A" path)
				 :input  :stream
				 :output :stream
				 :error-output nil
				 :wait nil))

    (when verbose
      (format *trace-output*
	      "~%Connected to UNIX process ~D~%" process-number)
      )

    (setf (external-stream-process   new-stream) process-number)
    (setf (external-stream-direction new-stream) direction)

    (when (member direction '(:INPUT :IO))
      (setf (external-stream-input-stream new-stream)
	    #+LUCID to-external   #+ALLEGRO external-io
	    ))

    (when (member direction '(:OUTPUT :IO))
      (setf (external-stream-output-stream new-stream)
	    #+LUCID to-external     #+ALLEGRO external-io
	    ))

    new-stream
    )
  )


;; SHUTDOWN-xx is for backward compatibility.  Use CLOSE-xx
(defun shutdown-external-program (ext-stream)
  (close-external-program ext-stream))

(defun close-external-program (ext-stream
			       &OPTIONAL (quit-command *command-quit*))

  ;; The external program may have shut down already, so
  ;; we catch errors here.

  (handler-case
      (progn
	(unless (eq (external-stream-direction ext-stream) :INPUT)
	  (external-raw-output ext-stream quit-command)
	  )

	(when (external-stream-input-stream ext-stream)
	  (close (external-stream-input-stream ext-stream))
	  )

	(when (external-stream-output-stream ext-stream)
	  (close (external-stream-output-stream ext-stream))
	  )
	)

    (FILE-ERROR ()     
      (format *error-output* "~%;; The external program has terminated."))
    )

  (setf (external-stream-input-stream  ext-stream) NIL)
  (setf (external-stream-output-stream ext-stream) NIL)

  (setf (external-stream-direction     ext-stream) NIL)
  (setf (external-stream-process       ext-stream) NIL)
  ;;Don't reset user state.

#+ALLEGRO
  (progn
    (format *standard-output* "~%;; Waiting for external process to end...")
    (finish-output *standard-output*)
    (sys::os-wait)       ;; Wait for the process to shut down.
    (format *standard-output* ";; done.~%")
    (finish-output *standard-output*)
    )

  :CLOSED
  )


(defun external-flush (ext-stream)
  "Like flush, for output streams."

  (when *external-flush-enabled*
    (if (external-output-stream-p ext-stream)
	(finish-output (external-stream-output-stream ext-stream))
      ;;ELSE
      (break "EXTERNAL-FLUSH: ext-stream is not an output stream.")
      )
    )
  )
      

(defun external-string-output (ext-stream &REST format-and-args)
  "Like FORMAT."

  (if (external-output-stream-p ext-stream)
      (#+Lucid WITH-INTERRUPTS-DEFERRED
       #-Lucid PROGN
	       
        (princ *command-text-begin* (external-stream-output-stream ext-stream))
	(let ((*print-pretty* NIL))
	  (apply #'format
		 (cons (external-stream-output-stream ext-stream)
		       format-and-args)))

	(princ *command-text-end* (external-stream-output-stream ext-stream))

	(external-flush ext-stream)
	)
    ;;else
      (break "EXTERNAL-FORMAT: ext-stream is not open for output.")
      )
  )


(defun external-raw-output (ext-stream &REST args)
  "Like PRINC."

  (if (external-output-stream-p ext-stream)
      (#+Lucid WITH-INTERRUPTS-DEFERRED
       #-Lucid PROGN
	       
	(dolist (arg args)    ;;Print each arg to the stream.
	  (princ arg (external-stream-output-stream ext-stream))
	  )

	(external-flush ext-stream)
	)
    ;;else
      (break "EXTERNAL-RAW-OUTPUT: ext-stream is not open for output.")
      )
  )


(defun external-read (ext-stream &OPTIONAL (listen-first NIL))
  "Does a (listen) first so it won't wait for input if there
is none available.  It is recommended that you send commands from
the external program as a LISP list containing keywords to 
facilitate reading.  Example:  (:SET-GRAPH-DIRECTION :VERTICAL)"

  (if (external-input-stream-p ext-stream)
      (if listen-first
	  (#+Lucid WITH-INTERRUPTS-DEFERRED
	   #-Lucid PROGN
	   (when (listen (external-stream-input-stream ext-stream))
	     (read (external-stream-input-stream ext-stream))))
	;;ELSE
  	  (read (external-stream-input-stream ext-stream))
	)
    ;;else
      (break "EXTERNAL-READ: ext-stream not open for input.")
      )
  )

;;;----------- TEST Function -----------


(defun test-l2u (&OPTIONAL (test-prog "/u/hewett/bin/xecho"))

  (let ((ext-stream (open-external-program test-prog :IO))
	)

    (external-raw-output ext-stream "a string" #\linefeed
			 #\A #\linefeed
			 #\Z #\linefeed)
    
;;    (dotimes (i 100)
;;      (external-string-output ext-stream
;;			      "~200A: ~A ~5d~%" "Hello" 'close 7)
;;      )
;;    (print (external-read ext-stream))
    (print (external-read ext-stream :LISTEN))
    (external-string-output ext-stream
			    "~20A: ~A ~5d~%" "Again" 'close 8)
    (external-string-output ext-stream
			    "~20A: ~A ~5d~%" "Goodbye" 'close 9)

    (close-external-program ext-stream)
    )
  )


(defun test-l2u-2 (&OPTIONAL (test-prog "/u/hewett/bin/xecho"))

  (let ((ext-stream (open-external-program test-prog :IO))
	)

    (external-raw-output ext-stream "a string" #\linefeed
			 #\A #\linefeed
			 #\Z #\linefeed)
    
;;    (dotimes (i 100)
;;      (external-string-output ext-stream
;;			      "~200A: ~A ~5d~%" "Hello" 'close 7)
;;      )
    (print (external-read ext-stream :LISTEN))
    (test-crash2 ext-stream)
    (external-string-output ext-stream
			    "~20A: ~A ~5d~%" "Again" 'close 8)
    (external-string-output ext-stream
			    "~20A: ~A ~5d~%" "Goodbye" 'close 9)

    (shutdown-external-program ext-stream)
    )
  )

(defun test-crash (ext-stream
		   &OPTIONAL
		   (filename "/u/hewett/research/algy/test/units-trace1.text"))

  ;;; Read into a string, then write it.

  (with-open-file (infile filename :direction :input)
    (let ((ch       NIL)
	  )
      (dotimes (i 25)
	(print i) (finish-output)
	(with-output-to-string (str nil :ELEMENT-TYPE 'character)
	  (loop
	    (setq ch (read-char infile))
	    (if (char= ch #\null)
		(return)
	      ;;ELSE
	        (write ch :stream str)
		)
	    )
	  (external-string-output ext-stream (get-output-stream-string str))
	  )
	)
      )
    )
  )



(defun test-crash2 (ext-stream)
 
  ;;; This sequence of outputs is known to crash the external program.

  (EXTERNAL-RAW-OUTPUT  ext-stream  #\E)
  (EXTERNAL-RAW-OUTPUT  ext-stream  #\G)
  (EXTERNAL-RAW-OUTPUT  ext-stream
			"H    1 : TELL - ((TAXONOMY (SETS (CONTEXTS C_0))) (SLOT RELEVANT (CONTEXTS SETS))
 (SLOT INHERIT (CONTEXTS SLOTS)) (SLOT COREF (THINGS THINGS))) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H    2 : PUSH - RETRIEVE   (NAME ?$x9 CONTEXTS) #S(ARESULT :BINDINGS NIL :LABEL (NIL)
                                           :OTHER NIL) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "FRETRIEVE   (NAME ?$x9 CONTEXTS) #S(ARESULT :BINDINGS NIL :LABEL (NIL)
                                           :OTHER NIL) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H    3 : POP  - RETRIEVE   (NAME ?$x9 CONTEXTS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream   #\P)

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H    4 : KB-PUT : (IMP-SUPERSET CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H    5 : KB-VERIFY : (ISA CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H    6 : PUSH - RETURN-SUCCESS (IMP-SUPERSET CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "FRETURN-SUCCESS (IMP-SUPERSET CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H    7 : PUSH - RULE8 : -> [CONTEXTS,IMP-SUPERSET] index=1, bindings=((?SET2 SETS)
                                                      (?SET1 CONTEXTS)) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "FRULE8 : -> [CONTEXTS,IMP-SUPERSET] index=1, bindings=((?SET2 SETS)
                                                      (?SET1 CONTEXTS)) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H    8 : PUSH - RULE10 : -> [CONTEXTS,IMP-SUPERSET] index=1, bindings=((?SET2 SETS)
                                                       (?SET1 CONTEXTS)) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "FRULE10 : -> [CONTEXTS,IMP-SUPERSET] index=1, bindings=((?SET2 SETS)
                                                       (?SET1 CONTEXTS)) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H    9 : POP  - RULE-FWD   #S(RI :RULE RULE10 :RESULT #S(ARESULT :BINDINGS ((?SET2 SETS) (?SET1 CONTEXTS)) :LABEL (NIL) :OTHER NIL) :INDEX 1 :ANTE ((IMP-SUPERSET ?SET1 ?SET2)) :CONSE ((RULES ?SET1 ((ISA ?X ?SET1) -> (ISA ?X ?SET2)))) :PRIORITY NIL) ")

  (EXTERNAL-RAW-OUTPUT ext-stream   #\P)

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   10 : APPLY-RULE : RULE10 (->) index = 1 ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   11 : KB-VERIFY : (IMP-SUPERSET CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   12 : KB-VERIFY : (ISA CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   13 : POP  - RULE-FWD   #S(RI :RULE RULE8 :RESULT #S(ARESULT :BINDINGS ((?SET2 SETS) (?SET1 CONTEXTS)) :LABEL (NIL) :OTHER NIL) :INDEX 1 :ANTE ((IMP-SUPERSET ?SET1 ?SET2)) :CONSE ((SUBSET ?SET2 ?SET1)) :PRIORITY NIL) ")

  (EXTERNAL-RAW-OUTPUT ext-stream   #\P)

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   14 : APPLY-RULE : RULE8 (->) index = 1 ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   15 : KB-VERIFY : (IMP-SUPERSET CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   16 : KB-VERIFY : (ISA CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   17 : KB-PUT : (SUBSET SETS CONTEXTS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   18 : KB-VERIFY : (ISA CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   19 : PUSH - RETURN-SUCCESS (SUBSET SETS CONTEXTS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "FRETURN-SUCCESS (SUBSET SETS CONTEXTS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   20 : PUSH - RULE6 : -> [SETS,SUBSET] index=1, bindings=((?X CONTEXTS) (?Y SETS)) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "FRULE6 : -> [SETS,SUBSET] index=1, bindings=((?X CONTEXTS) (?Y SETS)) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   21 : POP  - RULE-FWD   #S(RI :RULE RULE6 :RESULT #S(ARESULT :BINDINGS ((?X CONTEXTS) (?Y SETS)) :LABEL (NIL) :OTHER NIL) :INDEX 1 :ANTE ((SUBSET ?Y ?X)) :CONSE ((SUPERSET ?X ?Y)) :PRIORITY NIL) ")

  (EXTERNAL-RAW-OUTPUT ext-stream  #\P)

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   22 : APPLY-RULE : RULE6 (->) index = 1 ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   23 : KB-VERIFY : (SUBSET SETS CONTEXTS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   24 : KB-VERIFY : (ISA CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   25 : KB-PUT : (SUPERSET CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   26 : KB-VERIFY : (ISA CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   27 : PUSH - RETURN-SUCCESS (SUPERSET CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "FRETURN-SUCCESS (SUPERSET CONTEXTS SETS) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   28 : PUSH - RULE4 : -> [CONTEXTS,SUPERSET] index=1, bindings=((?Y SETS) (?X CONTEXTS)) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "FRULE4 : -> [CONTEXTS,SUPERSET] index=1, bindings=((?Y SETS) (?X CONTEXTS)) ")

  (EXTERNAL-RAW-OUTPUT ext-stream
		       "H   29 : POP  - RULE-FWD   #S(RI :RULE RULE4 :RESULT #S(ARESULT :BINDINGS ((?Y SETS) (?X CONTEXTS)) :LABEL (NIL) :OTHER NIL) :INDEX 1 :ANTE ((SUPERSET ?X ?Y)) :CONSE ((SUBSET ?Y ?X)) :PRIORITY NIL) ")



  )
