// ===================================================================
// Copyright (c) 1997, All rights reserved, by Micheal S. Hewett
//
// This software is free for educational and non-profit use.
// Any for-profit use must be governed by a license available
// from the author, who can be contacted via email at 
// "hewett@cs.stanford.edu"
//
// ===================================================================
//
//  SymbolTable.java  - A symbol table for use with LISP and similar 
//                      programs.
//
//  25 Jan 1997 (mh)
//
//  Main functions:
//     put(read(InputStream);        -- general read from a file or pipe
//     read(String);             -- read from a string
//
// -------------------------------------------------------------------
//  Content Summary
//  ---------------
//
//  A symbol table, using put() and get() as the primary access methods.
//  Also has map().  Uses the java.util.HashTable class.
//
//		LispValue SymbolTable.get(LispString key);
//		LispValue SymbolTable.put(LispString key, LispValue value);
//
//
// ===================================================================


package lib.util;

import java.io.*;
import java.util.*;

import lib.dynatype.*;




// This defines a hash table containing LispSymbol class elements.  They are
// indexed by LispString class elements.

public class SymbolTable extends java.util.Hashtable
{
  public static final int   HASH_TABLE_DEFAULT_SIZE        = 4096;
  public static final float HASH_TABLE_DEFAULT_LOAD_FACTOR = (float) 0.8;
  
  public SymbolTable()
  {
    super(HASH_TABLE_DEFAULT_SIZE, HASH_TABLE_DEFAULT_LOAD_FACTOR);
  }
	
  public SymbolTable(int defaultSize)
  {
    super(defaultSize, HASH_TABLE_DEFAULT_LOAD_FACTOR);
  }
	
  public SymbolTable(int initialSize, float loadFactor)
  {
    super(initialSize, loadFactor);
  }
	
  // Use the Hashtable methods PUT and GET.	
  // The 'value' is supposed to always be a symbol,
  // but because NIL is both a Symbol and a List, we 
  // must make the type be 'LispValue' instead of
  // 'LispSymbol'.
  public synchronized LispValue put(LispValue key, LispValue value)
  {
    LispString k = null;

    if (key instanceof LispString)
      k = (LispString)key;

    try {
      super.put(k.getValue(), value);   // Index by the Java String.
    }
    catch (NullPointerException e)
      {
	System.err.println("NullPointerException for " + k + "/" +
			   value + " in SymbolTable.put.");
	return null;
      }
    finally {
      return value;
    }
  }

  // Returns NIL if the entry is not there.
  public synchronized LispValue get(LispValue key)
  {
    // Index by the Java String
    Object val = super.get(((LispString)key).getValue());

    if (val == null)
      return LispValue.NIL;
    else
      return (LispValue) val;
  }			
}
